// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_VIEWPORTLAYER_H
#define FNVIEWER_VIEWPORTLAYER_H

#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/suite/FnViewportLayerSuite.h>
#include <FnViewer/plugin/FnMathTypes.h>
#include <FnViewer/plugin/FnPickingTypes.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <map>
#include <string>
#include <iostream>
#include <memory>


namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnViewportLayer ViewportLayer Plugin
 * @{
 */

/**
 * @brief Interface for a layer in the Viewport.
 *
 * A ViewportLayer (or layer) is an optional component of a Viewport that is
 * responsible for a specific set of drawing and/or event processing. Examples
 * of possible layers that can be implemented in a viewer:
 *
 *  - Manipulators Layer: draws and interacts with manipulators;
 *  - Mesh Layer: draws all the meshes in the scene;
 *  - Camera Pan Layer: deals with the user panning the camera in the viewer.
 *
 * The ViewportLayer allows a better code organization, but the most important
 * reason for their existence is code reusability. Since they are independent
 * plug-ins they can be reused on different Viewports and Viewers.
 *
 * Every ViewportLayer instance has a Viewport associated with it. That
 * Viewport can create and manage several ViewportLayers. The Viewport makes
 * sure that the setup(), draw(), resize() and event() functions in its
 * ViewportLayers are called for all of them.
 *
 * A ViewportLayer also allows picking / selection of rendered objects in the
 * scene inside a certain region of the viewport. This can be a deep picking
 * (all objects inside the region, independently if they are occluded or not
 * from the current camera point of view) or it can select only the visible
 * objects. The Viewport provides an optional internal mechanism that makes use
 * of an internal ID pass framebuffer. This can be used by implementing the
 * \c pickerDraw() virtual function, calling \c addPickableObject() in it. If
 * the rendering technology provides its own picking mechanism, then it can be
 * used by implementing the \c customPick() function, which will not make use
 * of the internal ID picking, and pickerDraw() will not be called. This
 * mechanism also is present in the Viewport plugin type.
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static ViewportLayer* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewerDelegate is the class that plugins should extend and implement.
 *
 * ViewerDelegateWrapper is the class that allows other plugin types to access
 * the ViewerDelegate plugin.
 *
 * ViewerDelegatePluginBase is the base class that provides the common methods
 * between ViewerDelegate and ViewerDelegateWrapper.
 *
 */
class ViewportLayerPluginBase
{
public:
    // API methods (to be implemented / used by sub-classes)

    ViewportLayerPluginBase();
    virtual ~ViewportLayerPluginBase();
public: /* Methods to be called by the plugin. */

    /**
     * @brief Gets the Viewport.
     *
     * @return The Viewport that created and manages this layer.
     */
    ViewportWrapperPtr getViewport();

    /**
     * @brief Called when the mouse hovers the ViewportLayer.
     *
     * Layers can detect when the mouse is hovering them so that, for example,
     * objects under the mouse pointer can be highlighted. This is called with
     * the correct GL context so that GL based picking can be executed.
     *
     * @param isHovering Flag that specifies if the mouse is hovering the
     *     layer. If false, then the mouse left the layer. In this case x and y
     *     should be ignored.
     *
     * @param x The horizontal pixel coordinate of the mouse pointer in the
     *     layer's local coordinate system.
     *
     * @param y The vertical pixel coordinate of the mouse pointer in the
     *     layer's local coordinate system.
     */
    void hover(bool isHovering, int x, int y);

    /**
     * @brief Picks the objects that are inside a given viewport region.
     *
     * Allows to query the scene for objects that are visible inside a given
     * rectangular region in the viewport. This region can be a single pixel
     * on the screen, by setting both its width and height to 1.
     *
     * The object picking can optionally be deep, meaning that all the objects
     * viewed inside the region will be picked even if they are currently
     * occluded by other objects. A non-deep picking means that only currently
     * visible objects inside the region will be picked.
     *
     * Picking can be implemented in two ways:
     *
     *   - Using the internal ID framebuffer based technique implemented via
     *     \c pickerDraw() and \c addPickableObject().
     *
     *   - Implementing a custom picking implemented using some third party
     *     technology via \c customPick().
     *
     * This function makes use of whatever picking technique is implemented in
     * this ViewportLayer.
     *
     * This function returns a map of picking IDs to Attributes that represent
     * the picked objects and, optionally a depth value for when the picked
     * region is one single pixel.
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param deepPicking Specifies if all objects inside the region,
     *      including occluded ones, will be picked. If set to false, only the
     *      visible objects should be picked. This should not be set to true
     *      in very frequent events, like a mouse move, since this might
     *      trigger several calls to pickerDraw().
     * @param[out] pickedAttrs A map top be filled with \c FnPickId (key) to
     *      \c Attribute (value) pairs that represent the picked objects.
     *      Internally, this will be either populated by \c customPick() or by
     *      \c addPickableObject() calls inside \c pickerDraw().
     *      \c PickedAttrsMap has the same public interface as
     *      \c std::map<FnPickId, FnAttributes::Attribute>.
     * @param[out] singlePointDepth The value pointed by this will be set
     *      with the GL depth of a single pixel when the region with and height
     *      are both 1 and this pointer is set to something other than NULL.
     *      This can be used to solve occlusion between picked objects from
     *      the Viewport and different ViewportLayers when, for example, the
     *      user clicks on a single pixel when selecting something.
     *
     */
    void pick(unsigned int x, unsigned int y,
              unsigned int w, unsigned int h,
              bool deepPicking,
              PickedAttrsMap& pickedAttrs,
              float* singlePointDepth=NULL);

/// @cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    FnViewportLayerHostSuite_v2* m_hostSuite;
    FnViewportLayerHostHandle m_hostHandle;

    static FnPluginHost* m_host;

private:
    ViewportWrapperPtr m_viewportWrapper;

/// @endcond
};

/** @brief The ViewportLayer class to be extended by plugins. */
class ViewportLayer : public ViewportLayerPluginBase
{
public:

    /**
     * @brief Constructor.
     *
     * @param usePickingOnHover If true, then this layer will be calling
     *      \c pick() very frequently to detect if the mouse is hovering any
     *      pickable object. This can be used, for example, to highlight an
     *      object hovered by the mouse. This specifies that \c pickerDraw()
     *      will run right after any \c draw() call, so that the IDs readily
     *      available on each mouse move. If set to false, the ID pass render
     *      will only occur when \c pick() is called, which is more efficient.
     */
    ViewportLayer(bool usePickingOnHover=false);

    /** @brief Destructor. */
    virtual ~ViewportLayer();

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Initializes the GL components of the ViewportLayer.
     *
     * Called when a ViewportLayer is created. It runs inside the correct GL
     * context. Can be used to initialize anything GL related. For example, in
     * the case of an OpenGL renderer this function should set up any required
     * OpenGL context rendering flags, defining display lists, etc.
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void setup() = 0;

    /**
     * @brief Cleans up the ViewportLayer resources.
     *
     * Called when a ViewportLayer is removed by the ViewerDelegate.
     */
    virtual void cleanup() = 0;

    /**
     * @brief Processes UI events.
     *
     * Called whenever a user interaction event occurs. This will be also
     * propagated through the ViewerLayers of this ViewportLayer.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, in that case it will not be
     *         passed to the following layers. Otherwise false and the event
     *         will be passed to the following layers.
     */
    virtual bool event(const FnEventWrapper& eventData) { return false; }

    /**
     * @brief Draws the scene.
     *
     * Called when the scene needs to be drawn in the correct GL context. In a
     * non-GL renderer the generated image should be drawn in the GL
     * framebuffer in order to be displayed. This will be also propagated
     * through the ViewerLayers of this ViewportLayer.
     *
     * This function is exposed in the ViewportLayer Python class.
     */
    virtual void draw() = 0;

    /**
     * @brief Processes viewport resizing.
     *
     * Called when the viewport widget is resized. This can be extended by
     * sub-clases to accomodate viewport size changes in the renderer.
     */
    virtual void resize(unsigned int width, unsigned int height) = 0;

    /**
     * @brief Called when the mouse hovers the ViewportLayer.
     *
     *  The Viewport can detect when the mouse is hovering it so that, for
     *  example, objects under the mouse pointer can be highlighted. This is
     *  called with the correct GL context so that GL based picking can be
     *  executed.
     *
     *  @param isHovering Flag that specifies if the mouse is hovering the
     *      viewport. If false, then the mouse left the viewport. In this case
     *      x and y should be ignored.
     *  @param x The horizontal pixel coordinate of the mouse pointer in the
     *       Viewport's local coordinate system.
     *  @param y The vertical pixel coordinate of the mouse pointer in the
     *      Viewport's local coordinate system.
     */
    virtual void hover(bool isHovering, int x, int y) {}

    /**
     * @brief Freezes the ViewportLayer when the viewport widget is hidden.
     *
     * Allows the ViewportLayer to freeze its activities when the viewport is
     * not visible. This allows the ViewerDelegate to stop any kind of
     * unecessary processing that might happen during that time.
     *
     */
    virtual void freeze() = 0;

    /**
     * @brief Thaws the ViewportLayer when the viewport widget is shown.
     *
     * Allows the ViewportLayer to restart its activities when the viewport
     * becomes visible. This restarts the activities paused by freeze().
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void thaw() = 0;

    /**
     * @brief Flush plugin Caches.
     *
     * Allows to discard any cache for this plugin when a Flush Caches
     * event occurs.
     *
     * This function is exposed in the Viewport Python class.
     */
    static void flush() {}

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /* Options setting and getting. */

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the ViewportLayer.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

    /**
     * @brief Tells if the layer picks objects when the mouse is hovering them.
     *
     * This returns the value passed to the ViewportLayer's constructor
     * argument \a usePickingOnHover. This leads to pickerDraw() to be called
     * right after any draw() call, so that each frame produces the correct
     * pickable IDs pass internall. This will not happen if \c customPick() is
     * implemented and returns true.
     */
    bool usesPickingOnHover() { return m_usesPickingOnHover; }

    /**
     * @brief Draws an ID pass to be used by the optional internal picking.
     *
     * This allows this layer to make use of the internal ID framebuffer
     * pass technique provided to the Viewport and ViewportLayers. This
     * function should draw all the pickable objects into the current GL
     * framebuffer using an ID color. Inside this function each pickable object
     * has to be registered using \c addPickableObject(), which will return a
     * FnPickId. This id can be converted into a color using \c pickIdToColor()
     * (see FnPickingTypes.h), the objects should be rendered without any kind
     * of antialiasing and with that flat color, so that they cover their
     * pickable pixels.
     *
     * This will be called by Katana if customPick() returns false or if it is
     * not implemented.
     *
     * This function receives a list of Attributes that refer to pickable
     * objects to be ignored in this render. This is internally used to do a
     * multi-pass id render for deep picking, in which an onion peeling
     * technique is used to detect all the occluded objects. On each iteration
     * this function should not render the objects that were detected in
     * previous iterations. These Attributes correspond to the ones passed
     * previously to \c addPickableObject().
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param ignoreAttrs Map that contains information about the objects that
     *      should not be rendered here. \c PickedAttrsMap has the same
     *      public interface as \c std::map<FnPickId, FnAttributes::Attribute>.
     */
    virtual void pickerDraw(unsigned int x, unsigned int y,
                            unsigned int w, unsigned int h,
                            const PickedAttrsMap& ignoreAttrs) {}

    /**
     * @brief Overrides the internal ID picking using a third party technique.
     *
     * If the technology used in the Viewport implements its own picking, or if
     * a GL ID pass is not feasible, then this function allows to override the
     * internal ID picking and to implement the picking of what is present in
     * the scene. If this is implemented and returns true, then \c pickerDraw()
     * will never be called by Katana. This is called internally by \c pick(),
     * which will return the returned values of this function.
     *
     * An example of the use of this function is when using a non-realtime
     * renderer that is able to produce its own ID pass or when the renderer
     * data structures allows to query the geometry present inside the frustrum
     * defined by the picking area.
     *
     * There is no need to call \c addPickableObject() inside this function.
     *
     * This should return a map of \c FnPickId to Attributes, similar to the
     * one returned by \c pickerDraw(), which can identify or contain
     * information about each picked objects. The
     *
     * This can optionally return a depth value for when the picked region is
     * one single pixel.
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param deepPicking Specifies if all objects inside the region,
     *      including occluded ones, will be picked. If set to false, only the
     *      visible objects should be picked.
     * @param[out] pickedAttrs A map top be filled with \c FnPickId (key) to
     *      \c Attribute (value) pairs that represent the picked objects.
     *      Internally, this will be either populated by \c customPick() or by
     *      \c addPickableObject() calls inside \c pickerDraw().
     *      \c PickedAttrsMap has the same public interface as
     *      \c std::map<FnPickId, FnAttributes::Attribute>.
     * @param[out] singlePointDepth The value pointed by this will be set
     *      with the GL depth of a single pixel when the region with and height
     *      are both 1 and this pointer is set to something other than NULL.
     *      This can be used to solve occlusion between picked objects from
     *      the Viewport and different ViewportLayers when, for example, the
     *      user clicks on a single pixel when selecting something.
     */
    virtual bool customPick(unsigned int x, unsigned int y,
                            unsigned int w, unsigned int h,
                            bool deepPicking,
                            PickedAttrsMap& pickedAttrs,
                            float* singlePointDepth=NULL);


    /**
     * @brief Registers a pickable object during \c pickerDraw().
     *
     * This should be called inside \c pickerDraw() in order to let the
     * internal ID picking system know about each pickable object.
     *
     * The ID returned by this function can be used to both identify an object
     * in some data structure implemented inside this plugin and to define the
     * color to be used by the object when rendering itself in the current ID
     * framebuffer in \c pickerDraw(), via the \c pickIdToColor() function
     * (see FnPickingTypes.h). Also see FnGLShaderProgram.h, which implements a
     * way of loading GLSL shaders.
     *
     * The Attribute passed to this function can contain further information
     * about the pickable object. For some generic cases there will be
     * Attribute conventions that prescribe how this Attribute should be
     * structured in order to be recognized as some typical objects. This
     * allows some out-of-the-box or third party plugins, like other
     * ViewportLayer plugins, to identify objects like, for example, a
     * location:
     *   - A location should be identified using a GroupAttribute containing at
     *     least a child StringAttribute named "location" and with a single
     *     value containing the full location path. This will be recognized by
     *     the ViewportLayer plugin "SelectionLayer" shipped with Katana, which
     *     is responsible for dealing with locations selection in the Viewer.
     *
     * Any kind of rendered object, other than locations, can be pickable. For
     * example, a handle of some overlay widget that can be manipulated using
     * the mouse. For this, each Viewport will make use of some Attribute
     * convention that is suitable to identify its own pickable objects.
     *
     * The \c pick() function will return the \c FnPickID / \c Attribute pairs
     * of all picked objects inside its region.
     *
     * @param attr The Attribute that describes the pickable object.
     * @return A picking ID assigned to the pickable object.
     */
    FnPickId addPickableObject(FnAttribute::Attribute attr);

/// @cond FN_INTERNAL_DEV
public:
    static FnViewportLayerPluginSuite_v2 createSuite(
        FnViewportLayerPluginHandle (*create)(
            FnViewportLayerHostHandle hostHandle));
    static FnViewportLayerPluginHandle newViewportLayerHandle(
        ViewportLayer* viewportLayer);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnViewportLayerHostHandle m_hostHandle);
    FnViewportLayerHostHandle getHostHandle();

private:
    bool m_usesPickingOnHover;

/// @endcond
};

/** @brief The ViewportLayer class accessed by other plugins. */
class ViewportLayerWrapper : public ViewportLayerPluginBase
{
public:
    ViewportLayerWrapper(
        FnPluginHost* host,
        FnViewportLayerHostHandle hostHandle,
        FnViewportLayerPluginHandle pluginHandle,
        FnViewportLayerPluginSuite_v2* pluginSuite);

    ~ViewportLayerWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of ViewportLayer
     * that is a registerd plugin. The type of that child class needs to be
     * specified in the template, so that it can be properly cast. If the
     * specified type  doesn't match the plugin type then NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /**
     * @brief Draws the Viewport Layer.
     *
     * This function is exposed in the Viewport Python class.
     */
    void draw();

    /**
     * @brief Processes the UI events on this ViewportLayer.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, in which case it will not be
     *          passed to the following layers. Otherwise false and the event
     *          will be passed to the following layers.
     */
    bool event(FnEventWrapper eventData);

    /** @brief Sets a generic option. */
    void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);
    /** @brief Sets a generic option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
private:
    ViewportLayer* getPluginPointer();

    /** The ViewportLayer plug-in. */
    FnViewportLayerPluginSuite_v2* m_pluginSuite;
    FnViewportLayerPluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ViewportLayerWrapper> ViewportLayerWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry


/// @cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnViewportLayerPluginStruct
{
public:
    FnViewportLayerPluginStruct(
        Foundry::Katana::ViewerAPI::ViewportLayer* viewportLayer)
    : m_viewportLayer(viewportLayer)
    {}

    ~FnViewportLayerPluginStruct()
    {};

    Foundry::Katana::ViewerAPI::ViewportLayer* getViewportLayer()
    {
        return m_viewportLayer.get();
    }

    FnAttribute::GroupAttribute getAttributes(const std::string& locationPath);

private:
    std::shared_ptr<Foundry::Katana::ViewerAPI::ViewportLayer> m_viewportLayer;
};


// Plugin Registering Macro.
#define DEFINE_VIEWPORT_LAYER_PLUGIN(VIEWPORT_LAYER_CLASS)                    \
                                                                              \
    FnPlugin VIEWPORT_LAYER_CLASS##_plugin;                                   \
                                                                              \
    FnViewportLayerPluginHandle VIEWPORT_LAYER_CLASS##_create(                \
        FnViewportLayerHostHandle hostHandle)                                 \
    {                                                                         \
        Foundry::Katana::ViewerAPI::ViewportLayer* viewportLayer =            \
            VIEWPORT_LAYER_CLASS::create();                                   \
                                                                              \
        viewportLayer->setHostHandle(hostHandle);                             \
        return Foundry::Katana::ViewerAPI::ViewportLayer::newViewportLayerHandle(viewportLayer); \
    }                                                                         \
                                                                              \
    FnViewportLayerPluginSuite_v2 VIEWPORT_LAYER_CLASS##_suite =              \
            Foundry::Katana::ViewerAPI::ViewportLayer::createSuite(           \
                    VIEWPORT_LAYER_CLASS##_create);                           \
                                                                              \
    const void* VIEWPORT_LAYER_CLASS##_getSuite()                             \
    {                                                                         \
        return &VIEWPORT_LAYER_CLASS##_suite;                                 \
    }

/// @endcond

#endif
