// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_VIEWPORTCAMERA_H
#define FNVIEWER_VIEWPORTCAMERA_H

#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/suite/FnViewportCameraSuite.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>
#include <FnViewer/plugin/FnMathTypes.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <map>
#include <string>
#include <iostream>
#include <memory>

// Built-in camera types
#define kFnViewportCameraTypeUnknown 0
#define kFnViewportCameraTypePerspective 1
#define kFnViewportCameraTypeOrthographic 2
#define kFnViewportCameraTypeSpherical 3

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnViewportCamera ViewportCamera Plugin
 * @{
 */

/**
 * @brief Interface for a camera.
 *
 * A ViewportCamera is an important component of a Viewport that is used by the
 * Viewport, ViewportLayers and Manipulators to determine the view and
 * projection matrices, and also to perform useful functions such as
 * projecting points between world and window spaces. A Viewport requires
 * a reference to a single ViewportCamera, however multiple viewports can
 * reference the same camera if required. A camera can optionally be controlled
 * by a location (see hasLocationPath(), getLocationPath() and
 * setLocationPath).
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static ViewportCamera* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewportCamera is the class that plugins should extend and implement.
 *
 * ViewportCameraWrapper is the class that allows other plugin types to access
 * the ViewportCamera plugin.
 *
 * ViewportCameraPluginBase is the base class that provides the common methods
 * between ViewportCamera and ViewportCameraWrapper.
 *
 */
class ViewportCameraPluginBase
{
public:
    enum CameraDirtyBits
    {
        Clean = 0,
        DirtyParams = 1 << 0,
        AllDirty = (DirtyParams)
    };

public:
    // API methods (to be implemented / used by sub-classes)

    ViewportCameraPluginBase();
    virtual ~ViewportCameraPluginBase();
public: /* Methods to be called by the plugin. */

    /**
     * @brief Gets the Viewport.
     *
     * @return The Viewport that created and manages this layer.
     */
    ViewportWrapperPtr getViewport();

    /**
     * @brief Gets the View Matrix for this camera.
     *
     * @return The 4x4 Viewer Matrix represented by 16 doubles.
     */
    const double* getViewMatrix();

    /**
     * @brief Gets the View Matrix for this camera.
     *
     * @return The 4x4 View Matrix.
     */
    Matrix44d getViewMatrix44d();

    /**
     * @brief Sets the View Matrix for this camera.
     *
     * @param matrix The 4x4 Viewer Matrix represented by 16 doubles.
     */
    void setViewMatrix(const double* matrix);

    /**
     * @brief Gets the Projection Matrix for this camera.
     *
     * @return The 4x4 Projection Matrix represented by 16 doubles.
     */
    const double* getProjectionMatrix();

    /**
     * @brief Gets the Projection Matrix for this camera.
     *
     * @return The 4x4 Projection Matrix.
     */
    Matrix44d getProjectionMatrix44d();

    /**
     * @brief Sets the Projection Matrix for this camera.
     *
     * @param matrix The 4x4 Projection Matrix represented by 16 doubles.
     */
    void setProjectionMatrix(const double* matrix);

    /**
     * @brief Gets the center of interest for this camera.
     *
     * @return The distance along the cameras look direction to the center of
     *          interest.
     */
    double getCenterOfInterest();

    /**
     * @brief Sets the center of interest for this camera.
     *
     * @param coi The distance along the cameras look direction to the center
     *             of interest.
     */
    void setCenterOfInterest(double coi);

    /** @brief Gets the field of view for this camera. */
    double getFOV();

    /**
     * @brief Sets the field of view for this camera.
     *
     * @param fov The field of view angle in degrees.
     */
    void setFOV(double fov);

    /**
     * @brief Gets the orthographic camera dimensions (if applicable)
     *
     * @return The orthographic width, or 0 in not applicable.
     */
    double getOrthographicWidth();

    /**
     * @brief Sets the orthographic width of the camera.
     *
     * @param width The new orthographic width.
     */
    void setOrthographicWidth(double width);

    /**
     * @brief Gets the near and far plane distances for this camera.
     *
     * @param near Will be set to the distance to the near clipping plane.
     * @param far Will be set to the distance to the far clipping plane.
     */
    void getNearFar(double &near, double &far);

    /**
     * @brief Sets the near and far plane distances for this camera.
     *
     * @param near The distance to the near clipping plane.
     * @param far The distance to the far clipping plane.
     */
    void setNearFar(double near, double far);

    /**
     * @brief Gets the screen window dimensions for this camera.
     *
     * @param left Will be set to the left position of the screen window.
     * @param right Will be set to the right position of the screen window.
     * @param bottom Will be set to the bottom position of the screen window.
     * @param top Will be set to the top position of the screen window.
     */
    void getScreenWindow(double &left, double &right, double &bottom, double &top);

    /**
     * @brief Sets the screen window dimensions for this camera.
     *
     * @param left The left position of the screen window.
     * @param right The right position of the screen window.
     * @param bottom The bottom position of the screen window.
     * @param top The top position of the screen window.
     */
    void setScreenWindow(double left, double right, double bottom, double top);

    /**
     * @brief Returns true if this camera is controlled by a location.
     *
     * A camera that is controlled by a location has its properties (including
     * the transform) defined by a location produced by the ViewerDelegate.
     */
    bool hasLocationPath();

    /**
     * @brief Gets the location path if this camera is controlled by one.
     *
     * A camera that is controlled by a location has its properties (including
     * the transform) defined by a location produced by the ViewerDelegate.
     *
     * @return The location path that controls this camera or an empty string
     *     if it is not controlled by any location.
     */
    std::string getLocationPath();

    /**
     * @brief Sets the location that will control this camera.
     *
     * A camera that is controlled by a location has its properties (including
     * the transform) defined by a location produced by the ViewerDelegate.
     */
    void setLocationPath(const std::string& locationPath);

    /** @brief Gets the camera origin. */
    Vec3d getOrigin();

    /** @brief Gets the camera look direction. */
    Vec3d getDirection();

    /** @brief Gets the camera up direction. */
    Vec3d getUp();

    /** @brief Gets the camera left direction. */
    Vec3d getLeft();

    /**
     * @brief Checks whether the camera has been flagged as non-interactive.
     *
     * Layers that wish to start an interaction should check for this predicate
     * before making any change to the camera.
     *
     * @return True if interaction has been explicitly disabled.
     */
    bool isInteractionDisabled();

    /**
     * @brief Flags the camera as non-interactive.
     *
     * @param disabled Set it to true to disable interaction or false to
     *     re-enable it.
     */
    void disableInteraction(bool disabled);

///@cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

    FnViewportCameraHostHandle getCameraHostHandle() const {
        return m_hostHandle;
    }

protected:
    FnViewportCameraHostSuite_v2* m_hostSuite;
    FnViewportCameraHostHandle m_hostHandle;

    static FnPluginHost* m_host;

private:
    ViewportWrapperPtr m_viewportWrapper;

/// @endcond
};


/** @brief The ViewportCamera class to be extended by plugins. */
class ViewportCamera : public ViewportCameraPluginBase
{
public:
    ViewportCamera();
    virtual ~ViewportCamera();

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Flush plugin Caches.
     *
     * Allows to discard any cache for this plugin when a Flush Caches
     * event occurs.
     *
     */
    static void flush() {}

    /**
     * @brief Returns an integer indicating the type of the camera.
     *
     * Can be kPERSPECTIVE_CAMERA, kORTHOGRAPHIC_CAMERA, or another custom
     * type.
     *
     * @return The camera type ID.
     */
    virtual int getCameraTypeID() = 0;

    /**
     * @brief Processes Viewport resizing.
     *
     * Used to specify the dimensions of a Viewport which can affect picking
     * calculations.
     *
     * @param width The viewport width in pixels.
     * @param height The viewport height in pixels.
     */
    virtual void setViewportDimensions(unsigned int width, unsigned int height) = 0;

    /**
     * @brief Sets a generic option.
     *
     * Reacts to a generic option being set by other C++ Viewer plugin
     * classes. This can be used as a message passing mechanism from
     * the outside into the ViewportCamera.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

    /**
     * @brief Called when the user starts interacting with the camera.
     *
     * This is called when starts interacting with this camera while, for
     * example, is viewing through it. This allows the camera to initialize
     * some initial values used during the user interaction. The translate()
     * and rotate() functions are expected to be called after
     * startInteraction() and before endInteraction().
     */
    virtual void startInteraction() = 0;

    /**
     * @brief Called when the user finishes interacting with the camera.
     *
     * This is called when finishes interacting with this camera while, for
     * example, is viewing through it. This allows the camera to commit or tear
     * down some values used during the user interaction. One example could be
     * set the final camera transform values back into the node graph in a
     * location-backed camera. The translate() and rotate() functions are
     * expected to be called after startInteraction() and before
     * endInteraction().
     */
    virtual void endInteraction() = 0;

    /**
     * @brief Handles translation movement of the camera.
     *
     * This is expected to be called after startInteraction() and
     * before endInteraction().
     *
     * @param x The X-axis translation in camera's local space.
     * @param y The Y-axis translation in camera's local space.
     * @param z The Z-axis translation in camera's local space.
     */
    virtual void translate(double x, double y, double z) = 0;

    /**
     * @brief Handles rotation movement of the camera.
     *
     * This is expected to be called after startInteraction() and
     * before endInteraction().
     *
     * @param x The X-axis rotation in camera's local space.
     * @param y The Y-axis rotation in camera's local space.
     * @param z The Z-axis rotation in camera's local space.
     */
    virtual void rotate(double x, double y, double z) = 0;

    /**
     * @brief Initializes the camera with values from an attribute.
     *
     * If the camera is based on a scene graph camera, the passed attribute
     * will be the relevant attributes from the scene graph location.
     *
     * @param attr Attribute with the initial camera state values.
     */
    virtual void setup(FnAttribute::GroupAttribute attr) = 0;

    /**
     * @brief Returns the camera settings as a GroupAttribute
     *
     * This function should return a GroupAttribute that fully represents the
     * current camera settings in a format that can be passed into the setup()
     * function (as a way of duplicating the camera) or into a scene graph
     * location in order to populate a camera location.
     *
     * @return A GroupAttribute representing the camera state.
     */
    virtual FnAttribute::GroupAttribute asAttribute() = 0;

    /**
     * @brief Maps window coordinates to a 3D point on a plane.
     *
     * @param x The X co-ordinate of the window.
     * @param y The Y co-ordinate of the window.
     * @param planeOrigin The coordinates of the plane origin.
     * @param planeNormal The normal direction of the plane.
     * @param intersection Output argument where the intersection position
     *                       (if any) is set.
     *
     * @return true if an intersection occured.
     */
    virtual bool getPointOnPlane(int x, int y,
        const Vec3d& planeOrigin,
        const Vec3d& planeNormal, Vec3d& intersection) = 0;

    /**
     * @brief Calculates the ray from the camera through the given window
     *     coordinates.
     *
     * @param x The X co-ordinate of the window.
     * @param y The Y co-ordinate of the window.
     * @param pos Output argument where the line position is set.
     * @param dir Output argument where the line direction is set.
     */
    virtual void getRay(int x, int y, Vec3d& pos, Vec3d& dir) = 0;

    /**
     * @brief Converts a 3D point in the scene into window co-ordinates.
     *
     * @param point The point in world space to transform.
     *
     * @return A 3 dimensional vector representing the screen coordinates
     *          of the point in its X, Y members and the depth of the point
     *          in the Z member.
     */
    virtual Vec3d projectObjectIntoWindow(Vec3d point) = 0;

    /**
     * @brief Converts a window co-ordinates into a point in the scene.
     *
     *
     * @param point The point in window space to transform into world space.
     *              point.x and point.y indicate the X and Y position in pixels
     *              and point.z indicates the depth into the scene in world
     *              space.
     *
     * @return The position of the transformed point in world space.
     */
    virtual Vec3d projectWindowIntoObject(Vec3d point) = 0;

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /**
     * @brief Specifies what has changed in the camera.
     *
     * Something has changed and the camera needs to be updated. DirtyFlag is
     * passed to specify what has been changed so the plugin can update
     * accordingly.
     *
     * @param dirtyBits Flag to specify what has changed.
     */
    virtual void setDirty(CameraDirtyBits dirtyBits) = 0;

/// @cond FN_INTERNAL_DEV
public:
    static FnViewportCameraPluginSuite_v2 createSuite(
        FnViewportCameraPluginHandle (*create)(
            FnViewportCameraHostHandle hostHandle));
    static FnViewportCameraPluginHandle newViewportCameraHandle(
        ViewportCamera* viewportCamera);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnViewportCameraHostHandle m_hostHandle);
    FnViewportCameraHostHandle getHostHandle();

/// @endcond
};

/** @brief The ViewportCamera class accessed by other plugins. */
class ViewportCameraWrapper : public ViewportCameraPluginBase
{
public:
    ViewportCameraWrapper(
        FnPluginHost* host,
        FnViewportCameraHostHandle hostHandle,
        FnViewportCameraPluginHandle pluginHandle,
        FnViewportCameraPluginSuite_v2* pluginSuite);

    ~ViewportCameraWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of ViewportCamera
     * that is a registerd plugin. The type of that child class needs to be
     * specified in the template, so that it can be properly cast. If the
     * specified type doesn't match the plugin type then NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /** @brief Returns an integer indicating the type of the camera. */
    int getCameraTypeID();
    /** @brief Informs the camera of the Viewport dimensions in pixels. */
    void setViewportDimensions(unsigned int width, unsigned int height);
    /** @brief Sets a generic camera option. */
    void setOption(OptionIdGenerator::value_type optionId, FnAttribute::Attribute attr);
    /** @brief Gets a generic camera option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);
    /** @brief Sets a generic camera option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);
    /** @brief Gets a generic camera option. */
    FnAttribute::Attribute getOption(const std::string& name);
    /** @brief Translates the camera. */
    void translate(double x, double y, double z);
    /** @brief Rotates the camera. */
    void rotate(double x, double y, double z);
    /** @brief Called when the user starts interacting with the camera. */
    void startInteraction();
    /** @brief Called when the user finishes interacting with the camera. */
    void endInteraction();
    /** @brief Initializes the camera from a GroupAttribute. */
    void setup(FnAttribute::GroupAttribute attr);
    /** @brief Returns the camera settings as a GroupAttribute. */
    FnAttribute::GroupAttribute asAttribute();
    /**
     * @brief Returns the point 3D on a plane when passed X and Y window
     *     coordinates.
     */
    bool getPointOnPlane(int x, int y, const Vec3d& planeOrigin,
                         const Vec3d& planeNormal, Vec3d& intersection);
    /**
     * @brief Returns the ray from the observer through the X and Y window
     *     coordinates.
     */
    void getRay(int x, int y, Vec3d& pos, Vec3d& dir);
    /** @brief Converts a 3D point into window space. */
    Vec3d projectObjectIntoWindow(Vec3d point);
    /** @brief Converts a window space point into 3D space. */
    Vec3d projectWindowIntoObject(Vec3d point);
    /** @brief Specifies what has changed in the camera. */
    void setDirty(CameraDirtyBits dirtyBits);

/// @cond FN_INTERNAL_DEV
private:
    ViewportCamera* getPluginPointer();

    /** The ViewportCamera plug-in. */
    FnViewportCameraPluginSuite_v2* m_pluginSuite;
    FnViewportCameraPluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ViewportCameraWrapper> ViewportCameraWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry



///@cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnViewportCameraPluginStruct
{
public:
    FnViewportCameraPluginStruct(
        Foundry::Katana::ViewerAPI::ViewportCamera* viewportCamera)
    : m_viewportCamera(viewportCamera)
    {}

    ~FnViewportCameraPluginStruct()
    {};

    Foundry::Katana::ViewerAPI::ViewportCamera* getViewportCamera()
    {
        return m_viewportCamera.get();
    }

private:
    std::shared_ptr<Foundry::Katana::ViewerAPI::ViewportCamera>
        m_viewportCamera;
};


// Plugin Registering Macro.
#define DEFINE_VIEWPORT_CAMERA_PLUGIN(VIEWPORT_CAMERA_CLASS)                    \
                                                                              \
    FnPlugin VIEWPORT_CAMERA_CLASS##_plugin;                                   \
                                                                              \
    FnViewportCameraPluginHandle VIEWPORT_CAMERA_CLASS##_create(                \
        FnViewportCameraHostHandle hostHandle)                                 \
    {                                                                         \
        Foundry::Katana::ViewerAPI::ViewportCamera* viewportCamera =            \
            VIEWPORT_CAMERA_CLASS::create();                                   \
                                                                              \
        viewportCamera->setHostHandle(hostHandle);                             \
        return Foundry::Katana::ViewerAPI::ViewportCamera::newViewportCameraHandle(viewportCamera); \
    }                                                                         \
                                                                              \
    FnViewportCameraPluginSuite_v2 VIEWPORT_CAMERA_CLASS##_suite =              \
            Foundry::Katana::ViewerAPI::ViewportCamera::createSuite(           \
                    VIEWPORT_CAMERA_CLASS##_create);                           \
                                                                              \
    const void* VIEWPORT_CAMERA_CLASS##_getSuite()                             \
    {                                                                         \
        return &VIEWPORT_CAMERA_CLASS##_suite;                                 \
    }

///@endcond

#endif
