// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_VIEWPORT_H
#define FNVIEWER_VIEWPORT_H

#include <FnViewer/suite/FnViewportSuite.h>
#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/plugin/FnViewerDelegate.h>
#include <FnViewer/plugin/FnViewportCamera.h>
#include <FnViewer/plugin/FnManipulator.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>
#include <FnViewer/plugin/FnMathTypes.h>
#include <FnViewer/plugin/FnPickingTypes.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <string>
#include <map>
#include <vector>
#include <memory>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

// Forward declarations
class ViewportLayerWrapper;
class ManipulatorWrapper;
class ViewportCameraWrapper;
typedef std::shared_ptr<ViewportLayerWrapper> ViewportLayerWrapperPtr;
typedef std::shared_ptr<ManipulatorWrapper> ManipulatorWrapperPtr;
typedef std::shared_ptr<ViewportCameraWrapper> ViewportCameraWrapperPtr;

/**
 * \defgroup FnViewport Viewport Plugin
 * @{
 */

/**
 * @brief Interface for the functionality of a single viewport.
 *
 * A Viewport's main responsibilities are to draw the contents of a single
 * viewport widget and to handle the UI events of that widget. When the Qt GL
 * Widget for the viewport is created an instance of the Viewport class is also
 * instantiated and associated with it.
 *
 * Several Viewports can share and access a single ViewerDelegate (see
 * getViewerDelegate()). Some shared data structures can be owned by the
 * ViewerDelegate, so it is convenient to build both plug-ins using the same
 * compiler, so that the result of getViewerDelegate() can be cast into the
 * known subclass of ViewerDelegate, which might contain specific methods and
 * members that can be useful to the plug-in. If that is not possible, then the
 * more generic getOption() / setOption() in both the Viewport and the
 * ViewerDelegate can be used.
 *
 * By default, the Katana UI will wait for an idle event, and then call the
 * draw() function if the viewport has been marked as dirty via the setDirty()
 * method. It is important to note that prior to draw(), resize() and setup()
 * being called, the OpenGL context of the viewport widget will be made
 * current. The makeGLContextCurrent() / doneGLContextCurrent() functions can
 * also be used to force the GL context to be made current, but this should be
 * used springly, as frequent GL context changes can lead to performance
 * degradation. All of the viewport widgets hosted in a particular Viewer tab
 * will attempt to share Open GL context data with each other. This allows
 * viewports to share certain resources, such as Vertex Buffer Objects, but is
 * subject to the usual limitations of context sharing, such as containers
 * (Vertex Array Objects) not being shared. The default framebuffer in this
 * context is the Qt widget's framebuffer, so any drawing on it will end up
 * being shown on the widget.
 *
 * The Viewport can create and own several ViewportLayers. These are
 * independent plug-ins that can potentially be reusable between different
 * Viewports that are responsible for a specific set of drawing and/or event
 * processing activities. The Viewport delegates the draw() function and the
 * event() function on its layers. These layers are organized in a linear
 * sequence, in which the draw(), event() and resize() member functions
 * will be called starting in the layer with the lowest index and ending with
 * the one with the highest index.
 *
 * The Viewport class is also responsible for activating and deactivating
 * Manipulators on a list of locations (see activateManipulator() and
 * deactivateManipulator()). The compatibility of these Manipulators should be
 * checked previously with the ViewerDelegate.getCompatibleManipulatorsInfo()
 * function, to guarantee that they are compatible with the locations to be
 * manipulated. A ManipulatorLayer can be used then to actually draw and
 * process the events of these manipulators, but the Viewport is responsible
 * for keeping track of which ones are currently activated (see the functions
 * getNumberOfActiveManipulators() and getActiveManipulator()).
 *
 * A Viewport also allows picking / selection of rendered objects in the scene
 * inside a certain region of the viewport. This can be a deep picking (all
 * objects inside the region, independently if they are occluded or not from
 * the current camera point of view) or it can select only the visible objects.
 * The Viewport provides an optional internal mechanism that makes use of an
 * internal ID pass framebuffer. This can be used by implementing the
 * pickerDraw() virtual function, calling addPickableObject() in it. If the
 * rendering technology provides its own picking mechanism, it can be used by
 * implementing the customPick() function, which will not make use of the
 * internal ID picking, and pickerDraw() will not be called. This mechanism
 * also is present in ViewportLayers.
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static Viewport* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewerDelegate is the class that plugins should extend and implement.
 *
 * ViewerDelegateWrapper is the class that allows other plugin types to access
 * the ViewerDelegate plugin.
 *
 * ViewerDelegatePluginBase is the base class that provides the common methods
 * between ViewerDelegate and ViewerDelegateWrapper.
 *
 */
class ViewportPluginBase
{
public:

    /** @brief Constructor. */
    ViewportPluginBase();

    /** @brief Destructor. */
    virtual ~ViewportPluginBase();

public: /* Methods to be called by the plugin. */

    /**
     * @brief Gets the Viewport name.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @return The name given to the Viewport when it was created.
     */
    std::string getName();

    /**
     * @brief Gets the ViewerDelegate of thie Viewport.
     *
     * @return The ViewerDelegate that serves this Viewport.
     */
    ViewerDelegateWrapperPtr getViewerDelegate();

    /**
     * @brief Gets the width in pixels of the Viewport.
     *
     * @return The width in pixels of 0 if the Viewport hasn't been
     *      initialized.
     */
    unsigned int getWidth();

    /**
     * @brief Gets the Height in pixels of the Viewport.
     *
     * @return The height in pixels of 0 if the Viewport hasn't been
     *      initialized.
     */
    unsigned int getHeight();

    /**
     * @brief Gets the ViewportCamera of the Viewport.
     *
     * @return The ViewportCamera for this Viewport.
     */
    ViewportCameraWrapperPtr getActiveCamera();

    /**
     * @brief Creates a new ViewportCamera of the passed type.
     *
     * If a camera of the same name already exists, it will be deleted. If said
     * camera was the active camera, the new camera will become the active
     * camera.
     *
     * @param pluginName The name of the ViewportCamera plugin.
     * @param name The name of the new ViewportCamera.
     *
     * @return The newly created ViewportCamera.
     */
    ViewportCameraWrapperPtr addCamera(const std::string& pluginName,
                                       const std::string& name);

    /**
     * @brief Gets a ViewportCamera by name.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param name The name of the ViewportCamera.
     *
     * @return The ViewportCamera in this Viewport with the given name or 0
     *          if no such layer exists.
     */
    ViewportCameraWrapperPtr getCamera(const std::string& name) const;

    /**
     * @brief Gets a ViewportCamera by index.
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport.
     *
     * This function is exposed in the Viewport Python class as
     * Viewport.getCameraByIndex().
     *
     * @return The ViewportCamera in this Viewport with the given index or 0
     *          if no such layer exists.
     */
    ViewportCameraWrapperPtr getCamera(unsigned int index) const;

    /**
     * @brief Removes a ViewportCamera by name.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param name The name of the ViewportCamera to be removed.
     */
    void removeCamera(const std::string& name);

    /**
     * @brief Removes a ViewportCamera by name.
     *
     * This function is exposed in the Viewport Python class as
     * Viewport.removeCameraByIndex().
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport
     */
    void removeCamera(unsigned int index);

    /**
     * @brief Gets the number of ViewportCameras
     *
     * This function is exposed in the Viewport Python class.
     *
     * @return The number of ViewportCameras added to this Viewport.
     */
    unsigned int getNumberOfCameras() const;

    /**
     * @brief Gets the name of the ViewportCamera on a given index.
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport.
     *
     * @return The name of the ViewportCamera on the given index or empty
     *          string if it doesn't exist.
     */
    std::string getCameraName(unsigned int index);


    /**
     * @brief Gets the name of the given camera.
     *
     * @param camera ViewportCamera whose name is to be returned.
     *
     * @return The name of the given ViewportCamera or empty string if it
     *          doesn't exist in the viewport.
     */
    std::string getCameraName(ViewportCameraWrapperPtr camera);

    /**
     * @brief Gets the index of the ViewportCamera with a given name.
     *
     * @param name The name of the ViewportCamera.
     *
     * @return The index of the ViewportCamera with the given name or -1 if it
     *          doesn't exist.
     */
    int getCameraIndex(const std::string& name);

    /** @brief Destorys the passed ViewportCamera. */
    void removeCamera(ViewportCameraWrapperPtr camera);

    /** @brief Makes the passed ViewportCamera active. */
    void setActiveCamera(ViewportCameraWrapperPtr camera);

    /**
     * @brief Freezes/thaws the viewport view.
     *
     * When the view is frozen, viewers are not allowed to make changes in the
     * active camera. However, this is only a soft condition that viewers might
     * decide to ignore.
     *
     * The viewer delegate will emit an event
     * "__VIEWPORT_VIEW_FROZEN_STATE_CHANGED" containing the viewport's name and
     * the new state.
     */
    void setViewFrozen(bool isViewFrozen);

    /**
     * @brief Returns whether the viewport view is currently frozen.
     *
     * When the view is frozen, viewers are not allowed to make changes in the
     * active camera. However, this is only a soft condition that viewers might
     * decide to ignore.
     */
    bool isViewFrozen() const;

    /**
     * @brief Gets the View Matrix for this Viewport.
     *
     * @return The 4x4 Viewer Matrix represented by 16 doubles.
     */
    const double* getViewMatrix();

    /**
     * @brief Gets the View Matrix for this camera.
     *
     * @return The 4x4 View Matrix.
     */
    Matrix44d getViewMatrix44d();

    /**
     * @brief Gets the Projection Matrix for this Viewport.
     *
     * @return The 4x4 Projection Matrix represented by 16 doubles.
     */
    const double* getProjectionMatrix();

    /**
     * @brief Gets the Projection Matrix for this camera.
     *
     * @return The 4x4 Projection Matrix.
     */
    Matrix44d getProjectionMatrix44d();

    /**
     * @brief Returns whether the viewer state is dirty.
     *
     * Returns whether the viewport state is dirty, if so it will be re-drawn.
     *
     * This function is exposed in the Viewport Python class.
     */
    bool isDirty();

    /**
     * @brief Sets the viewer state of the Viewport.
     *
     * Sets whether the viewport state is dirty (due to something that happened
     * that led to the need or re-drawing the scene).
     *
     * This function is exposed in the Viewport Python class.
     */
    void setDirty(bool dirty);

    /* ViewportLayer management Methods */

    /**
     * @brief Adds a ViewportLayer.
     *
     * Instantiates a ViewportLayer and adds it to the end of the list of
     * layers on this Viewport.
     *
     * @param pluginName The name of the ViewportLayer plugin.
     * @param name The name for the new ViewportLayer.
     *
     * @return The newly created ViewportLayer.
     */
    ViewportLayerWrapperPtr addLayer(const std::string& pluginName,
                                     const std::string& name);

    /**
     * @brief Inserts a ViewportLayer.
     *
     * Instantiates a ViewportLayer and adds it a specified index on the list
     * of layers on this Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param pluginName The name of the ViewportLayer plugin.
     * @param name The name for the new ViewportLayer.
     * @param index The index of the parameter
     *
     * @return The newly created ViewportLayer.
     */
    ViewportLayerWrapperPtr insertLayer(const std::string& pluginName,
                                        const std::string& name,
                                        unsigned int index);

    /**
     * @brief Gets a ViewportLayer by name.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param name The name of the ViewportLayer.
     *
     * @return The ViewportLayer in this Viewport with the given name or 0
     *          if no such layer exists.
     */
    ViewportLayerWrapperPtr getLayer(const std::string& name) const;

    /**
     * @brief Gets a ViewportLayer by index.
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport.
     *
     * This function is exposed in the Viewport Python class as
     * Viewport.getLayerByIndex().
     *
     * @return The ViewportLayer in this Viewport with the given index or 0
     *          if no such layer exists.
     */
    ViewportLayerWrapperPtr getLayer(unsigned int index) const;

    /**
     * @brief Removes a ViewportLayer by name.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param name The name of the ViewportLayer to be removed.
     */
    void removeLayer(const std::string& name);

    /**
     * @brief Removes a ViewportLayer by name.
     *
     * This function is exposed in the Viewport Python class as
     * Viewport.removeLayerByIndex().
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport
     */
    void removeLayer(unsigned int index);

    /**
     * @brief Gets the number of ViewportLayers
     *
     * This function is exposed in the Viewport Python class.
     *
     * @return The number of ViewportLayers added to this Viewport.
     */
    unsigned int getNumberOfLayers() const;

    /**
     * @brief Gets the name of the ViewportLayer on a given index.
     *
     * @param index The position in the list of ViewportLayers on this
     *               Viewport.
     *
     * @return The name of the ViewportLayer on the given index or empty
     *          string if it doesn't exist.
     */
    std::string getLayerName(unsigned int index);

    /**
     * @brief Gets the index of the ViewportLayer on a given name.
     *
     * @param name The name of the ViewportLayer.
     *
     * @return The index of the ViewportLayer with the given name or -1 if it
     *          doesn't exist.
     */
    int getLayerIndex(const std::string& name);


    /* Active Manipulators Methods */

    /**
     * @brief Activates a Manipulator on the Viewport.
     *
     * Activates a Manipulator with the given plugin name for the specified
     * locations. This will instantiate a Manipulator plugin on this Viewport
     * that manipulates those locations. After this the manipulator will be
     * ready to be drawn and interacted with, which can either be implemented
     * directly on the Viewport or in a ViewportLayer dedicated to
     * manipulators. Only one Manipulator instance of the given type can be
     * active at once.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param pluginName The name of the Manipulator plugin.
     * @param locationPaths The locations to be manipulated.
     */
    void activateManipulator(const std::string& pluginName,
                             const std::vector<std::string>& locationPaths);

    /**
     * @brief Deactivates a Manipulator on the Viewport.
     *
     * Deactivates a Manipulator with the given plugin name that has been
     * previously activated in this Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param pluginName The name of the Manipulator plugin.
     */
    void deactivateManipulator(const std::string& pluginName);

    /**
     * @brief Deactivates all Manipulators on the Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * Deactivates all Manipulators that have been previously activated in this
     * Viewport.
     */
    void deactivateAllManipulators();

    /**
     * @brief Gets the number of active Manipulators on the Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * Deactivates all Manipulators that have been previously activated in this
     * Viewport.
     */
    unsigned int getNumberOfActiveManipulators();

    /**
     * @brief Gets an active Manipulator by index.
     *
     * Get the active Manipulator on the given index. Each active Manipulator
     * has an associated index internally. The indices are always sequential,
     * so this can be used to iterate through all the active Manipulators,
     * using getNumberOfActiveManipulators() to know how many there are.
     *
     * @note An index might not refer to the same Manipulator on consecutive
     *     calls.
     *
     * @param index The index of the Manipulator.
     *
     * @return The active Manipulator with the given index or 0 if no
     *          Manipulator has that index.
     */
    ManipulatorWrapperPtr getActiveManipulator(unsigned int index);

    /**
     * @brief Gets an active Manipulator by plugin name.
     *
     * @param pluginName The name of the Manipulator plugin.
     *
     * @return The active Manipulator with the given plugin name or 0 if there
     *          no active Manipulators with that pluginName.
     */
    ManipulatorWrapperPtr getActiveManipulator(const std::string& pluginName);

    /**
     * @brief Makes the Viewport OpenGL context current.
     *
     * @return true if successful or false on failure.
     */
    bool makeGLContextCurrent();

    /**
     * @brief Makes the Viewport OpenGL context no longer current.
     *
     * @return true if successful or false on failure.
     */
    bool doneGLContextCurrent();

    /**
     * @brief Determines if the Viewport's OpenGL context is the current
     *     context.
     *
     * @return true if the Viewport's OpenGL context is current; false
     *     otherwise.
     */
    bool isGLContextCurrent();

    /**
     * @brief Gets the default framebuffer object of the viewport.
     *
     * This is of type \c GLuint.
     *
     * By the time \c draw() is invoked, the default framebuffer object will
     * have been automatically bound. If the plug-in implementation binds a
     * different framebuffer object, this function can be used to retrieve the
     * original framebuffer object, which is where the plug-in needs to perform
     * the final drawing.
     *
     * @return The GL name that identifies the framebuffer object.
     */
    unsigned int getDefaultFramebufferObject();

    /**
     * @brief Enables a server-side OpenGL capability.
     *
     * @param cap Specifies a symbolic constant indicating a OpenGL capability.
     *
     * @note To temporarily change the OpenGL state, please use
     *     \c GLStateRestore.
     */
    void enableCap(uint32_t cap);

    /**
     * @brief Disables a server-side OpenGL capability.
     *
     * @param cap Specifies a symbolic constant indicating a OpenGL capability.
     *
     * @note To temporarily change the OpenGL state, please use
     *     \c GLStateRestore.
     */
    void disableCap(uint32_t cap);

    /**
     * @brief Picks the objects that are inside a given viewport region.
     *
     * Allows to query the scene for objects that are visible inside a given
     * rectangular region in the viewport. This region can be a single pixel
     * on the screen, by setting both its width and height to 1.
     *
     * The object picking can optionally be deep, meaning that all the objects
     * viewed inside the region will be picked even if they are currently
     * occluded by other objects. A non-deep picking means that only currently
     * visible objects inside the region will be picked.
     *
     * Picking can be implemented in two ways:
     *
     *   - Using the internal ID framebuffer based technique implemented via
     *     \c pickerDraw() and \c addPickableObject()
     *
     *   - Implementing a custom picking implemented using some third party
     *     technology via \c customPick().
     *
     * This function makes use of whatever picking technique is implemented in
     * this Viewport.
     *
     * This function returns a map of picking IDs to Attributes that represent
     * the picked objects and, optionally a depth value for when the picked
     * region is one single pixel. If \c pickerDraw() is used, as opposed to
     * \c customPick(), then the IDs are returned by the \c addPickableObject()
     * calls inside \c pickerDraw() and the Attributes are the ones passed to
     * it. They can be used to identify what objects have been picked.
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param deepPicking Specifies if all objects inside the region,
     *      including occluded ones, will be picked. If set to false, only the
     *      visible objects should be picked.
     * @param[out] pickedAttrs A map top be filled with \c FnPickId (key) to
     *      \c Attribute (value) pairs that represent the picked objects.
     *      Internally, this will be either populated by \c customPick() or by
     *      \c addPickableObject() calls inside \c pickerDraw().
     *      \c PickedAttrsMap has the same public interface as:
     *      \c std::map<FnPickId, FnAttributes::Attribute>.
     * @param[out] singlePointDepth The value pointed by this will be set
     *      with the GL depth of a single pixel when the region with and height
     *      are both 1 and this pointer is set to something other than NULL.
     *      This can be used to solve occlusion between picked objects from
     *      the Viewport and different ViewportLayers when, for example, the
     *      user clicks on a single pixel when selecting something.
     *
     */
    void pick(unsigned int x, unsigned int y,
              unsigned int w, unsigned int h,
              bool deepPicking,
              PickedAttrsMap& pickedAttrs,
              float* singlePointDepth=NULL);

    /** @brief Sets the pan and zoom active state. */
    void setPanAndZoomActive(bool active);

    /** @brief Returns whether the pan and zoom is active. */
    bool isPanAndZoomActive();

    /** @brief Returns the amount of horizontal offset. */
    float getHorizontalOffset();

    /** @brief Returns the amount of vertical offset. */
    float getVerticalOffset();

    /** @brief Returns the zoom value. */
    float getZoom();

/// @cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    static FnPluginHost* m_host;

    FnViewportHostSuite_v3* m_hostSuite;
    FnViewportHostHandle m_hostHandle;

private:
    ViewerDelegateWrapperPtr m_viewerDelegateWrapper;

/// @endcond
};


/** @brief The Viewport class to be extended by plugins. */
class Viewport : public ViewportPluginBase
{
public:
    // API methods (to be implemented / used by sub-classes)

    /** @brief Constructor. */
    Viewport();

    /** @brief Destructor. */
    virtual ~Viewport();

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Initializes the GL components of the Viewport.
     *
     * Called when a Viewport is created. It runs inside the correct GL
     * context. Can be used to initialize anything GL related. For example, in
     * the case of an OpenGL renderer this function should set up any required
     * OpenGL context rendering flags, defining display lists, etc.
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void setup() = 0;

    /**
     * @brief Cleans up the Viewport resources.
     *
     * Called when a Viewport is removed by the ViewerDelegate.
     */
    virtual void cleanup() = 0;

    /**
     * @brief Processes UI events.
     *
     * Called whenever a user interaction event occurs. This will be also
     * propagated through the ViewerLayers of this Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return true if the event has been handled, false otherwise.
     */
    virtual bool event(const FnEventWrapper& eventData);

    /**
     * @brief Draws the scene.
     *
     * Called when the scene needs to be drawn in the correct GL context. In a
     * non-GL renderer the generated image should be drawn in the GL
     * framebuffer in order to be displayed. This will be also propagated
     * through the ViewerLayers of this Viewport.
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void draw();

    /**
     * @brief Processes Viewport resizing.
     *
     * Called when the Viewport widget is resized. This can be extended by
     * sub-clases to accomodate viewport size changes in the renderer.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param width The viewport width in pixels.
     * @param height The viewport height in pixels.
     */
    virtual void resize(unsigned int width, unsigned int height);

    /**
     * @brief Called when the mouse hovers the Viewport.
     *
     * The Viewport can detect when the mouse is hovering it so that, for
     * example, objects under the mouse pointer can be highlighted. This is
     * called with the correct GL context so that GL based picking can be
     * executed.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param isHovering Flag that specifies if the mouse is hovering the
     *     viewport. If false, then the mouse left the viewport. In this case
     *     x and y should be ignored.
     *
     * @param x The horizontal pixel coordinate of the mouse pointer in the
     *     Viewport's local coordinate system.
     *
     * @param y The vertical pixel coordinate of the mouse pointer in the
     *     Viewport's local coordinate system.
     */
    virtual void hover(bool isHovering, int x, int y);

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the Viewport.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
                           FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * This function is exposed in the Viewport Python class.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

    /* General Methods */

    /**
     * @brief Freezes the Viewport when the Viewer is hidden.
     *
     * Allows the Viewport to freeze its activities when the Viewer is not
     * visible. This allows the ViewerDelegate to stop any kind of unecessary
     * processing that might happen durging that time.
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void freeze() = 0;

    /**
     * @brief Thaws the Viewport when the Viewer is shown.
     *
     * Allows the Viewport to restart its activities when the Viewer becomes
     * visible. This restarts the activities paused by freeze().
     *
     * This function is exposed in the Viewport Python class.
     */
    virtual void thaw() = 0;

    /**
     * @brief Flush plugin Caches.
     *
     * Allows to discard any cache for this plugin when a Flush Caches
     * event occurs.
     */
    static void flush() {}

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /**
     * @brief Draws an ID pass to be used by the optional internal picking.
     *
     * This allows this Viewport to make use of the internal ID framebuffer
     * pass technique provided to the Viewport and ViewportLayers. This
     * function should draw all the pickable objects into the current GL
     * framebuffer using an ID color. Inside this function each pickable object
     * has to be registered using \c addPickableObject(), which will return a
     * FnPickId. This id can be converted into a color using \c pickIdToColor()
     * (see FnPickingTypes.h), the objects should be rendered without any kind
     * of antialiasing and with that flat color, so that they cover their
     * pickable pixels.
     *
     * This will be called by Katana if customPick() returns false or if it is
     * not implemented.
     *
     * This function receives a list of Attributes that refer to pickable
     * objects to be ignored in this render. This is internally used to do a
     * multi-pass id render for deep picking, in which an onion peeling
     * technique is used to detect all the occluded objects. On each iteration
     * this function should not render the objects that were detected in
     * previous iterations. These Attributes correspond to the ones passed
     * previously to \c addPickableObject().
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param ignoreAttrs Contains information about the objects that should
     *      not be rendered here. \c PickedAttrsMap has the same
     *      public interface as \c std::map<FnPickId, FnAttributes::Attribute>.
     */
    virtual void pickerDraw(unsigned int x, unsigned int y,
                            unsigned int w, unsigned int h,
                            const PickedAttrsMap& ignoreAttrs) {}

    /**
     * @brief Overrides the internal ID picking using a third party technique.
     *
     * If the technology used in the Viewport implements its own picking, or if
     * a GL ID pass is not feasible, then this function allows to override the
     * internal ID picking and to implement the picking of what is present in
     * the scene. If this is implemented and returns true, then \c pickerDraw()
     * will never be called by Katana. This is called internally by \c pick(),
     * which will return the returned values of this function.
     *
     * An example of the use of this function is when using a non-realtime
     * renderer that is able to produce its own ID pass or when the renderer
     * data structures allows to query the geometry present inside the frustrum
     * defined by the picking area.
     *
     * There is no need to call \c addPickableObject() inside this function.
     *
     * This should return a map of \c FnPickId to Attributes, similar to the
     * one returned by \c pickerDraw(), which can identify or contain
     * information about each picked objects. The
     *
     * This can optionally return a depth value for when the picked region is
     * one single pixel.
     *
     * @param x The region origin X component in viewport pixels.
     * @param y The region origin Y component in viewport pixels.
     * @param w The region width in viewport pixels.
     * @param h The region height in viewport pixels.
     * @param deepPicking Specifies if all objects inside the region,
     *      including occluded ones, will be picked. If set to false, only the
     *      visible objects should be picked.
     * @param[out] pickedAttrs A map top be filled with \c FnPickId (key) to
     *      \c Attribute (value) pairs that represent the picked objects.
     *      Internally, this will be either populated by \c customPick() or by
     *      \c addPickableObject() calls inside \c pickerDraw().
     *      \c PickedAttrsMap has the same public interface as:
     *      \c std::map<FnPickId, FnAttributes::Attribute>.
     * @param[out] singlePointDepth The value pointed by this will be set
     *      with the GL depth of a single pixel when the region with and height
     *      are both 1 and this pointer is set to something other than NULL.
     *      This can be used to solve occlusion between picked objects from
     *      the Viewport and different ViewportLayers when, for example, the
     *      user clicks on a single pixel when selecting something.
     */
    virtual bool customPick(unsigned int x, unsigned int y,
                            unsigned int w, unsigned int h,
                            bool deepPicking,
                            PickedAttrsMap& pickedAttrs,
                            float* singlePointDepth=NULL);

    /**
     * @brief Registers a pickable object during \c pickerDraw().
     *
     * This should be called inside \c pickerDraw() in order to let the
     * internal ID picking system know about each pickable object.
     *
     * The ID returned by this function can be used to both identify an object
     * in some data structure implemented inside this plugin and to define the
     * color to be used by the object when rendering itself in the current ID
     * framebuffer in \c pickerDraw(), via the \c pickIdToColor() function
     * (see FnPickingTypes.h). Also see FnGLShaderProgram.h, which implements a
     * way of loading GLSL shaders.
     *
     * The Attribute passed to this function can contain further information
     * about the pickable object. For some generic cases there will be
     * Attribute conventions that prescribe how this Attribute should be
     * structured in order to be recognized as some typical objects. This
     * allows some out-of-the-box or third party plugins, like other
     * ViewportLayer plugins, to identify objects like, for example, a
     * location:
     *   - A location should be identified using a GroupAttribute containing at
     *     least a child StringAttribute named "location" and with a single
     *     value containing the full location path. This will be recognized by
     *     the ViewportLayer plugin "SelectionLayer" shipped with Katana, which
     *     is responsible for dealing with locations selection in the Viewer.
     *
     * Any kind of rendered object, other than locations, can be pickable. For
     * example, a handle of some overlay widget that can be manipulated using
     * the mouse. For this, each Viewport will make use of some Attribute
     * convention that is suitable to identify its own pickable objects.
     *
     * The \c pick() function will return the \c FnPickID / \c Attribute pairs
     * of all picked objects inside its region.
     *
     * @param attr The Attribute that describes the pickable object.
     * @return A picking ID assigned to the pickable object.
     */
    FnPickId addPickableObject(FnAttribute::Attribute attr);

/// @cond FN_INTERNAL_DEV
public:
    static FnViewportPluginSuite_v2 createSuite(
        FnViewportPluginHandle (*create)(FnViewportHostHandle hostHandle));

    static FnViewportPluginHandle newViewportHandle(Viewport* viewport);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnViewportHostHandle m_hostHandle);

/// @endcond
};


/** @brief The Viewport class accessed by other plugins. */
class ViewportWrapper : public ViewportPluginBase
{
public:
    ViewportWrapper(FnPluginHost* host, FnViewportHostHandle hostHandle,
                    FnViewportPluginHandle pluginHandle,
                    FnViewportPluginSuite_v2* pluginSuite);

    ~ViewportWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of Viewport
     * that is a registerd plugin. The type of that child class needs to be
     * specified in the template, so that it can be properly cast. If the
     * specified type  doesn't match the plugin type then NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /**
     * @brief Processes UI events.
     *
     * @return true if the event has been handled, false if not.
     */
    bool event(const FnEventWrapper& eventData);

    /** @brief Sets a Viewport option. */
    void setOption(OptionIdGenerator::value_type optionId, FnAttribute::Attribute attr);

    /** @brief Gets a Viewport option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);

    /** @brief Sets a Viewport option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /** @brief Gets a Viewport option. */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
private:
    Viewport* getPluginPointer();

    /** The Viewport plug-in. */
    FnViewportPluginSuite_v2* m_pluginSuite;
    FnViewportPluginHandle m_pluginHandle;

/// @endcond
};

/// Pointer to a wrapper of a Viewport.
typedef std::shared_ptr<ViewportWrapper> ViewportWrapperPtr;

/** @} */

}  // namespace ViewerAPI
}  // namespace Katana
}  // namespace Foundry


/// @cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnViewportPluginStruct
{
public:
    FnViewportPluginStruct(
        Foundry::Katana::ViewerAPI::Viewport* viewport)
    : m_viewport(viewport)
    {}

    ~FnViewportPluginStruct()
    {};

    Foundry::Katana::ViewerAPI::Viewport* getViewport()
    {
        return m_viewport.get();
    }

private:
    std::shared_ptr<Foundry::Katana::ViewerAPI::Viewport> m_viewport;
};

// Plugin Registering Macro.
#define DEFINE_VIEWPORT_PLUGIN(VIEWPORT_CLASS)                                \
                                                                              \
    FnPlugin VIEWPORT_CLASS##_plugin;                                         \
    FnViewportPluginHandle VIEWPORT_CLASS##_create(                           \
        FnViewportHostHandle hostHandle)                                      \
    {                                                                         \
        Foundry::Katana::ViewerAPI::Viewport* viewport =                      \
            VIEWPORT_CLASS::create();                                         \
                                                                              \
        viewport->setHostHandle(hostHandle);                                  \
        return Foundry::Katana::ViewerAPI::Viewport::newViewportHandle(       \
            viewport);                                                        \
    }                                                                         \
                                                                              \
    FnViewportPluginSuite_v2 VIEWPORT_CLASS##_suite =                         \
            Foundry::Katana::ViewerAPI::Viewport::createSuite(                \
                    VIEWPORT_CLASS##_create);                                 \
                                                                              \
    const void* VIEWPORT_CLASS##_getSuite()                                   \
    {                                                                         \
        return &VIEWPORT_CLASS##_suite;                                       \
    }

/// @endcond

#endif
