// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_LOCATIONEVENT_H
#define FNVIEWER_LOCATIONEVENT_H

#include <FnAttribute/FnAttribute.h>
#include <FnViewer/suite/FnViewerDelegateSuite.h>

#include <string>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnViewerLocationEvent Viewer Location Event
 * @{
 */

/** @brief Struct containing Viewer Location Event state changes */
struct ViewerLocationStateChanges
{
    /**
     * Default constructor.
     */
    ViewerLocationStateChanges() :
        locationRemoved(false),
        attributesUpdated(false),
        localXformUpdated(false),
        excludedChanged(false),
        descendantsHiddenChanged(false) {}

    /**
     * Constructor for creating a \c ViewerLocationStateChanges object based on
     * a given viewer location event bit field of flags.
     *
     * @param flags The viewer location event bit field to parse in order to
     *     initialize the new object's boolean member variables.
     */
    explicit ViewerLocationStateChanges(FnViewerLocationEventBitField flags)
    {
        parseBitfield(flags);
    }

private:
    FnViewerLocationEventBitField writeBitfield() const
    {
        FnViewerLocationEventBitField flags = 0;
        if (locationRemoved)
        {
            flags |= kFnViewerLocationEvent_locationRemoved;
        }
        if (attributesUpdated)
        {
            flags |= kFnViewerLocationEvent_attributesUpdated;
        }
        if (localXformUpdated)
        {
            flags |= kFnViewerLocationEvent_localXformUpdated;
        }
        if (excludedChanged)
        {
            flags |= kFnViewerLocationEvent_excludedChanged;
        }
        if (descendantsHiddenChanged)
        {
            flags |= kFnViewerLocationEvent_descendantsHiddenChanged;
        }
        return flags;
    }

    void parseBitfield(FnViewerLocationEventBitField flags)
    {
        locationRemoved = flags & kFnViewerLocationEvent_locationRemoved;
        attributesUpdated = flags & kFnViewerLocationEvent_attributesUpdated;
        localXformUpdated = flags & kFnViewerLocationEvent_localXformUpdated;
        excludedChanged = flags & kFnViewerLocationEvent_excludedChanged;
        descendantsHiddenChanged = flags &
                kFnViewerLocationEvent_descendantsHiddenChanged;
    }

public:
    /// The scene graph location and all descendants have been removed from the
    /// scene graph.
    bool locationRemoved;

    /// The location has either been newly added, or its attributes have been
    /// updated. As 'xform' attributes are included, this is always true if
    /// localXformUpdated is true.
    bool attributesUpdated;

    /// The local xform as been updated (the local matrix and/or its
    /// 'isAbsolute' state).
    bool localXformUpdated;

    /// The 'excluded' property has changed (see ViewerLocationEvent).
    bool excludedChanged;

    /// The 'descendantsHidden' property has changed (see ViewerLocationEvent).
    bool descendantsHiddenChanged;

    // @cond FN_INTERNAL_DEV
    friend struct ViewerLocationEvent;
    // @endcond
};

/** @brief Struct describing a Viewer Location Event*/
struct ViewerLocationEvent
{
    /**
     * Default constructor.
     */
    ViewerLocationEvent()
        : localXformIsAbsolute(false),
          excluded(false),
          descendantsHidden(false),
          isVirtualLocation(false) {}

    /**
     * Constructor for creating a \c ViewerLocationEvent based on a given
     * viewer location event data structure.
     *
     * @param cStruct The viewer location event data structure to copy.
     */
    explicit ViewerLocationEvent(const FnViewerLocationEventStruct& cStruct)
    : locationPath(cStruct.locationPath),
      attributes(FnAttribute::Attribute::CreateAndRetain(cStruct.attributes)),
      localXformMatrix(
              FnAttribute::Attribute::CreateAndRetain(cStruct.localXformMatrix))
    {
        parseBitfield(cStruct.flags);
    }

    /**
     * @return A viewer location event data structure containing the details of
     *     this viewer location event.
     */
    FnViewerLocationEventStruct getCStruct() const
    {
        FnViewerLocationEventStruct event;
        event.locationPath = locationPath.c_str();
        event.attributes = attributes.getHandle();
        event.localXformMatrix = localXformMatrix.getHandle();
        event.flags = writeBitfield();
        return event;
    }

private:
    FnViewerLocationEventBitField writeBitfield() const
    {
        FnViewerLocationEventBitField flags = 0;

        if (localXformIsAbsolute)
        {
            flags |= kFnViewerLocationEvent_localXformIsAbsolute;
        }
        if (excluded)
        {
            flags |= kFnViewerLocationEvent_excluded;
        }
        if (descendantsHidden)
        {
            flags |= kFnViewerLocationEvent_descendantsHidden;
        }
        if (isVirtualLocation)
        {
            flags |= kFnViewerLocationEvent_isVirtualLocation;
        }

        flags |= stateChanges.writeBitfield();

        return flags;
    }

    void parseBitfield(FnViewerLocationEventBitField flags)
    {
        localXformIsAbsolute = flags &
                kFnViewerLocationEvent_localXformIsAbsolute;
        excluded = flags & kFnViewerLocationEvent_excluded;
        descendantsHidden = flags & kFnViewerLocationEvent_descendantsHidden;
        isVirtualLocation = flags & kFnViewerLocationEvent_isVirtualLocation;

        stateChanges.parseBitfield(flags);
    }

public:
    /// Scene graph location path (always provided).
    std::string locationPath;

    /// Location attributes (always provided).
    FnAttribute::GroupAttribute attributes;

    /// Location local transform as a 4x4 matrix (16 value DoubleAttribute) if
    /// one exists (always provided), otherwise an invalid attribute (which
    /// should be interpreted as an identity matrix, or otherwise ignored).
    FnAttribute::DoubleAttribute localXformMatrix;

    /// True if the local transform is reset to identity using the 'origin'
    /// attribute convention. If true, the location's ancestors' transforms do
    /// not contribute to the global transform of this location and its
    /// descendants.
    bool localXformIsAbsolute;

    /// A location can be 'excluded' if it is not explicitly made visible, but
    /// a descendant location is. A location is also excluded if it is
    /// explicitly hidden or has a hidden ancestor. In these cases, any
    /// representation of the non-leaf location should not be drawn.
    bool excluded;

    /// This property determines whether or not Viewer proxies are displayed,
    /// and is provided for use by custom proxy implementations. It is true if
    /// all of the location's potential [non-virtual] descendants are
    /// explicitly hidden, or it has none to hide. This property is determined
    /// when the location is produced, and cannot rely on the final number of
    /// existing children once they are also cooked: i.e. this property be
    /// false for leaf locations that have *potential* children are deleted
    /// when cooked.
    bool descendantsHidden;

    /// Virtual locations are not part of the scene graph, but are generated as
    /// additional children based on location attributes, such as Viewer proxy
    /// locations. Note that location paths of virtual locations should always
    /// be distinguishable from real scene graph location paths.
    bool isVirtualLocation;

    /// Location state changes signalled by this event.
    ViewerLocationStateChanges stateChanges;
};

/** @} */

}  // namespace ViewerAPI
}  // namespace Katana
}  // namespace Foundry

#endif
