// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_VIEWERDELEGATECOMPONENT_H
#define FNVIEWER_VIEWERDELEGATECOMPONENT_H

#include <FnViewer/plugin/FnViewerDelegate.h>
#include <FnViewer/plugin/FnViewerLocationEvent.h>
#include <FnViewer/suite/FnViewerDelegateComponentSuite.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <string>
#include <vector>
#include <memory>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnViewerDelegateComponent ViewerDelegateComponent Plugin
 * @{
 */

/**
 * @brief Extends the functionality of an existing ViewerDelegate.
 *
 * A ViewerDelegateComponent allows to extend the support for more location
 * types or attribute conventions that are not supported by the ViewerDelegate.
 * Together with the ViewportLayer plugin type, an already compiled Viewer can
 * be extended. This can also be used to modularize the features of a
 * ViewerDelegate. For example each location type can have its own
 * ViewerDelegateComponent, which can be added/removed to the Viewer at
 * runtime.
 *
 * Whenever a ViewerDelegateComponent has to access some private data that is
 * kept in the ViewerDelegate the method ViewerDelegate::getPrivateData() can
 * be used to access that data. In this case the compiler used to build the
 * ViewerDelegate and the ViewerDelegateComponent have to be the same.
 *
 * A ViewerDelegateComponent 'locationEvent' callback function can return true
 * in order to inform other ViewerDelegateComponents and the ViewerDelegate that
 * it is handling the location. This may allow existing functionality in the
 * ViewerDelegate to be removed/ignored. For example, if the ViewerDelegate
 * has 2 ViewerDelegateComponent (A and B) and a new one (C) is added, then if
 * C::locationEvent() returns true, then the locationEvent() callback of A, B,
 * and the ViewerDelegate will be called with locationHandled=true.
 *
 * ViewerDelegateComponent is the class that plugins should extend and
 * implement.
 *
 * ViewerDelegateComponentWrapper is the class that allows other plugin types
 * to access the ViewerDelegateComponent plugin.
 *
 * ViewerDelegateComponentPluginBase is the base class that provides the common
 * methods between ViewerDelegateComponent and ViewerDelegateComponentWrapper.
 *
 */
class ViewerDelegateComponentPluginBase
{
public:

    /** @brief Constructor. */
    ViewerDelegateComponentPluginBase();

    /** @brief Destructor. */
    virtual ~ViewerDelegateComponentPluginBase();

public: /* Methods to be called by the plugin. */

    /**
     * @brief Returns the viewer delegate that this component is associated
     *     with.
     *
     * @return A pointer to the ViewerDelegate.
     */
    ViewerDelegateWrapperPtr getViewerDelegate();

    /**
     * @brief Gets cooked attributes for a location.
     *
     * Gets the cached cooked attributes for the location. Returns an
     * invalid group attribute if the location hasn't been cooked.
     *
     * During the manipulation of an attribute by a Manipulator, the last
     * value set by it will be returned instead of the cached cooked
     * attribute. This allows faster interactive representation of the
     * scene graph during the manipulation, since the cooking might not be
     * fast enough to provide an interactive speed.
     *
     * @param locationPath The location path.
     *
     * @return The attributes of the location.
     */
    FnAttribute::GroupAttribute getAttributes(const std::string& locationPath);

///@cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    FnViewerDelegateComponentHostSuite_v2* m_hostSuite;
    FnViewerDelegateComponentHostHandle m_hostHandle;

    static FnPluginHost* m_host;

private:
    ViewerDelegateWrapperPtr m_viewerDelegateWrapper;

///@endcond
};


/** @brief The ViewerDelegateComponent class to be extended by plugins. */
class ViewerDelegateComponent : public ViewerDelegateComponentPluginBase
{
public:
    ViewerDelegateComponent();
    virtual ~ViewerDelegateComponent();

public: /* Virtual functions to be extended by the plugin. */

    /** @brief Initializes the ViewportDelegateComponent's resources. */
    virtual void setup() = 0;

    /** @brief Cleans up the ViewportDelegateComponent's resources. */
    virtual void cleanup() = 0;

    /* Scene Graph Events */

    /**
     * @brief Notification of scene graph location state changes.
     *
     * @param event Struct containing location event information (see
     *     "FnViewerLocationEvent.h" for details).
     * @param locationHandled True if an already processed (later added)
     *     ViewerDelegateComponent has stated that it is handling this
     *     location by returning true. In this case, the
     *     ViewerDelegateComponent should not draw its own representation.
     * @return true if the other ViewerDelegateComponents added to the
     *      ViewerDelegate before this one should receive locationHandled=true,
     *      false otherwise.
     */
    virtual bool locationEvent(const ViewerLocationEvent& event,
            bool locationHandled) = 0;

    /**
     * @brief Called when the location selection changes in Katana.
     *
     * @param locationPaths The selected location paths.
     */
    virtual void locationsSelected(
        const std::vector<std::string>& locationPaths) = 0;

    /**
     * @brief Queries the processing of the Viewer Delegate Component plug-in.
     *
     * @return true if the plug-in is processing anything that may result in a
     *     change to the Viewer.
     */
    virtual bool isProcessing() const { return false; }

    /**
     * @brief Flush plugin Caches.
     *
     * Allows to discard any cache for this plugin when a Flush Caches
     * event occurs.
     */
    static void flush() {}

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /* Options setting and getting. */

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the
     * ViewerDelegateComponent.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
                           FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

    /**
     * @brief Gets the bounds of the given location.
     *
     * @param locationPath The location path.
     *
     * @return The bounds of the location (or an invalid attribute if not
     *     found).
     */
    virtual FnAttribute::DoubleAttribute getBounds(
        const std::string& locationPath);

    /**
     * @brief Calculates the extent of the given location.
     *
     * It calculates the extent of the location based on its geometry.
     *
     * @param locationPath The location path.
     *
     * @return The extent of the location (or an invalid attribute if no
     *     extent was computed).
     */
    virtual FnAttribute::DoubleAttribute computeExtent(
        const std::string& locationPath);

/// @cond FN_INTERNAL_DEV
public:
    static FnViewerDelegateComponentPluginSuite_v2 createSuite(
        FnViewerDelegateComponentPluginHandle (*create)(
            FnViewerDelegateComponentHostHandle hostHandle));
    static FnViewerDelegateComponentPluginHandle newViewerDelegateComponentHandle(
        ViewerDelegateComponent* viewerDelegateComponent);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnViewerDelegateComponentHostHandle m_hostHandle);
    FnViewerDelegateComponentHostHandle getHostHandle();

/// @endcond
};


/** @brief The ViewerDelegateComponent class accessed by other plugins. */
class ViewerDelegateComponentWrapper : public ViewerDelegateComponentPluginBase
{
public:
    ViewerDelegateComponentWrapper(
        FnPluginHost* host,
        FnViewerDelegateComponentHostHandle hostHandle,
        FnViewerDelegateComponentPluginHandle pluginHandle,
        FnViewerDelegateComponentPluginSuite_v2* pluginSuite);

    ~ViewerDelegateComponentWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of
     * ViewerDelegateComponent that is a registerd plugin. The type of that
     * child class needs to be specified in the template, so that it can be
     * properly cast. If the specified type  doesn't match the plugin type then
     * NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /** @brief Sets a generic option. */
    void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);
    /** @brief Sets a generic option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(const std::string& name);

    /** @brief Gets the bounds of the given location. */
    FnAttribute::DoubleAttribute getBounds(const std::string& location);
    /** @brief Calculates the extent of the given location. */
    FnAttribute::DoubleAttribute computeExtent(const std::string& location);

/// @cond FN_INTERNAL_DEV
private:
    ViewerDelegateComponent* getPluginPointer();

    /** The ViewerDelegateComponent plug-in. */
    FnViewerDelegateComponentPluginSuite_v2* m_pluginSuite;
    FnViewerDelegateComponentPluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ViewerDelegateComponentWrapper>
    ViewerDelegateComponentWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry



///@cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnViewerDelegateComponentPluginStruct
{
public:
    FnViewerDelegateComponentPluginStruct(
        Foundry::Katana::ViewerAPI::ViewerDelegateComponent* viewerDelegateComponent)
    : m_ViewerDelegateComponent(viewerDelegateComponent)
    { }

    ~FnViewerDelegateComponentPluginStruct()
    { }

    Foundry::Katana::ViewerAPI::ViewerDelegateComponent* getViewerDelegateComponent()
    {
        return m_ViewerDelegateComponent.get();
    }

private:
    std::shared_ptr<Foundry::Katana::ViewerAPI::ViewerDelegateComponent>
        m_ViewerDelegateComponent;
};

// Plugin Registering Macro.
#define DEFINE_VIEWER_DELEGATE_COMPONENT_PLUGIN(VIEWER_DELEGATE_COMPONENT_CLASS)        \
                                                                              \
    FnPlugin VIEWER_DELEGATE_COMPONENT_CLASS##_plugin;                        \
                                                                              \
    FnViewerDelegateComponentPluginHandle VIEWER_DELEGATE_COMPONENT_CLASS##_create(     \
        FnViewerDelegateComponentHostHandle hostHandle)                       \
    {                                                                         \
        Foundry::Katana::ViewerAPI::ViewerDelegateComponent* viewerDelegateComponent =  \
            VIEWER_DELEGATE_COMPONENT_CLASS::create();                        \
                                                                              \
        viewerDelegateComponent->setHostHandle(hostHandle);                   \
        return Foundry::Katana::ViewerAPI::ViewerDelegateComponent::newViewerDelegateComponentHandle( \
            viewerDelegateComponent);                                         \
    }                                                                         \
                                                                              \
    FnViewerDelegateComponentPluginSuite_v2                                   \
        VIEWER_DELEGATE_COMPONENT_CLASS##_suite =                             \
            Foundry::Katana::ViewerAPI::ViewerDelegateComponent::createSuite( \
                    VIEWER_DELEGATE_COMPONENT_CLASS##_create);                \
                                                                              \
    const void* VIEWER_DELEGATE_COMPONENT_CLASS##_getSuite()                  \
    {                                                                         \
        return &VIEWER_DELEGATE_COMPONENT_CLASS##_suite;                      \
    }

///@endcond

#endif
