// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_VIEWERDELEGATE_H
#define FNVIEWER_VIEWERDELEGATE_H

#include <FnViewer/suite/FnViewerDelegateSuite.h>
#include <FnViewer/plugin/FnOptionIdGenerator.h>
#include <FnViewer/plugin/FnMathTypes.h>
#include <FnViewer/plugin/FnViewerLocationEvent.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <string>
#include <vector>
#include <memory>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

// Forward declaration.
class ViewerDelegateComponentWrapper;
typedef std::shared_ptr<ViewerDelegateComponentWrapper>
    ViewerDelegateComponentWrapperPtr;
class ViewportWrapper;
typedef std::shared_ptr<ViewportWrapper> ViewportWrapperPtr;

/**
 * \defgroup FnViewerDelegate ViewerDelegate Plugin
 * @{
 */

/**
 * @brief Main data source for a Viewer that feeds the different Viewports.
 *
 * A ViewerDelegate can serve several Viewports with data from location
 * cooking, deletion, opening and closing events that are delegated to it.
 * It uses a Geolib3 client that listens to a given Op-Tree (see setViewOp()).
 * It can access all the attributes of all the locations cooked so far via
 * getAttributes().
 *
 * Scene graph locations are made available through one of the following scene
 * views:
 *
 *  - Primary (kFnViewerDelegatePrimary):
 *    Observes all Viewer-visible locations (expanded/pinned/included).
 *    Location changes are signalled via ViewerDelegate::locationEvent().
 *
 *  - Sources (kFnViewerDelegateSources):
 *    Observes all subtrees activated as a 'source', e.g. an instance source.
 *    Location changes are signalled via ViewerDelegate::sourceLocationEvent().
 *
 * Scene graph query methods such as ViewerDelegate::getAttributes() and
 * ViewerDelegate::getWorldXform() query the primary scene view unless the
 * 'sceneView' parameter stipulates otherwise.
 *
 * Any data structure that is shared between different Viewports should be
 * maintained by the ViewerDelegate. It can access all of its Viewports by name
 * or by index (see getViewport()).
 *
 * Since it has access to the cooked attributes for each location it can also
 * get the Manipulators that are compatible with a location (i.e. that can
 * manipulate that location) by looking at its attributes and matching each
 * Manipulator's requirements (see getCompatibleManipulatorsInfo()).
 *
 * It can also communicate with the Python layer via callPythonCallback().
 *
 * An already compiled ViewerDelegate can be extended by the plugin type
 * ViewerDelegateComponent. Several of these can be added to allow new location
 * types and attributes to be processed. ViewerDelegateComponents can inform
 * other ViewerDelegateComponents and the ViewerDelegate that they are handling
 * a location by return 'true' from their locationEvent() callback. This may
 * allow existing functionality in the ViewerDelegate to be removed/ignored.
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static ViewerDelegate* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewerDelegate is the class that plugins should extend and implement.
 *
 * ViewerDelegateWrapper is the class that allows other plugin types to access
 * the ViewerDelegate plugin.
 *
 * ViewerDelegatePluginBase is the base class that provides the common methods
 * between ViewerDelegate and ViewerDelegateWrapper.
 *
 */
class ViewerDelegatePluginBase
{
public:

    /** @brief Constructor. */
    ViewerDelegatePluginBase();

    /** @brief Destructor. */
    virtual ~ViewerDelegatePluginBase();

public: /* Methods to be called by the plugin. */

    /* Geolib3 Methods */

    /**
     * @brief Gets cooked attributes for a location.
     *
     * Gets the cached cooked attributes for the location. Returns an
     * invalid group attribute if the location hasn't been cooked.
     *
     * During the manipulation of an attribute by a Manipulator, the last
     * value set by it will be returned instead of the cached cooked
     * attribute. This allows faster interactive representation of the
     * scene graph during the manipulation, since the cooking might not be
     * fast enough to provide an interactive speed.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPath The location path.
     * @param sceneView Scene view to query.
     *
     * @return The attributes of the location.
     */
    FnAttribute::GroupAttribute getAttributes(
        const std::string& locationPath,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Gets the location's local space transform.
     *
     * @param [in] locationPath The location.
     * @param [out] isAbsolute Set to true if the local transform is reset to
     *     identity using the 'origin' attribute convention. If true, the
     *     location's ancestors' transforms do not contribute to the global
     *     transform of this location and its descendants.
     * @param sceneView Scene view to query.
     *
     * @return Gets the location's local space transform as a matrix.
     */
    Matrix44d getLocalXform(
        const std::string& locationPath, bool& isAbsolute,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Gets the location's world space transform.
     *
     * @param locationPath The location.
     * @param sceneView Scene view to query.
     *
     * @return Gets the location's world space transform as a matrix.
     */
    Matrix44d getWorldXform(
        const std::string& locationPath,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Gets the location parent's world space transform.
     *
     * @param locationPath The location.
     * @param sceneView Scene view to query.
     *
     * @return Gets the location parent's world space transform as a matrix.
     */
    Matrix44d getParentXform(
        const std::string& locationPath,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Transform from rotate/scale/translate space to world space.
     *
     * NOTE: in this case a "component" refers to a transformation component
     * (scale, rotate, translate), rather than a ViewerDelegateComponent.
     *
     * This defines the transformation from a transform component space (scale,
     * translate, rotate) to world space. It is defined by the transform of all
     * the components that are applied before the one specified by componentName
     * concatenated by the location's parent transform. This transform will be
     * affected by the transform order.
     *
     * The includeComponent flag defines if the component transformation should
     * be included in the result or not. For example, for a TRS transform order
     * in a Rotate manipulator and the flag includeComponent set to false, this
     * will return the following matrix:
     *    Translate * XformGroupBefore * ParentLocation
     * And if includeComponent is set to true, then this is the result:
     *    Rotate * Translate * XformGroupBefore * ParentLocation
     *
     * The componentName argument defines name of the xform component
     * attribute. For example: "translate" for xform.interactive.translate". In
     * the case of a rotation, in which there are several attributes suffixed
     * with the axis letter, a "?" should be added instead of the axis name
     * (example: "xform.interactive.rotate?") This way the whole component,
     * with all its 3 attributes, will be considered. The includeComponent will
     * also define if the components that match this wildcard will all be
     * included or all excluded from the result.
     *
     * If the location has no xform attribute or if no component attribute is
     * found then an identity matrix is returned.
     *
     * @param locationPath The location with the transform.
     * @param groupName The name of the parent GroupAttribute that contains
     *     the transform attributes. Example: "xform.interactive"
     * @param componentName The name of the attribute of the transform
     *     component. The wildcard "?" at the end of this string will match
     *     all the transform components prefixed by componentName that are
     *     followed by an axis letter at the end. Examples: "transform",
     *     "rotateX" or "rotate?".
     * @param includeComponent Specifies if the component specified by
     *     componentName should be included in the resulting matrix.
     * @param sceneView Scene view to query.
     * @param includeParentXform Specifies parent locations xform should be
     *     included in the resulting matrix.
     * @param includeBeforeGroup Specifies if the transform groups before
     *     the one specified by componentName (example: "xform.group1" before
     *     "xform.interactive") should also be included in the resulting
     *     transform.
     * @param includeAfterGroup Specifies if the transform groups after
     *     the one specified by componentName (example: "xform.group1" after
     *     "xform.interactive") should also be included in the resulting
     *     transform.
     *
     * @return A matrix representing the transform component-to-world
     *     transform. It is given by the multiplication of all the components
     *     that are applied before the specified component.
     *
     */
    Matrix44d getPartialXform(
        const std::string& locationPath, const std::string& groupName,
        const std::string& componentName, bool includeComponent,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary,
        bool includeParentXform = true, bool includeBeforeGroup = true,
        bool includeAfterGroup = false);

    /* Location open/close Methods */

    /**
     * @brief Opens a location
     *
     * Opens a location, meaning that its potential child locations will be
     * cooked.
     *
     * @param locationPath The location path.
     */
    void openLocation(const std::string& locationPath);

    /**
     * @brief Opens a list of locations
     *
     * Opens a list of locations, meaning that their potential child
     * locations will be cooked.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPaths The location paths.
     */
    void openLocations(const std::vector<std::string>& locationPaths);

    /**
     * @brief Closes a location
     *
     * Closes a location, meaning that its child locations will cease to
     * be cooked, and a deleted event (see locationDeleted()) will be
     * issued for these children.
     *
     * @param locationPath The location path.
     */
    void closeLocation(const std::string& locationPath);

    /**
     * @brief Closes a list of locations
     *
     * Closes a list of location, meaning that their child locations will
     * cease to be cooked, and a deleted event (see locationDeleted()) will
     * be issued for these children.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPaths The location paths.
     */
    void closeLocations(const std::vector<std::string>& locationPaths);

    /* Source activation methods */

    /**
     * @brief Activate the source tree rooted at the given location.
     *
     * While a source location is active, that location and all of its
     * descendants and ancestors are included in the 'sources' scene view.
     *
     * @param locationPath The source tree root location path.
     */
    void activateSourceLocation(const std::string& locationPath);

    /**
     * @brief Activate the source trees rooted at the given locations.
     *
     * While a source location is active, that location and all of its
     * descendants and ancestors are included in the 'sources' scene view.
     *
     * @param locationPaths The source tree root location paths.
     */
    void activateSourceLocations(
        const std::vector<std::string>& locationPaths);

    /**
     * @brief Deactivate the source tree rooted at the given location.
     *
     * @param locationPath The source tree root location path.
     */
    void deactivateSourceLocation(const std::string& locationPath);

    /**
     * @brief Deactivate the source trees rooted at the given locations.
     *
     * @param locationPaths The source tree root location paths.
     */
    void deactivateSourceLocations(
        const std::vector<std::string>& locationPaths);

    /* Location selection functions */

    /**
     * @brief Selects a location in Katana.
     *
     * @param locationPath The location path to be selected.
     */
    void selectLocation(const std::string& locationPath);

    /**
     * @brief Selects a list of location in Katana.
     *
     * @param locationPaths The location paths to be selected.
     */
    void selectLocations(const std::vector<std::string>& locationPaths);

    /**
     * @brief Get the list of selected locations.
     *
     * @param locationPaths Will be filled with the selected location paths.
     */
    void getSelectedLocations(std::vector<std::string>& locationPaths);

    /* ViewerDelegateComponent Management Methods */

    /**
     * @brief Adds a ViewerDelegateComponent.
     *
     * Instantiates a ViewerDelegateComponent and adds it to the end of the
     * list of components on this ViewerDelegate.
     *
     * @param pluginName The ViewerDelegateComponent plugin name.
     * @param name The name for the new ViewerDelegateComponent.
     *
     * @return The newly created ViewerDelegateComponent.
     */
    ViewerDelegateComponentWrapperPtr addComponent(
        const std::string& pluginName, const std::string& name);

    /**
     * @brief Inserts a ViewerDelegateComponent.
     *
     * Instantiates a ViewerDelegateComponent and inserts it  into the specified
     * index of the list of components on this ViewerDelegate.
     *
     * @param pluginName The ViewerDelegateComponent plugin name.
     * @param name The name for the new ViewerDelegateComponent.
     * @param index The position for the new ViewerDelegateComponent to be
     *               inserted into.
     *
     * @return The newly created ViewerDelegateComponent.
     */
    ViewerDelegateComponentWrapperPtr insertComponent(
        const std::string& pluginName, const std::string& name,
        unsigned int index);

    /**
     * @brief Gets a ViewerDelegateComponent by name.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param name The name of the ViewerDelegateComponent.
     *
     * @return The ViewerDelegateComponent in this ViewerDelegate with the
     *          given name or 0 if no such component exists.
     */
    ViewerDelegateComponentWrapperPtr getComponent(
        const std::string& name) const;

    /**
     * @brief Gets a ViewerDelegateComponent by index.
     *
     * This function is exposed in the ViewerDelegate Python class as
     * ViewerDelegate.getComponentByIndex().
     *
     * @param index The position in the list of ViewerDelegateComponents on
     *               this ViewerDelegate.
     *
     * @return The ViewerDelegateComponent in this ViewerDelegate with the
     *          given index or 0 if no such component exists.
     */
    ViewerDelegateComponentWrapperPtr getComponent(unsigned int index) const;

    /**
     * @brief Removes a ViewerDelegateComponent by name.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param name The name of the ViewerDelegateComponent to be removed.
     */
    void removeComponent(const std::string& name);

    /**
     * @brief Removes a ViewerDelegateComponent by name.
     *
     * This function is exposed in the ViewerDelegate Python class as
     * ViewerDelegate.removeLayerByIndex().
     *
     * @param index The position in the list of ViewerDelegateComponents on
     *               this ViewerDelegate
     */
    void removeComponent(unsigned int index);

    /**
     * @brief Gets the number of ViewerDelegateComponents
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @return The number of ViewerDelegateComponents added to this
     *          ViewerDelegate.
     */
    unsigned int getNumberOfComponents() const;

    /**
     * @brief Gets the name of the ViewerDelegateComponent on a given index.
     *
     * @param index The position in the list of ViewerDelegateComponents on
     *               this ViewerDelegate.
     *
     * @return The name of the ViewerDelegateComponent on the given index or
     *          empty string if it doesn't exist.
     */
    std::string getComponentName(unsigned int index);

    /**
     * @brief Gets the index of the ViewerDelegateComponent with a given name.
     *
     * @param name The name of the ViewerDelegateComponent.
     *
     * @return The position in the list of ViewerDelegateComponents on
     *          this ViewerDelegate or -1 if it doesn't exist.
     */
    int getComponentIndex(const std::string& name);

    /* Viewport access methods */

    /**
     * @brief Gets a Viewport by name.
     *
     * Gets a Viewport associated with this ViewerDelegate by its name.
     * The Vieweport must have been created via Python using the
     * ViewerDelegate.addViewport() call.
     *
     * @param name The name of the Viewport.
     *
     * @return A pointer to the Viewport, or 0 if it doesn't exist.
     */
    ViewportWrapperPtr getViewport(const std::string& name);

    /**
     * @brief Gets a Viewport by index.
     *
     * Gets a Viewport associated with this ViewerDelegate by its index.
     * The Vieweport must have been created via Python using the
     * ViewerDelegate.addViewport() call.
     *
     * This function is exposed in the ViewerDelegate Python class as
     * ViewerDelegate.getViewportByIndex().
     *
     * @param index The index of the Viewport.
     *
     * @return A pointer to the Viewport, or 0 if the index does not
     *     refer to any Viewport.
     */
    ViewportWrapperPtr getViewport(unsigned int index);

    /**
     * @brief Gets the number of Viewports
     *
     * Gets the number of Viewports associated with this ViewerDelegate.
     * The Vieweports must have been created via Python using the
     * ViewerDelegate.addViewport() call.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @return The number of Viewports.
     */
    unsigned int getNumberOfViewports();

    /**
     * @brief Gets the Viewport name for a given index.
     *
     * Gets the name of the Viewport associated with this ViewerDelegate
     * with the given index. The Vieweports must have been created via
     * Python using the ViewerDelegate.addViewport() call.
     *
     * @return The name of the Viewport or empty string "" if the index
     *     does not refer to any Viewport.
     */
    std::string getViewportName(unsigned int index);


    /* Other Methods */

    /**
     * @brief Gets the Manipulators compatible with a list of locations.
     *
     * Returns the information about the Manipulator plugins that are
     * compatible with a given list of locations. This is done by calling
     * Manipulator::match() for the cooked attributes of each of these
     * locations. The Manipulators have to match ALL of the given
     * locations. The information is returned as a GroupAttribute that
     * contains other GroupAttributes with each manipulator's tags
     * (see Manipulator::getTags()) indexed by the Manipulator plugin name:
     *
     *    GroupAttribute
     *    {
     *      "manipName1": GroupAttribute
     *                    {
     *                      "tagName1": "tagValue1",
     *                      "tagName2": "tagValue2",
     *                      ...
     *                    },
     *      "manipName2": GroupAttribute
     *                    {
     *                      ...
     *                    },
     *      ...
     *    }
     *
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPaths The location paths.
     *
     * @return A GroupAttribute with the tags of the compatible manipulators.
     */
    FnAttribute::GroupAttribute getCompatibleManipulatorsInfo(
         const std::vector<std::string>& locationPaths);

    /**
     * @brief Calls a Python callback.
     *
     * Calls a Python callback that has been registered in Python via the
     * method ViewerDelegate.registerCallback() (see also the Python function
     * ViewerDelegate.registerCallback()). This allows the ViewerDelegate to
     * communicate with the Viewer tab and Katana via Python. Both the
     * arguments and return value of the callback are in the form of an
     * Attribute.
     *
     * @param name The name with which the callaback was registered.
     * @param message The message to pass to the Python callback.
     *
     * @return The Attribute returned by the callback, or an invalid
     *     Attribute if there is no such callback registered.
     */
    FnAttribute::Attribute callPythonCallback(const std::string& name,
                                              FnAttribute::Attribute message);

    /**
     * @brief Gets the live attributes for a location.
     *
     * Gets the cached, manipulated attributes for the location. Returns an
     * invalid group attribute if the location is not being manipulated. Only
     * attributes that are being manipulated will be present in the group.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPath The location path.
     * @param sceneView Scene view to query.
     *
     * @return The live attributes of the location.
     */
    FnAttribute::GroupAttribute getLiveAttributes(
        const std::string& locationPath,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Gets the cooked attributes for a location.
     *
     * Gets the cooked attributes from the runtime. Unlike getAttributes(),
     * this function does not apply the manipulated attributes on top.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param locationPath The location path.
     * @param sceneView Scene view to query.
     *
     * @return The original attributes of the location.
     */
    FnAttribute::GroupAttribute getCookedAttributes(
        const std::string& locationPath,
        FnViewerDelegateSceneView sceneView = kFnViewerDelegatePrimary);

    /**
     * @brief Sets a manipulated attribute back into Katana.
     *
     * When the user interacts with a an editable attribute some values will
     * have to be sent back to Katana. Setting these values will typically end
     * up setting some node parameters in Katana. Parameters will not be set
     * explicitly by the Manipulators because the same Manipulator might be
     * able to serve different nodes with different parameters.
     *
     * Katana will know what parameters to change via the protocol defined by
     * discovering the node via the "attributeEditor.exclusiveTo" attribute
     * convention and then via the setOverride() function on that node. This
     * function will return false if no parameter capable of editing the given
     * attribute is found.
     *
     * For example in order to change the X position of an object setValue()
     * would be called for the location of that object on its attribute
     * 'xform.interactive.translate.x'.
     *
     * Once a value is set the cooked result might take some time to arrive
     * back. During that time, attributes provided by Viewer Location Events and
     * returned by ViewerDelegate::getAttributes() contain the manipulated
     * value, rather than the cooked one, for those attributes.
     *
     * While a user is scrubbing a UI element that manipulates the attribute,
     * the values might not end up being committed immediately into Geolib3 to
     * be cooked, as this might not perform at an interactive speed (depending
     * on the project complexity). For this the concept of 'final' value is
     * used. While the user is scrubbing a Manipulator handle the value will
     * not be sent back to Katana as the final one. That will happen only once
     * the user releases it, which, at that point, Katana will, guaranteedly,
     * be informed that the new scene can be cooked.
     *
     * @param locationPath The location of the object being manipulated.
     * @param attrName The attribute name for the value being manipulated.
     * @param valueAttr The value being set.
     * @param isFinal True if this is a final value.
     * @return True if a parameter capable of editing the given attribute was
     *          found and was successfully set, false otherwise.
     */
    bool setManipulatedAttribute(const std::string& locationPath,
                                 const std::string& attrName,
                                 FnAttribute::Attribute valueAttr,
                                 bool isFinal);

    /**
     * @brief Marks the beginning of a batch of manipulations.
     *
     * Multiple manipulations can be batched into a group, allowing them to be
     * processed at the same time. Once called any subsequent calls to
     * setManipulatedAttribute() will be deferred until
     * closeManipulationGroup()
     * is called.
     *
     * @param locationPath The scene graph location that this group is for.
     *                      Multiple groups can be created for different
     *                      locations.
     */
    void openManipulationGroup(const std::string& locationPath);

    /**
     * @brief Marks the end of a batch of manipulations.
     *
     * @param locationPath The scene graph location that this group is for.
     *                      Multiple groups can be created for different
     *                      locations.
     */
    void closeManipulationGroup(const std::string& locationPath);

    /**
     * @brief Turns the display of bounding boxes on or off.
     *
     * @param enabled Flag that controls whether to turn the display of
     *                 bounding boxes on or off.
     */
    void setBoundingBoxesEnabled(bool enabled);

    /**
     * @brief Turns the display of proxy geometries on or off.
     *
     * @param enabled Flag that controls whether to turn the display of
     *                 proxy geometries on or off.
     */
    void setProxyGeometryEnabled(bool enabled);

    /**
     * @brief Returns all descendant locations from the specified location.
     *
     * The descendant locations are those that the viewer delegate has opened.
     * Full scene-graph expansion is not performed by this function.
     *
     * @param location Parent of all returned locations.
     *
     * @return StringAttribute containing zero or more descendant locations.
     */
    FnAttribute::StringAttribute getDescendantLocations(
        const std::string& location);

    /**
     * @brief Computes the merged extent of the given locations in world space.
     *
     * The bounds of every location will be merged. If the location does not
     * have defined bounds, its extent will be computed based on its geometry,
     * as well as the bounds or extent of its descendants.
     *
     * @param locations Locations whose bounds (or extent) need to be merged.
     * @param excludeLocation Location that must be excluded from the merged
     *     bounds (typically the viewport's look-through location).
     *
     * @return The merged bounds of all the locations in world space, or an
     *     invalid attribute if it could not be computed.
     */
    FnAttribute::DoubleAttribute computeMergedExtent(
        const std::vector<std::string>& locations,
        const std::string& excludeLocation);

///@cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    static FnPluginHost* m_host;

    FnViewerDelegateHostSuite_v4* m_hostSuite;
    FnViewerDelegateHostHandle m_hostHandle;

///@endcond
};


/** @brief The ViewerDelegate class to be extended by plugins. */
class ViewerDelegate : public ViewerDelegatePluginBase
{
public:

    /** @brief Constructor. */
    ViewerDelegate();

    /** @brief Destructor. */
    virtual ~ViewerDelegate();

    /** @brief Initializes the ViewportDelegate's resources. */
    virtual void setup() = 0;

    /** @brief Cleans up the ViewportDelegate's resources. */
    virtual void cleanup() = 0;

    /**
     * @brief Gets the bounds of the given location, if available.
     *
     * This function is called only if bounds are not given by the "bound"
     * attribute on the scene graph location. It should not traverse scene
     * graph in order to compute bounds.
     *
     * @param locationPath The location path.
     *
     * @return The bounds of the location in local space, or an invalid
     *     attribute if not available.
     */
    virtual FnAttribute::DoubleAttribute getBounds(
        const std::string& locationPath);

    /**
     * @brief Computes the extent of any local geometry representation of this
     *     location.
     *
     * The computed extent returned by this function will be merged with a
     * default extent computed from standard "geometry.point" attribute.
     *
     * @param locationPath The location path.
     *
     * @return The extent of the location in local space, or an invalid
     *     attribute if no extent was computed.
     */
    virtual FnAttribute::DoubleAttribute computeExtent(
        const std::string& locationPath);

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * An example: A ViewerDelegate maintains some renderer specific data
     * structure that contains the expanded scene data in a format that the
     * render can use. A ViewerDelegateComponent could be added to the Viewer
     * and through this function an access this data so that new location types
     * can be interpreted, added to the data structure and rendered.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /* Scene Graph Events */

    /**
     * @brief Notification of scene graph location state changes.
     *
     * @param event Struct containing location event information (see
     *      "FnViewerLocationEvent.h" for details).
     * @param locationHandled True if a ViewerDelegateComponent has, on
     *      processing this event, stated that it is handling this location. In
     *      this case, the Viewer Delegate should not draw its own
     *      representation.
     */
    virtual void locationEvent(const ViewerLocationEvent& event,
                               bool locationHandled) = 0;

    /**
     * @brief Notification of 'sources' scene graph location state changes.
     *
     * @param event Struct containing location event information (see
     *     "FnViewerLocationEvent.h" for details).
     */
    virtual void sourceLocationEvent(const ViewerLocationEvent& event) = 0;

    /**
     * @brief Called when the location selection changes in Katana.
     *
     * @param locationPaths The selected location paths.
     */
    virtual void locationsSelected(
        const std::vector<std::string>& locationPaths) = 0;

    /**
     * @brief Queries the processing of the Viewer Delegate plug-in.
     *
     * @return true if the plug-in is processing anything that may result in a
     *     change to the Viewer.
     */
    virtual bool isProcessing() const { return false; }

    /* Options setting and getting. */

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the ViewerDelegate.
     *
     * This function is exposed in the ViewerDelegate Python class.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
                           FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

    /* General Methods */

    /**
     * @brief Freezes the ViewerDelegate when the Viewer is hidden.
     *
     * Allows the ViewerDelegate to freeze its activities when the Viewer
     * is not visible. This allows the ViewerDelegate to stop any kind of
     * unecessary processing that might happen durging that time.
     *
     * This function is exposed in the ViewerDelegate Python class and in the
     * ViewerDelegateWrapper wrapper class.
     */
    virtual void freeze() = 0;

    /**
     * @brief Thaws the ViewerDelegate when the Viewer is shown.
     *
     * Allows the ViewerDelegate to restart its activities when the Viewer
     * becomes visible. This restarts the activities paused by freeze().
     *
     * This function is exposed in the ViewerDelegate Python class.
     */
    virtual void thaw() = 0;

    /**
     * @brief Flush plugin Caches.
     *
     * Allows to discard any cache for this plugin when a Flush Caches
     * event occurs.
     */
    static void flush() {}

///@cond FN_INTERNAL_DEV
public:
    static FnViewerDelegatePluginSuite_v3 createSuite(
      FnViewerDelegatePluginHandle (*create)(
          FnViewerDelegateHostHandle hostHandle));

    static FnViewerDelegatePluginHandle newViewerDelegateHandle(
      ViewerDelegate* viewerDelegate);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnViewerDelegateHostHandle hostHandle);

///@endcond
};

/** @brief The ViewerDelegate class accessed by other plugins. */
class ViewerDelegateWrapper : public ViewerDelegatePluginBase
{
public:
    ViewerDelegateWrapper(
        FnPluginHost* host,
        FnViewerDelegateHostHandle hostHandle,
        FnViewerDelegatePluginHandle pluginHandle,
        FnViewerDelegatePluginSuite_v3* pluginSuite);

    ~ViewerDelegateWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of ViewerDelegate
     * that is a registerd plugin. The type of that child class needs to be
     * specified in the template, so that it can be properly cast. If the
     * specified type  doesn't match the plugin type then NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /** @brief Sets a ViewerDelegate option. */
    void setOption(OptionIdGenerator::value_type optionId, FnAttribute::Attribute attr);

    /** @brief Gets a ViewerDelegate option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);

    /** @brief Sets a ViewerDelegate option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /** @brief Gets a ViewerDelegate option. */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
private:
    ViewerDelegate* getPluginPointer();

    /** The ViewerDelegate plug-in. */
    FnViewerDelegatePluginSuite_v3* m_pluginSuite;
    FnViewerDelegatePluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ViewerDelegateWrapper> ViewerDelegateWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry



///@cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnViewerDelegatePluginStruct
{
public:
    FnViewerDelegatePluginStruct(
        Foundry::Katana::ViewerAPI::ViewerDelegate* viewerDelegate)
    : m_viewerDelegate(viewerDelegate)
    { }

    ~FnViewerDelegatePluginStruct()
    { };

    Foundry::Katana::ViewerAPI::ViewerDelegate* getViewerDelegate()
    {
        return m_viewerDelegate.get();
    }

private:
    std::shared_ptr<
        Foundry::Katana::ViewerAPI::ViewerDelegate> m_viewerDelegate;
};

// Plugin Registering Macro.
#define DEFINE_VIEWER_DELEGATE_PLUGIN(VIEWER_DELEGATE_CLASS)                  \
                                                                              \
    FnPlugin VIEWER_DELEGATE_CLASS##_plugin;                                  \
                                                                              \
    FnViewerDelegatePluginHandle VIEWER_DELEGATE_CLASS##_create(              \
        FnViewerDelegateHostHandle hostHandle)                                \
    {                                                                         \
        Foundry::Katana::ViewerAPI::ViewerDelegate* viewerDelegate =          \
            VIEWER_DELEGATE_CLASS::create();                                  \
                                                                              \
        viewerDelegate->setHostHandle(hostHandle);                            \
        return Foundry::Katana::ViewerAPI::ViewerDelegate::newViewerDelegateHandle( \
            viewerDelegate);                                                  \
    }                                                                         \
                                                                              \
    FnViewerDelegatePluginSuite_v3 VIEWER_DELEGATE_CLASS##_suite =            \
            Foundry::Katana::ViewerAPI::ViewerDelegate::createSuite(          \
                    VIEWER_DELEGATE_CLASS##_create);                          \
                                                                              \
    const void* VIEWER_DELEGATE_CLASS##_getSuite()                            \
    {                                                                         \
        return &VIEWER_DELEGATE_CLASS##_suite;                                \
    }

///@endcond

#endif
