// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_PICKINGTYPES_H
#define FNVIEWER_PICKINGTYPES_H



#include <FnViewer/plugin/FnMathTypes.h>
#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>

#include <stdint.h>
#include <sstream>
#include <map>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnPickingTypes Picking Types
 * @{
 */

/**
 * @brief An integer that identifies a pickable object.
 *
 * See \c pickIdToColor() to see how this can be converted into an ID color
 * that can be used in \c Viewport::pickerDraw() and
 * \c ViewportLayer::pickerDraw().
 */
typedef uint64_t FnPickId;


/**
 * Maps picking IDs with an \c Attribute that contains information about the
 * corresponding pickable object. This can be converted from/into
 * a \c GroupAttribute that contains a string representation of the \c FnPickId
 * numbers (map keys) as the child names and the \c Attribute objects (map
 * values) as its children.
 */
class PickedAttrsMap : public std::map<FnPickId, FnAttribute::Attribute>
{
public:

    /**
     * Utility function that returns a \c GroupAttribute that contains a string
     * representation of the \c FnPickId numbers (map keys) as the child names
     * and the \c Attribute objects (map values) as its children.
     *
     * @return A \c GroupAttribute that contains a string representation of
     *     this map.
     */
    FnAttribute::GroupAttribute toGroupAttribute()
    {
        if (size() == 0)
        {
            return FnAttribute::GroupAttribute();
        }

        FnAttribute::GroupBuilder gb;
        for (const_iterator it = begin(); it != end(); ++it)
        {
            // Convert the \c FnPickId into a string representation
            std::ostringstream idStr;
            idStr << it->first;
            gb.set(idStr.str(), it->second);
        }

        return gb.build();
    }

    /**
     * Utility function that fills the map with entries from a
     * \c GroupAttribute that contains a string representation of the
     * \c FnPickId numbers (map keys) as the child names and the \c Attribute
     * objects (map values) as its children.
     *
     * @param groupAttr The \c GroupAttribute to parse to fill this map.
     */
    void fromGroupAttribute(FnAttribute::GroupAttribute groupAttr)
    {
        clear();

        if (groupAttr.isValid())
        {
            for (unsigned int i = 0; i < groupAttr.getNumberOfChildren(); ++i)
            {
                // Get the \c FnPickId by parsing the entry name
                FnPickId pickId;
                std::istringstream istr(groupAttr.getChildName(i));
                istr >> pickId;
                (*this)[pickId] = groupAttr.getChildByIndex(i);
            }
        }
    }
};

/**
 * @brief Updates the given color with values that represent the given pick ID.
 *
 * Given a pick ID, this returns a color that should be used when rendering the
 * IDs for picking in the functions \c Viewport::pickerDraw() and
 * \c ViewportLayer::pickerDraw(). Since this is a 64bit ID, this color will
 * refer only to its first 32 lower bits.
 */
inline void pickIdToColor(FnPickId pickId, Vec4i& color)
{
    color.x = (pickId >> 24 & 0xff);
    color.y = (pickId >> 16 & 0xff);
    color.z = (pickId >> 8  & 0xff);
    color.w = (pickId       & 0xff);
}

/**
 * @brief Updates the given color with values that represent the given pick ID.
 *
 * Given a pick ID, this returns a color that should be used when rendering the
 * IDs for picking in the functions \c Viewport::pickerDraw() and
 * \c ViewportLayer::pickerDraw(). Since this is a 64bit ID, this color will
 * refer only to its first 32 lower bits.
 */
inline void pickIdToColor(FnPickId pickId, Vec4f& color)
{
    color.x = (float)(pickId >> 24 & 0xff) / 0xff;
    color.y = (float)(pickId >> 16 & 0xff) / 0xff;
    color.z = (float)(pickId >> 8  & 0xff) / 0xff;
    color.w = (float)(pickId       & 0xff) / 0xff;
}

/** @} */

} // ViewerAPI
} // Katana
} // Foundry

#endif /* FNVIEWER_PICKINGTYPES_H */
