// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNVIEWER_MATHTYPES_H
#define FNVIEWER_MATHTYPES_H

#include <cassert>
#include <cmath>
#include <cstring>

namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnMathTypes Math Types
 * @{
 */

    /**
     * Template for a data structure representing a 4x4 matrix of values of the
     * given value data type T.
     *
     * \tparam T The data type of the values to store in the matrix.
     */
    template <class T> struct Matrix44
    {
    public:
        // @cond FN_INTERNAL_DEV
        // Unfortunately, Breathe/Sphinx v1.4.1 is unable to parse T data[16].

        /**
         * The 16 values of the 4x4 matrix of the given value data type T.
         */
        T           data[16];

        // @endcond

        Matrix44();
        Matrix44(const T matrix[16]);
        Matrix44(T m00, T m01, T m02, T m03,
                 T m10, T m11, T m12, T m13,
                 T m20, T m21, T m22, T m23,
                 T m30, T m31, T m32, T m33);

        const T* getData() const;
    };

    template <class T> struct Vec2
    {
        T x;
        T y;

        Vec2();
        Vec2(T x, T y);

        Vec2 operator-() const;
        Vec2 operator+(const Vec2 &rhs) const;
        Vec2 operator-(const Vec2 &rhs) const;
        T dot(const Vec2 &rhs) const;
        T length() const;
        Vec2 normalized() const;
        bool isParallelTo(const Vec2 &rhs) const;
        Vec2<T> operator*(const T scalar) const;
        Vec2<T> operator/(const T scalar) const;
    };

    template <class T> struct Vec3
    {
        T x;
        T y;
        T z;

        Vec3();
        Vec3(T x, T y, T z);

        Vec3 operator-() const;
        Vec3 operator+(const Vec3 &rhs) const;
        Vec3 operator-(const Vec3 &rhs) const;
        T dot(const Vec3 &rhs) const;
        T length() const;
        Vec3 normalized() const;
        Vec3 cross(const Vec3 &rhs) const;
        bool isParallelTo(const Vec3 &rhs) const;
        Vec3<T> operator*(const T scalar) const;
        Vec3<T> operator/(const T scalar) const;
    };

    template <class T> struct Vec4
    {
        T x;
        T y;
        T z;
        T w;

        Vec4();
        Vec4(T x, T y, T z, T w);
    };

    typedef Matrix44<float> Matrix44f;
    typedef Matrix44<double> Matrix44d;

    typedef Vec2<int> Vec2i;
    typedef Vec2<float> Vec2f;
    typedef Vec2<double> Vec2d;

    typedef Vec3<int> Vec3i;
    typedef Vec3<float> Vec3f;
    typedef Vec3<double> Vec3d;

    typedef Vec4<int> Vec4i;
    typedef Vec4<float> Vec4f;
    typedef Vec4<double> Vec4d;

/** @} */

    //-----------------
    // Implementation
    //-----------------


    template <class T>
    inline Matrix44<T>::Matrix44()
    {
        // Set to identity
        memset (data, 0, sizeof(data));
        data[0] = 1;
        data[5] = 1;
        data[10] = 1;
        data[15] = 1;
    }

    template <class T>
    inline Matrix44<T>::Matrix44(const T matrix[16])
    {
        memcpy (data, matrix, sizeof (data));
    }

    template <class T>
    inline
        Matrix44<T>::Matrix44(T m00, T m01, T m02, T m03,
            T m10, T m11, T m12, T m13,
            T m20, T m21, T m22, T m23,
            T m30, T m31, T m32, T m33)
    {
        data[0] = m00;
        data[1] = m01;
        data[2] = m02;
        data[3] = m03;
        data[4] = m10;
        data[5] = m11;
        data[6] = m12;
        data[7] = m13;
        data[8] = m20;
        data[9] = m21;
        data[10] = m22;
        data[11] = m23;
        data[12] = m30;
        data[13] = m31;
        data[14] = m32;
        data[15] = m33;
    }

    template <class T>
    inline const T* Matrix44<T>::getData() const
    {
        return &data[0];
    }


    template <class T>
    inline Vec2<T>::Vec2()
        : x(0), y(0) {}

    template <class T>
    inline Vec2<T>::Vec2(T _x, T _y)
        : x(_x), y(_y) {}

    template <class T>
    inline Vec2<T> Vec2<T>::operator-() const
    {
        Vec2 result;
        result.x = -x;
        result.y = -y;
        return result;
    }

    template <class T>
    inline Vec2<T> Vec2<T>::operator+(const Vec2 &rhs) const
    {
        Vec2 result(*this);
        result.x += rhs.x;
        result.y += rhs.y;
        return result;
    }

    template <class T>
    inline Vec2<T> Vec2<T>::operator-(const Vec2 &rhs) const
    {
        Vec2 result(*this);
        result.x -= rhs.x;
        result.y -= rhs.y;
        return result;
    }

    template <class T>
    inline T Vec2<T>::dot(const Vec2 &rhs) const
    {
        return x * rhs.x + y * rhs.y;
    }

    template <class T>
    inline T Vec2<T>::length() const
    {
        const T dot_self = this->dot(*this);
        return sqrt(dot_self);
    }

    template <class T>
    inline Vec2<T> Vec2<T>::normalized() const
    {
        const T recip_length = 1 / this->length();
        Vec2 result(*this);
        result.x *= recip_length;
        result.y *= recip_length;
        return result;
    }

    template <class T>
    inline bool Vec2<T>::isParallelTo(const Vec2 &rhs) const
    {
        const T epsilon = 1.0e-6;
        const Vec2 thisNormalized = this->normalized();
        const Vec2 rhsNormalized = rhs.normalized();
        const T dp = thisNormalized.dot(rhsNormalized);
        // parallel normalized vectors have a dot product of +1 or -1
        bool parallel = fabs((fabs(dp) - 1)) < epsilon;
        return parallel;
    }

    template <class T>
    inline Vec2<T> Vec2<T>::operator*(const T scalar) const
    {
        Vec2 result(*this);
        result.x *= scalar;
        result.y *= scalar;
        return result;
    }

    template <class T>
    inline Vec2<T> Vec2<T>::operator/(const T scalar) const
    {
        Vec2 result(*this);
        result.x /= scalar;
        result.y /= scalar;
        return result;
    }


    template <class T>
    inline Vec3<T>::Vec3()
        : x(0), y(0), z(0) {}

    template <class T>
    inline Vec3<T>::Vec3(T _x, T _y, T _z)
        : x(_x), y(_y), z(_z) {}

    template <class T>
    inline Vec3<T> Vec3<T>::operator-() const
    {
        Vec3 result;
        result.x = -x;
        result.y = -y;
        result.z = -z;
        return result;
    }

    template <class T>
    inline Vec3<T> Vec3<T>::operator+(const Vec3 &rhs) const
    {
        Vec3 result(*this);
        result.x += rhs.x;
        result.y += rhs.y;
        result.z += rhs.z;
        return result;
    }

    template <class T>
    inline Vec3<T> Vec3<T>::operator-(const Vec3 &rhs) const
    {
        Vec3 result(*this);
        result.x -= rhs.x;
        result.y -= rhs.y;
        result.z -= rhs.z;
        return result;
    }

    template <class T>
    inline T Vec3<T>::dot(const Vec3 &rhs) const
    {
        return x * rhs.x + y * rhs.y + z * rhs.z;
    }

    template <class T>
    inline T Vec3<T>::length() const
    {
        const T dot_self = this->dot(*this);
        return sqrt(dot_self);
    }

    template <class T>
    inline Vec3<T> Vec3<T>::normalized() const
    {
        const T recip_length = 1 / this->length();
        Vec3 result(*this);
        result.x *= recip_length;
        result.y *= recip_length;
        result.z *= recip_length;
        return result;
    }

    template <class T>
    inline Vec3<T> Vec3<T>::cross(const Vec3 &rhs) const
    {
        Vec3 result;
        result.x =   y * rhs.z - z * rhs.y;
        result.y = -(x * rhs.z - z * rhs.x);
        result.z =   x * rhs.y - y * rhs.x;
        return result;
    }

    template <class T>
    inline bool Vec3<T>::isParallelTo(const Vec3 &rhs) const
    {
        const T epsilon = 1.0e-6;
        const Vec3 thisNormalized = this->normalized();
        const Vec3 rhsNormalized = rhs.normalized();
        const T dp = thisNormalized.dot(rhsNormalized);
        // parallel normalized vectors have a dot product of +1 or -1
        bool parallel = fabs((fabs(dp) - 1)) < epsilon;
        return parallel;
    }

    template <class T>
    inline Vec3<T> Vec3<T>::operator*(const T scalar) const
    {
        Vec3 result(*this);
        result.x *= scalar;
        result.y *= scalar;
        result.z *= scalar;
        return result;
    }

    template <class T>
    inline Vec3<T> Vec3<T>::operator/(const T scalar) const
    {
        Vec3 result(*this);
        result.x /= scalar;
        result.y /= scalar;
        result.z /= scalar;
        return result;
    }

    template <class T>
    inline Vec4<T>::Vec4()
        : x(0), y(0), z(0), w(0) {}

    template <class T>
    inline Vec4<T>::Vec4(T _x, T _y, T _z, T _w)
        : x(_x), y(_y), z(_z), w(_w) {}

} // ViewerAPI
} // Katana
} // Foundry


#endif
