// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNMANIPULATORHANDLE_H_
#define FNMANIPULATORHANDLE_H_

#include <FnViewer/suite/FnManipulatorHandleSuite.h>
#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/plugin/FnManipulator.h>
#include <FnViewer/plugin/FnMathTypes.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <stdint.h>
#include <string>
#include <memory>

#ifndef FNKAT_UNUSED
    #define FNKAT_UNUSED(x) (void)x
#endif



namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

/**
 * \defgroup FnManipulatorHandle ManipulatorHandle Plugin
 * @{
 */

/**
 * @brief Interface for a Viewer Manipulator Handle.
 *
 * A ManipulatorHandle is a reusable component that can be used by different
 * Manipulators. It is a scene interaction pattern that can appear in different
 * Manipulators to perform different tasks. An example of a ManipulatorHandle:
 * an Arrow handle that draws an arrow that can be clicked and dragged, this
 * can be used, for example, as the axis handle for a translate manipulator and
 * as a light's image projection positioning handle.
 *
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static ManipulatorHandle* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewerDelegate is the class that plugins should extend and implement.
 *
 * ViewerDelegateWrapper is the class that allows other plugin types to access
 * the ViewerDelegate plugin.
 *
 * ViewerDelegatePluginBase is the base class that provides the common methods
 * between ViewerDelegate and ViewerDelegateWrapper.
 *
 */
class ManipulatorHandlePluginBase
{
public:

    /** @brief Constructor. */
    ManipulatorHandlePluginBase();

    /** @brief Destructor. */
    virtual ~ManipulatorHandlePluginBase();

public: /* Methods to be called by the plugin. */

    /**
     * @brief Gets the Manipulator.
     *
     * Gets the Manipulator that created and manages this instance of
     * ManipulatorHandle. This can be used to access the Manipulator's
     * functionality and also to access the Viewport and ViewerDelegate
     * associated with it, via:
     *
     *     getManipulator()->getViewport()->getViewerDelegate()
     *
     * @return The Manipulator that created and manages this handle.
     */
    ManipulatorWrapperPtr getManipulator();

    /**
     * @brief Gets the world transform of the handle.
     *
     * @return A 4x4 matrix that is calculated by multiplying the local
     *     transfrom (see setLocalXform() / getLocalXform()) with the
     *     Manipulator's transform (accessible by Manipulator::getXform().
     */
    Matrix44d getXform();

    /**
     * @brief Sets the local transform of the handle.
     *
     * @param xform A 4x4 matrix that defines the handle's local transform in
     *     relation to the Manipulator's transform.
     */
    void setLocalXform(const Matrix44d& xform);

    /**
     * @brief Gets the local transform of the handle.
     *
     * @return A 4x4 matrix previously set by setLocalXform().
     */
    Matrix44d getLocalXform();

    /** @brief Tells if the Handle is currently being manipulated. */
    bool isActive();

    /** @brief Tells if the Handle is currently being hovered by the mouse. */
    bool isHovered();

///@cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    FnManipulatorHandleHostSuite_v1* m_hostSuite;
    FnManipulatorHandleHostHandle m_hostHandle;

    static FnPluginHost* m_host;

private:
    ManipulatorWrapperPtr m_manipulatorWrapper;

///@endcond
};


/** @brief The ManipulatorHandle class to be extended by plugins. */
class ManipulatorHandle : public ManipulatorHandlePluginBase
{
public:
    ManipulatorHandle();
    virtual ~ManipulatorHandle();

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Cancels the manipulation.
     *
     * Typically called before the handle is destroyed to discard any pending
     * transform that hasn't been made persistent while dragging the handle.
     */
    virtual void cancelManipulation() = 0;

    /**
     * @brief Draws the Manipulator Handle.
     *
     * Called when the scene needs to be drawn in the correct GL context. In a
     * non-GL renderer the generated image should be drawn in the GL
     * framebuffer in order to be displayed.
     */
    virtual void draw() = 0;

    /**
     * @brief Draws the ManipulatorHandle for picking.
     *
     * Called when the scene needs to be drawn in order to perform objects
     * selection. This will run in the correct GL context. In a non-GL renderer
     * the generated image should be drawn in the GL framebuffer in order to be
     * displayed.
     */
    virtual void pickerDraw(int64_t pickerId) = 0;

    /**
     * @brief Processes UI events.
     *
     * Called whenever a user interaction event occurs.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, in that case it should not
     *          be passed to the following handles. Otherwise false and the
     *          event should be passed to the following handles.
     */
    virtual bool event(const FnEventWrapper& eventData);

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /* Options setting and getting. */

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the ManipulatorHandle.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
                           FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
public:
    static FnManipulatorHandlePluginSuite_v1 createSuite(
        FnManipulatorHandlePluginHandle (*create)(
            FnManipulatorHandleHostHandle hostHandle));
    static FnManipulatorHandlePluginHandle newManipulatorHandleHandle(
        ManipulatorHandle* manipulatorHandle);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnManipulatorHandleHostHandle m_hostHandle);
    FnManipulatorHandleHostHandle getHostHandle();

/// @endcond
};


/** @brief The ManipulatorHandle class accessed by other plugins. */
class ManipulatorHandleWrapper : public ManipulatorHandlePluginBase
{
public:
    ManipulatorHandleWrapper(
        FnPluginHost* host,
        FnManipulatorHandleHostHandle hostHandle,
        FnManipulatorHandlePluginHandle pluginHandle,
        FnManipulatorHandlePluginSuite_v1* pluginSuite);

    ~ManipulatorHandleWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of
     * ManipulatorHandle that is a registerd  plugin. The type of that child
     * class needs to be specified in the template, so that it can be properly
     * cast. If the specified type  doesn't match the plugin type then NULL is
     * returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    /** @brief Sets if the Handle is currently active. */
    void setActive(bool active);

    /** @brief Sets if the Handle is currently being hovered by the mouse. */
    void setHovered(bool hovered);

    /**
     * @brief Cancels the manipulation.
     *
     * This is called just before the handle is destroyed to revert any pending
     * transform changes that haven't yet been applied while dragging.
     */
    void cancelManipulation();

    /**
     * @brief Draws the Manipulator Handle.
     *
     * This should be called with the correct GL context since the
     * final image will be rendered on the main framebuffer.
     */
    void draw();

    /** Processes the UI events on this ManipulatorHandle.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, false otherwise.
     */
    bool event(FnEventWrapper eventData);

    /**
     * @brief Draws the Manipulator Handle for picking.
     *
     * This should be called with the correct GL context since
     * the final image will be rendered on the main framebuffer.
     *
     * @param pickerId The first ID assigned to this manipulator by the picker.
     */
    void pickerDraw(int64_t pickerId);

    /** @brief Sets a generic option. */
    void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);
    /** @brief Sets a generic option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
private:
    ManipulatorHandle* getPluginPointer();

    /** The ManipulatorHandle plug-in. */
    FnManipulatorHandlePluginSuite_v1* m_pluginSuite;
    FnManipulatorHandlePluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ManipulatorHandleWrapper> ManipulatorHandleWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry



///@cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnManipulatorHandlePluginStruct
{
public:
    FnManipulatorHandlePluginStruct(
        Foundry::Katana::ViewerAPI::ManipulatorHandle* manipulatorHandle)
    : m_manipulatorHandle(manipulatorHandle)
    {}

    ~FnManipulatorHandlePluginStruct()
    {
        if (m_manipulatorHandle)
            m_manipulatorHandle->cancelManipulation();
    };

    Foundry::Katana::ViewerAPI::ManipulatorHandle* getManipulatorHandle()
    {
        return m_manipulatorHandle.get();
    }

private:
    std::shared_ptr<
        Foundry::Katana::ViewerAPI::ManipulatorHandle> m_manipulatorHandle;
};


// Plugin Registering Macro.
#define DEFINE_MANIPULATOR_HANDLE_PLUGIN(MANIPULATOR_HANDLE_CLASS)            \
                                                                              \
FnPlugin MANIPULATOR_HANDLE_CLASS##_plugin;                                   \
                                                                              \
FnManipulatorHandlePluginHandle MANIPULATOR_HANDLE_CLASS##_create(            \
    FnManipulatorHandleHostHandle hostHandle)                                 \
{                                                                             \
    Foundry::Katana::ViewerAPI::ManipulatorHandle* manipulatorHandle =        \
        MANIPULATOR_HANDLE_CLASS::create();                                   \
                                                                              \
    manipulatorHandle->setHostHandle(hostHandle);                             \
    return Foundry::Katana::ViewerAPI::ManipulatorHandle::newManipulatorHandleHandle(manipulatorHandle); \
}                                                                             \
                                                                              \
FnManipulatorHandlePluginSuite_v1 MANIPULATOR_HANDLE_CLASS##_suite =          \
        Foundry::Katana::ViewerAPI::ManipulatorHandle::createSuite(           \
                MANIPULATOR_HANDLE_CLASS##_create);                           \
                                                                              \
const void* MANIPULATOR_HANDLE_CLASS##_getSuite()                             \
{                                                                             \
    return &MANIPULATOR_HANDLE_CLASS##_suite;                                 \
}

///@endcond

#endif /* FNMANIPULATORHANDLE_H_ */
