// Copyright (c) 2016 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNMANIPULATOR_H_
#define FNMANIPULATOR_H_

#include <FnViewer/suite/FnManipulatorSuite.h>
#include <FnViewer/plugin/FnViewport.h>
#include <FnViewer/plugin/FnEventWrapper.h>
#include <FnViewer/plugin/FnMathTypes.h>

#include <FnPluginSystem/FnPluginSystem.h>
#include <FnPluginSystem/FnPlugin.h>
#include <FnAttribute/FnAttribute.h>

#include <vector>
#include <stdint.h>
#include <string>
#include <memory>


namespace Foundry
{
namespace Katana
{
namespace ViewerAPI
{

// Forward declarations
class ViewportWrapper;
class ManipulatorHandleWrapper;
typedef std::shared_ptr<ViewportWrapper> ViewportWrapperPtr;
typedef std::shared_ptr<ManipulatorHandleWrapper> ManipulatorHandleWrapperPtr;

/**
 * \defgroup FnManipulator Manipulator Plugin
 * @{
 */

/**
 * @brief Interface for a Viewer Manipulator.
 *
 * A Manipulator allows a user to interact with the Viewer scene by drawing
 * handles that can be interacted with and that can change values in the scene.
 *
 * A Manipulator can be optionally composed of a series of ManipulatorHandles,
 * which are a plug-in type on its own to allow reusability between Manipulator
 * types. The draw() and event() functions can implement all the drawing and UI
 * event handling without using any ManipulatorHandle. If there are
 * ManipulatorHandles being used then draw() and event() will be propagated by
 * each one of them.
 *
 * When interacting with a Manipulator Katana will be informed that something
 * changed and needs to potentially be stored in a node attribute and recooked.
 * This is done via the setValue(), which follows a protocol used by Katana
 * that maps Scene Graph attributes with nodes and parameters in the current
 * project. This allows setValue() to set attributes rather than parameters
 * directly.
 *
 * Because the cooking times can be non-interactive in certain scenes, the
 * Manipulators use a mechanism that will make the ViewerDelegate to return the
 * manipulated values while the cook is running. This means that the function
 * ViewerDelegate::getAttributes() will return these manipulated values so that
 * the Viewer can present interactive performance. Once the cooked value is
 * returned by Geolib3, the ViewerDelegate will return the cooked values as
 * before and the manipulated value is discarded.
 *
 * This is a virtual base class to be extended in your plug-in. In addition to
 * implementing all the pure virtual functions in this class, you must also
 * implement this static method in your derived class:
 *
 * \code
 * // Returns a new instance of your derived class.
 * static Manipulator* create();
 * \endcode
 *
 * To indicate an error to the caller, your implementation may throw an
 * exception derived from std::exception.
 *
 * ViewerDelegate is the class that plugins should extend and implement.
 *
 * ViewerDelegateWrapper is the class that allows other plugin types to access
 * the ViewerDelegate plugin.
 *
 * ViewerDelegatePluginBase is the base class that provides the common methods
 * between ViewerDelegate and ViewerDelegateWrapper.
 */
class ManipulatorPluginBase
{
public:

    /** @brief Constructor. */
    ManipulatorPluginBase();

    /** @brief Destructor. */
    virtual ~ManipulatorPluginBase();

public: /* Constants */

    // Some standard tag keys
    /// The UI display name of the manipulator.
    static const char kTagName[];
    /// The default keyboard shortcut.
    static const char kTagShortcut[];
    /// The manipulator group (Transform, Light etc).
    static const char kTagGroup[];
    /// The target viewer technology to differentiate between manipulators for
    /// different types of viewport.
    static const char kTagTechnology[];
    /// For setting whether the manipulator should always be available to
    /// choose in the UI, as opposed to only when scene graph locations with
    /// compatible attributes are selected.
    static const char kTagAlwaysAvailable[];
    /// For setting whether other manipulators in the same group can be active
    /// at the same time
    static const char kTagExclusiveInGroup[];
    /// For setting the priority within its group of manipulators. The greater
    /// the value, the higher the position in the list.
    static const char kTagPriorityInGroup[];

    // Some standard group tag values
    static const char kTagGroup_TRANSFORM[];  // Translate, Rotate, etc.
    static const char kTagGroup_LIGHT[];  // Light related manipulators.


public: /* Methods to be called by the plugin. */

    /**
     * @brief Gets the Manipulator plugin name.
     *
     * @return The name given to the Manipulator plugin when registered.
     */
    std::string getPluginName();

    /**
     * @brief Gets the Viewport.
     *
     * Gets the Viewport that activated this instance of Manipulator. This can
     * be used to access the Viewport functionality also to access the
     * ViewerDelegate associated with it, via:
     *
     *     getViewport()->getViewerDelegate()
     *
     * @return The Viewport that created and manages this layer.
     */
    ViewportWrapperPtr getViewport();

    /**
     * @brief Gets the matched locations manipulated by this Manipulator.
     *
     * These locations are only the ones that match this Manipulator. If any
     * location was associated with the manipulator and does not match it,
     * then that will not be returned by this function. If, after cooking,
     * those locations start/stop matching the Manipulator, then this will
     * return different locations. The result can be an empy vector.
     *
     * @param[out] paths The location paths or an empty vector.
     */
    void getLocationPaths(std::vector<std::string>& paths);

    /**
     * @brief Gets the Manipulator's transform.
     *
     * @return A 4x4 matrix in the form of a DoubleAttribute.
     */
    Matrix44d getXform();

    /**
     * @brief Sets the Manipulator's transform.
     *
     * @param xformMatrix A 4x4 matrix in the form of a DoubleAttribute.
     */
    void setXform(const Matrix44d& xformMatrix);

    /**
     * @brief Calculates the the averate of the locations positions.
     *
     * This returns a transform containing solely the world space average
     * translation of the manipulated locations.
     *
     * @return A 4x4 matrix in the form of a DoubleAttribute with
     *     translation values only.
     */
    Matrix44d calculateAveragePositionXform();

    /**
     * @brief Sets a manipulated value back into Katana.
     *
     * When the user interacts with a Manipulator some values will have to be
     * sent back to Katana. Setting these values will typically end up setting
     * some node parameters in Katana. Parameters will not be set explicitly
     * by the Manipulators because the same Manipulator might be able to serve
     * different nodes with different parameters. The protocol to communicate
     * these value back into Katana is by using setValue() on Scene Graph
     * attributes that might exist in the scene. For example in order to change
     * the position of an object setValue() would be called for the location
     * of that object on its attribute 'xform.interactive.translate'.
     *
     * Katana will know what parameters to change via the protocol defined by
     * discovering the node via the "attributeEditor.exclusiveTo" attribute
     * convention and then via the setOverride() function on that node. This
     * function will return false if no parameter capable of editing the given
     * attribute is found.
     *
     * Once a value is set the cooked result might take some time to arrive
     * back. During that time, ViewerDelegate::getAttributes() will return the
     * manipulated value, rather than the cooked one, for those attributes.
     *
     * While a user is scrubbing a Manipulator the values might not end up
     * being committed immediately into Geolib3 to be cooked, as this might not
     * perform at an interactive speed (depending on the project complexity).
     * For this the concept of 'final' value is used. While the user is
     * scrubbing a Manipulator handle the value will not be sent back to Katana
     * as the final one. That will happen only once the user releases it,
     * which, at that point, Katana will, guaranteedly, be informed that the
     * new scene can be cooked.
     *
     * @param locationPath The location of the object being manipulated.
     * @param attrName The attribute name for the value being manipulated.
     * @param valueAttr The value being set.
     * @param isFinal True if this is a final value.
     * @return True if a parameter capable of editing the given attribute was
     *          found and was successfully set, false otherwise.
     */
    bool setValue(const std::string& locationPath, const std::string& attrName,
                  FnAttribute::Attribute valueAttr, bool isFinal);

    /**
     * @brief Marks the beginning of a batch of manipulations.
     *
     * Multiple manipulations can be batched into a group, allowing them to be
     * processed at the same time. Once called any subsequent calls to
     * setManipulatedAttribute() will be deferred until
     * closeManipulationGroup() is called.
     *
     * @param locationPath The scene graph location that this group is for.
     *                      Multiple groups can be created for different
     *                      locations.
     */
    void openManipulationGroup(const std::string& locationPath);

    /**
     * @brief Marks the end of a batch of manipulations.
     *
     * @param locationPath The scene graph location that this group is for.
     *                      Multiple groups can be created for different
     *                      locations.
     */
    void closeManipulationGroup(const std::string& locationPath);

    /**
     * @brief Gets a value that has been manipulated.
     *
     * @param locationPath The location of the manipulated object.
     * @param attrName The name of the attribute name with the manipulated
     *                  value.
     *
     * @return The value previously set by setValue().
     */
    FnAttribute::Attribute getValue(const std::string& locationPath,
                                    const std::string& attrName);

    /**
     * @brief Instantiates a ManipulatorHandle.
     *
     * Instantiates a ManipulatorHandle that is part of this Manipulator.
     *
     * @param pluginName The name of the ManipulatorHandle plugin.
     * @param name The local name of the handle.
     *
     * @return The new ManipulatorHandle instance or 0 if no plugin with the
     *           given plugin name exists.
     */
    ManipulatorHandleWrapperPtr addManipulatorHandle(
        const std::string& pluginName, const std::string& name);

    /**
     * @brief Gets a ManipulatorHandle of this Manipulator by name.
     *
     * Gets a ManipulatorHandle instance created and managed by this instance
     * of the Manipulator;
     *
     * @param name The name given to the ManipulatorHandle.
     *
     * @return The new ManipulatorHandle instance or 0 if none was created
     *          with the given name.
     */
    ManipulatorHandleWrapperPtr getManipulatorHandle(const std::string& name);

    /**
     * @brief Gets a ManipulatorHandle of this Manipulator by index.
     *
     * Gets a ManipulatorHandle instance created and managed by this instance
     * of the Manipulator;
     *
     * @param index The position in the list of ManipulatorHandles.
     *
     * @return The new ManipulatorHandle instance or a null shared pointer if
     *     none exists with the given index.
     */
    ManipulatorHandleWrapperPtr getManipulatorHandle(unsigned int index);

    /**
     * @brief Removes a ManipulatorHandle by name.
     *
     * @param name The name of the ManipulatorHandle.
     */
    void removeManipulatorHandle(const std::string& name);

    /**
     * @brief Removes a ManipulatorHandle by index.
     *
     * @param index The position in the list of ManipulatorHandles.
     */
    void removeManipulatorHandle(unsigned int index);

    /**
     * @brief Gets the number of ManipulatorHandles
     *
     * @return The number of ManipulatorHandles added to this Manipulator.
     */
    unsigned int getNumberOfManipulatorHandles() const;

    /**
     * @brief Gets the name of the ManipulatorHandles on a given index.
     *
     * @param index The position in the list of ManipulatorHandles on this
     *               Manipulator.
     *
     * @return The name of the ManipulatorHandles on the given index or empty
     *          string if it doesn't exist.
     */
    std::string getManipulatorHandleName(unsigned int index);

    /**
     * @brief Returns information about the registered Manipulators.
     *
     * @return A GroupAttribute with a child for each registered Manipulator
     *          plug-in, containing the tags returned by the Manipulator
     *          plug-in's getTags() method.
     */
    static FnAttribute::GroupAttribute GetRegisteredManipulatorsInfo();

    /**
     * @brief Returns whether the manipulated location is interactive.
     *
     * An location is determined to be interactive if the location has an
     * 'attributeEditor' attribute that specifies a node where interactive
     * changes should be stored.
     *
     * @return true if the location is interactive, otherwise false.
     */
    bool isInteractive() const;

///@cond FN_INTERNAL_DEV
public:
    static FnPlugStatus setHost(FnPluginHost* host);
    static FnPluginHost* getHost();

protected:
    static FnPluginHost* m_host;

    FnManipulatorHostSuite_v2* m_hostSuite;
    FnManipulatorHostHandle m_hostHandle;

    static const FnManipulatorHostSuite_v2* _manipulatorSuite;

/// @endcond
};

/**
 * @brief Manipulator plug-in base class.
 */
class Manipulator : public ManipulatorPluginBase
{
public:
    /** @brief Constructor. */
    Manipulator();

    /** @brief Destructor. */
    virtual ~Manipulator();

public: /* Virtual functions to be extended by the plugin. */

    /**
     * @brief Specifies if the Manipulator can be applied to a location.
     *
     * Specifies if the passed locations attributes has the attributes that are
     * required by this Manipulator. This should be extended in sub-classes.
     */
    static bool matches(const FnAttribute::GroupAttribute& locationAttrs);

    /**
     * @brief Gets the Manipulator tags.
     *
     * Returns the tags for this Manipulator. These tags are key:value pairs
     * containing meta-data that can be used by the Viewers to categorize and
     * identify the available Manipulators. For example, a possible tag could
     * be {type:transform}, meaning that the Manipulator allows to manipulate
     * object transforms (manipulators like rotate, translate, scale could be
     * tagged like this). The Viewer tab can use these tags to, for example,
     * group the Manipulators according to their functionality on a UI menu,
     * or it can discard Manipulators that are not meant to be available on
     * on that Viewer.
     *
     * @return A GroupAttribute containing the key:value pairs that define the
     *          tags of the manipulator.
     */
    static FnAttribute::GroupAttribute getTags();

    /**
     * @brief Initializes the GL components of the Viewport.
     *
     * Called when a Manipulator is created. It runs inside the correct GL
     * context. This can be used to initialize anything GL related.
     */
    virtual void setup() = 0;

    /**
     * @brief Processes UI events.
     *
     * Called whenever a user interaction event occurs.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, false otherwise. By default
     *          this returns false.
     */
    virtual bool event(const FnEventWrapper& eventData);

    /**
     * @brief Draws the Manipulator.
     *
     * Called when the scene needs to be drawn in the correct GL context. In a
     * non-GL renderer the generated image should be drawn in the GL
     * framebuffer in order to be displayed.
     */
    virtual void draw();

    /**
     * @brief Draws the Manipulator for picking.
     *
     * Called when the scene needs to be drawn in order to perform objects
     * selection. This will run in the correct GL context. In a non-GL renderer
     * the generated image should be drawn in the GL framebuffer in order to be
     * displayed.
     */
    virtual void pickerDraw(int64_t pickerId);

    /**
     * @brief Returns some arbitrary data.
     *
     * This can be used by other plugins to access some data that is specific
     * to this object after it is compiled, allowing built-in parts of existing
     * Viewers to be extendable by other plugins like ViewerDelegateComponents,
     * Viewports and ViewportLayers.
     *
     * This function should be called by other plugins after getting a
     * ManipulatorWrapperPtr and converting it into a concrete instance via
     * ManipulatorWrapper::getPluginInstance(). These other plugins will have
     * to be built with the same compiler and using the same compiler flags as
     * the ViewerDelegate so that this data can be cast and used without
     * running into C++ name mangling issues.
     *
     * @param inputData A pointer to some input data that can be used to
     *     produce the returned data.
     *
     * @return The arbitrary private data. A void pointer that can be cast into
     *     specific object types by other plugins.
     */
    virtual void* getPrivateData(void* inputData) { return 0x0; }

    /* Options setting and getting. */

    /**
     * @brief Sets a generic option.
     *
     * Optional. Reacts to a generic option being set from Python or called
     * directly by other C++ Viewer plugin classes. This can be used as a
     * message passing mechanism from the outside into the Manipulator.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     * @param attr Attribute with the value being set.
     */
    virtual void setOption(OptionIdGenerator::value_type optionId,
                           FnAttribute::Attribute attr) {}

    /**
     * @brief Gets the value of a generic option.
     *
     * Optional. Returns the value of a generic option being requested from
     * Python or from other C++ Viewer plugin classes.
     *
     * @param optionId The ID of the option created from OptionIdGenerator
     *                  or manually defined by users.
     *
     * @return Attribute with the value of the option.
     */
    virtual FnAttribute::Attribute getOption(
        OptionIdGenerator::value_type optionId);

    /**
     * @brief Sets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * setOption(OptionIdGenerator::value_type optionId,
     * FnAttribute::Attribute attr).
     * Since the ID is generated on every call, it is more efficient to
     * generate the ID once, and store it for future use.
     *
     * @param name The name of the option whose value to set.
     * @param attr Attribute with the value to set for the option.
     */
    void setOption(const std::string& name, FnAttribute::Attribute attr);

    /**
     * @brief Gets a generic option by generating an option ID from the passed
     *     name.
     *
     * This generates an Option ID from the passed string and passes it to
     * getOption(OptionIdGenerator::value_type optionId). Since the ID is
     * generated on every call, it is more efficient to generate the ID once,
     * and store it for future use.
     *
     * @param name The name of the option whose value to retrieve.
     * @return The value of the option with the given name.
     */
    FnAttribute::Attribute getOption(const std::string& name);

///@cond FN_INTERNAL_DEV
public:
    static FnManipulatorPluginSuite_v1 createSuite(
        FnManipulatorPluginHandle (*create)(FnManipulatorHostHandle hostHandle),
        int (*matches)(FnAttributeHandle locationAttributes),
        FnAttributeHandle (*getTags)());

    static FnManipulatorPluginHandle newManipulatorPluginHandle(
        Manipulator* viewport);

    static unsigned int _apiVersion;
    static const char*  _apiName;

    void setHostHandle(FnManipulatorHostHandle m_hostHandle);

/// @endcond
};

/**
 * @brief Wrapper class for a Manipulator plug-in.
 */
class ManipulatorWrapper : public ManipulatorPluginBase
{
public:
    ManipulatorWrapper(FnPluginHost* host, FnManipulatorHostHandle hostHandle,
                       FnManipulatorPluginHandle pluginHandle,
                       FnManipulatorPluginSuite_v1* pluginSuite);

    ~ManipulatorWrapper();

    /**
     * @brief Gets a pointer to the real plugin instance.
     *
     * WARNING: This function purposely breaks the compiler agnostic pattern of
     * the Katana plugin API, so it needs to be used with care. This performs a
     * dynamic cast to the real type so the implications of what that means in
     * different circumstances should be understood. If the caller and plugin
     * instance are not contained in the same shared library the RTTI check may
     * fail.
     *
     * This function allows a plugin wrapper to return the actual plugin class.
     * The plugin is returned as a pointer to a child class of Manipulator
     * that is a registerd plugin. The type of that child class needs to be
     * specified in the template, so that it can be properly cast. If the
     * specified type  doesn't match the plugin type then NULL is returned.
     *
     */
    template<class T> T* getPluginInstance()
    {
        return dynamic_cast<T*>(getPluginPointer());
    }

    bool matches(const FnAttribute::GroupAttribute& locationAttrs);

    FnAttribute::GroupAttribute getTags();

    /**
     * @brief Draws the Manipulator.
     *
     * This should be called with the correct GL context since the
     * final image will be rendered on the main framebuffer.
     */
    void draw();

    /**
     * @brief Triggers the processing of UI events on this Manipulator.
     *
     * @param eventData The event data (see FnEventWrapper).
     *
     * @return True if the event has been handled, false otherwise.
     */
    bool event(FnEventWrapper eventData);

    /**
     * @brief Draws the Manipulator for picking.
     *
     * This should be called with the correct GL context since
     * the final image will be rendered on the main framebuffer.
     *
     * @param pickerId The first ID assigned to this manipulator by the picker.
     */
    void pickerDraw(int64_t pickerId);

    /** @brief Sets a generic option. */
    void setOption(OptionIdGenerator::value_type optionId,
        FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(OptionIdGenerator::value_type optionId);
    /** @brief Sets a generic option. */
    void setOption(const std::string& name, FnAttribute::Attribute attr);
    /** @brief Gets a generic option. */
    FnAttribute::Attribute getOption(const std::string& name);

/// @cond FN_INTERNAL_DEV
private:
    Manipulator* getPluginPointer();

    /** The Manipulator plug-in. */
    FnManipulatorPluginSuite_v1* m_pluginSuite;
    FnManipulatorPluginHandle m_pluginHandle;

/// @endcond
};

typedef std::shared_ptr<ManipulatorWrapper>
    ManipulatorWrapperPtr;

/** @} */

} // ViewerAPI
} // Katana
} // Foundry

///@cond FN_INTERNAL_DEV

// Plugin-side structure to be pointed by the plugin handles.
struct FnManipulatorPluginStruct
{
public:
    FnManipulatorPluginStruct(
        Foundry::Katana::ViewerAPI::Manipulator* manipulator)
    : m_manipulator(manipulator)
    {}

    ~FnManipulatorPluginStruct()
    {};

    Foundry::Katana::ViewerAPI::Manipulator* getManipulator()
    {
        return m_manipulator.get();
    }

private:
    std::shared_ptr<
        Foundry::Katana::ViewerAPI::Manipulator> m_manipulator;
};


// Plugin Registering Macro.
#define DEFINE_MANIPULATOR_PLUGIN(MANIPULATOR_CLASS)                          \
                                                                              \
FnPlugin MANIPULATOR_CLASS##_plugin;                                          \
                                                                              \
FnManipulatorPluginHandle MANIPULATOR_CLASS##_create(                         \
    FnManipulatorHostHandle hostHandle)                                       \
{                                                                             \
    Foundry::Katana::ViewerAPI::Manipulator* manipulator =                    \
        MANIPULATOR_CLASS::create();                                          \
                                                                              \
    manipulator->setHostHandle(hostHandle);                                   \
    return Foundry::Katana::ViewerAPI::Manipulator::newManipulatorPluginHandle( \
        manipulator);                                                         \
}                                                                             \
                                                                              \
int MANIPULATOR_CLASS##_matches(FnAttributeHandle locationAttributesHandle)   \
{                                                                             \
    return MANIPULATOR_CLASS::matches(                                        \
        FnAttribute::Attribute::CreateAndRetain(locationAttributesHandle));   \
}                                                                             \
                                                                              \
FnAttributeHandle MANIPULATOR_CLASS##_getTags()                               \
{                                                                             \
    return MANIPULATOR_CLASS::getTags().getRetainedHandle();                  \
}                                                                             \
                                                                              \
FnManipulatorPluginSuite_v1 MANIPULATOR_CLASS##_suite =                       \
        Foundry::Katana::ViewerAPI::Manipulator::createSuite(                 \
                MANIPULATOR_CLASS##_create,                                   \
                MANIPULATOR_CLASS##_matches,                                  \
                MANIPULATOR_CLASS##_getTags);                                 \
                                                                              \
const void* MANIPULATOR_CLASS##_getSuite()                                    \
{                                                                             \
    return &MANIPULATOR_CLASS##_suite;                                        \
}

///@endcond
#endif /* FNMANIPULATOR_H_ */
