// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FNGLSTATEHELPER_H_
#define FNGLSTATEHELPER_H_

#include <GL/glew.h>

namespace Foundry
{
namespace Katana
{
namespace ViewerUtils
{

    /**
     * \defgroup FnGLStateHelper OpenGL State Helper
     * @{
     */

    /// Each client is responsible for setting up their own renderstate before
    /// submitting draw jobs. Often a client will need to temporarily change
    /// state for their purpose.
    /// The purpose of \c GLStateRestore is to provide a RAII mechanism to
    /// do this.

    /// The various bits to restore.
    /// \sa https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPushAttrib.xml
    enum Bit
    {
        // GL_ALPHA_TEST flag
        // GL_AUTO_NORMAL flag
        // GL_BLEND flag
        // Enable bits for the user - definable clipping planes
        // GL_COLOR_MATERIAL
        // GL_CULL_FACE flag
        // GL_DEPTH_TEST flag
        // GL_DITHER flag
        // GL_FOG flag
        // GL_LIGHTi where 0 <= i < GL_MAX_LIGHTS
        // GL_LIGHTING flag
        // GL_LINE_SMOOTH flag
        // GL_LINE_STIPPLE flag
        // GL_COLOR_LOGIC_OP flag
        // GL_INDEX_LOGIC_OP flag
        // GL_MAP1_x where x is a map type
        // GL_MAP2_x where x is a map type
        // GL_MULTISAMPLE flag
        // GL_NORMALIZE flag
        // GL_POINT_SMOOTH flag
        // GL_POLYGON_OFFSET_LINE flag
        // GL_POLYGON_OFFSET_FILL flag
        // GL_POLYGON_OFFSET_POINT flag
        // GL_POLYGON_SMOOTH flag
        // GL_POLYGON_STIPPLE flag
        // GL_SAMPLE_ALPHA_TO_COVERAGE flag
        // GL_SAMPLE_ALPHA_TO_ONE flag
        // GL_SAMPLE_COVERAGE flag
        // GL_SCISSOR_TEST flag
        // GL_STENCIL_TEST flag
        // GL_TEXTURE_1D flag
        // GL_TEXTURE_2D flag
        // GL_TEXTURE_3D flag
        // Flags GL_TEXTURE_GEN_x where x is S, T, R, or Q
        Enable = GL_ENABLE_BIT,

        //Accumulation buffer clear value
        AccumuBuffer = GL_ACCUM_BUFFER_BIT,

        // GL_ALPHA_TEST enable bit
        // Alpha test function and reference value
        // GL_BLEND enable bit
        // Blending source and destination functions
        // Constant blend color
        // Blending equation
        // GL_DITHER enable bit
        // GL_DRAW_BUFFER setting
        // GL_COLOR_LOGIC_OP enable bit
        // GL_INDEX_LOGIC_OP enable bit
        // Logic op function
        // Color mode and index mode clear values
        // Color mode and index mode writemasks
        ColorBuffer = GL_COLOR_BUFFER_BIT,

        // Current RGBA color
        // Current color index
        // Current normal vector
        // Current texture coordinates
        // Current raster position
        // GL_CURRENT_RASTER_POSITION_VALID flag
        // RGBA color associated with current raster position
        // Color index associated with current raster position
        // Texture coordinates associated with current raster position
        // GL_EDGE_FLAG flag
        Current = GL_CURRENT_BIT,

        // GL_DEPTH_TEST enable bit
        // Depth buffer test function
        // Depth buffer clear value
        // GL_DEPTH_WRITEMASK enable bit
        DepthBuffer = GL_DEPTH_BUFFER_BIT,

        // GL_MAP1_x enable bits, where x is a map type
        // GL_MAP2_x enable bits, where x is a map type
        // 1D grid endpoints and divisions
        // 2D grid endpoints and divisions
        // GL_AUTO_NORMAL enable bit
        Eval = GL_EVAL_BIT,

        // GL_FOG enable bit
        // Fog color
        // Fog density
        // Linear fog start
        // Linear fog end
        // Fog index
        // GL_FOG_MODE value
        Fog = GL_FOG_BIT,

        // GL_PERSPECTIVE_CORRECTION_HINT setting
        // GL_POINT_SMOOTH_HINT setting
        // GL_LINE_SMOOTH_HINT setting
        // GL_POLYGON_SMOOTH_HINT setting
        // GL_FOG_HINT setting
        // GL_GENERATE_MIPMAP_HINT setting
        // GL_TEXTURE_COMPRESSION_HINT setting
        Hint = GL_HINT_BIT,

        // GL_COLOR_MATERIAL enable bit
        // GL_COLOR_MATERIAL_FACE value
        // Color material parameters that are tracking the current color
        // Ambient scene color
        // GL_LIGHT_MODEL_LOCAL_VIEWER value
        // GL_LIGHT_MODEL_TWO_SIDE setting
        // GL_LIGHTING enable bit
        // Enable bit for each light
        // Ambient, diffuse, and specular intensity for each light
        // Direction, position, exponent, and cutoff angle for each light
        // Constant, linear, and quadratic attenuation factors for each light
        // Ambient, diffuse, specular, and emissive color for each material
        // Ambient, diffuse, and specular color indices for each material
        // Specular exponent for each material
        // GL_SHADE_MODEL setting
        Lighting = GL_LIGHTING_BIT,

        // GL_LINE_SMOOTH flag
        // GL_LINE_STIPPLE enable bit
        // Line stipple pattern and repeat counter
        // Line width
        Line = GL_LINE_BIT,

        // GL_LIST_BASE setting
        List = GL_LIST_BIT,

        // GL_MULTISAMPLE flag
        // GL_SAMPLE_ALPHA_TO_COVERAGE flag
        // GL_SAMPLE_ALPHA_TO_ONE flag
        // GL_SAMPLE_COVERAGE flag
        // GL_SAMPLE_COVERAGE_VALUE value
        // GL_SAMPLE_COVERAGE_INVERT value
        Multisample = GL_MULTISAMPLE_BIT,

        // GL_RED_BIAS and GL_RED_SCALE settings
        // GL_GREEN_BIAS and GL_GREEN_SCALE values
        // GL_BLUE_BIAS and GL_BLUE_SCALE
        // GL_ALPHA_BIAS and GL_ALPHA_SCALE
        // GL_DEPTH_BIAS and GL_DEPTH_SCALE
        // GL_INDEX_OFFSET and GL_INDEX_SHIFT values
        // GL_MAP_COLOR and GL_MAP_STENCIL flags
        // GL_ZOOM_X and GL_ZOOM_Y factors
        // GL_READ_BUFFER setting
        PixelMode = GL_PIXEL_MODE_BIT,

        // GL_POINT_SMOOTH flag
        // Point size
        Point = GL_POINT_BIT,

        // GL_CULL_FACE enable bit
        // GL_CULL_FACE_MODE value
        // GL_FRONT_FACE indicator
        // GL_POLYGON_MODE setting
        // GL_POLYGON_SMOOTH flag
        // GL_POLYGON_STIPPLE enable bit
        // GL_POLYGON_OFFSET_FILL flag
        // GL_POLYGON_OFFSET_LINE flag
        // GL_POLYGON_OFFSET_POINT flag
        // GL_POLYGON_OFFSET_FACTOR
        // GL_POLYGON_OFFSET_UNITS
        Polygon = GL_POLYGON_BIT,

        // Polygon stipple image
        PolygonStipple = GL_POLYGON_STIPPLE_BIT,

        // GL_SCISSOR_TEST flag
        // Scissor box
        Scissor = GL_SCISSOR_BIT,

        // GL_STENCIL_TEST enable bit
        // Stencil function and reference value
        // Stencil value mask
        // Stencil fail, pass, and depth buffer pass actions
        // Stencil buffer clear value
        // Stencil buffer writemask
        StencilBuffer = GL_STENCIL_BUFFER_BIT,

        // Enable bits for the four texture coordinates
        // Border color for each texture image
        // Minification function for each texture image
        // Magnification function for each texture image
        // Texture coordinates and wrap mode for each texture image
        // Color and mode for each texture environment
        // Enable bits GL_TEXTURE_GEN_x, x is S, T, R, and Q
        // GL_TEXTURE_GEN_MODE setting for S, T, R, and Q
        // glTexGen plane equations for S, T, R, and Q
        // Current texture bindings(for example, GL_TEXTURE_BINDING_2D)
        Texture = GL_TEXTURE_BIT,

        // Coefficients of the six clipping planes
        // Enable bits for the user - definable clipping planes
        // GL_MATRIX_MODE value
        // GL_NORMALIZE flag
        // GL_RESCALE_NORMAL flag
        Transform = GL_TRANSFORM_BIT,

        // Depth range (near and far)
        // Viewport origin and extent
        Viewport = GL_VIEWPORT_BIT,
    };

    /// Bindings to unbind on destruction.
    enum Binding
    {
        // glBindBuffer(GL_ARRAY_BUFFER, 0)
        UnbindArrayBuffer = 1u << 0u,
        // glBindTexture(GL_TEXTURE_2D, 0)
        UnbindTexture = 1u << 1u,
        // glBindVertexArray(0)
        UnbindVertexArray = 1u << 2u,
        // glUseProgram(0);
        UnbindProgram = 1u << 3u,

        // Unbind all.
        UnbindAll = UnbindArrayBuffer | UnbindTexture | UnbindVertexArray |
                    UnbindProgram
    };

    /// Provide a RAII mechanism to temporarily change the OpenGL state.
    class GLStateRestore
    {
        using Bindings = unsigned;

    public:
        /// Constructor. \c mask should be any combination of the \c Bit
        /// enumeration values.
        explicit GLStateRestore(GLbitfield mask, Bindings bindings = 0)
            : m_bindings{bindings}
        {
            glPushAttrib(mask);
        }

        /// Destructor.
        ~GLStateRestore()
        {
            glPopAttrib();
            if (m_bindings)
            {
                if (m_bindings & UnbindArrayBuffer)
                    glBindBuffer(GL_ARRAY_BUFFER, 0);
                if (m_bindings & UnbindTexture)
                    glBindTexture(GL_TEXTURE_2D, 0);
                if (m_bindings & UnbindVertexArray)
                    glBindVertexArray(0);
                if (m_bindings & UnbindProgram)
                    glUseProgram(0);
            }
        }

    private:
        Bindings m_bindings;
    };
    /** @} */

}  // namespace ViewerUtils
}  // namespace Katana
}  // namespace Foundry

#endif  // FNGLSTATEHELPER_H_
