// Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.


#ifndef KATANA_PLUGINAPIS_FNSCENEGRAPHITERATOR_FNSCENEGRAPHITERATOR_H_
#define KATANA_PLUGINAPIS_FNSCENEGRAPHITERATOR_FNSCENEGRAPHITERATOR_H_

#include <string>

#include "FnAttribute/FnAttribute.h"
#include "FnPluginSystem/FnPluginSystem.h"

#include "FnScenegraphIterator/FnScenegraphIteratorAPI.h"
#include "FnScenegraphIterator/ns.h"
#include "FnScenegraphIterator/suite/FnScenegraphIteratorSuite.h"

FNSCENEGRAPHITERATOR_NAMESPACE_ENTER
{

/**
 * \defgroup SI Scenegraph Iterator (SI) API
 * @{
 *
 *@brief API that allows to iterate through the locations and attributes on a Scenegraph.
 *
 * This API allows to iterate through the locations on a scenegraph and query
 * their attributes. This is an auxiliary API that can be used when implementing,
 * for example, a connection of Katana to a Renderer. In this specific case the
 * renderer will have at some point to discover which locations and attributes
 * exist on the Katana scene being rendered.
 *
 * Given the lazy initialization nature of Katana the scenegraph locations and
 * attributes will be procedurally generated under the hood as the iterators
 * go through the scene. Whenever a FnScenegraphIterator is instantiated Katana
 * will generate its corresponding location if needed on that moment.
 *
 */




/**
 * @brief The Scenegraph Iterator that 'points' at a location in a scenegraph.
 *
 * Each instance of this class will be conceptually pointing at a specific
 * location of the scenegraph. It presents methods that return other iterators
 * that point at child, sibling, parent and other arbitraty locations on the
 * scenegraph. The location pointed by the instance is referred here as "current
 * location".
 */
class FNSCENEGRAPHITERATOR_API FnScenegraphIterator
{
public:
    FnScenegraphIterator(FnSgIteratorHandle handle);
    FnScenegraphIterator();

    ~FnScenegraphIterator();

    FnScenegraphIterator(const FnScenegraphIterator& rhs);
    FnScenegraphIterator& operator=(const FnScenegraphIterator& rhs);

    /**
     * @brief Returns true if this Scenegraph Iterator is valid.
     */
    bool isValid() const {return _handle != 0x0;}

    /**
     * @brief Returns the name of the current Location. (ex: "pony" in "/root/world/geo/pony")
     */
    std::string getName() const;

    /**
     * @brief Returns the full path name of the current Location. (ex: "/root/world/geo/pony")
     */
    std::string getFullName() const;

    /**
     * @brief Returns the type of the current Location. (ex: "subdmesh" or "group")
     */
    std::string getType() const;

    /**
     * @brief Returns an iterator that poins at the first child of the current location.
     */
    FnScenegraphIterator getFirstChild(bool evict=false) const;


    /**
     * @brief Returns an iterator that poins at the next sibling of the current location.
     */
    FnScenegraphIterator getNextSibling(bool evict=false) const;

    /**
     * @brief Returns an iterator that poins at the parent of the current location.
     */
    FnScenegraphIterator getParent() const;

    /**
     * @brief Returns an iterator that poins at the root location of the scenegraph where
     * the current locations lives.
     */
    FnScenegraphIterator getRoot() const;

    /**
     * @return a list of immediate potential child names
     */

    FnAttribute::StringAttribute getPotentialChildren() const;

    /**
     * @brief Returns an iterator to a child of the current location.
     *
     * Returns an iterator that points at a child location with the given name
     * underneath the current location.
     *
     * @param name The name of the child location for which to return a scene
     *     graph iterator.
     * @param evict Flag that controls whether the Geolib3 Runtime will flush
     *     cached scene data, while protecting data that is deemed potentially
     *     relevant to the child scene graph location with the given name, for
     *     example that of ancestor locations.
     */
    FnScenegraphIterator getChildByName(const std::string &name, bool evict=false) const;

    /**
     * @brief Returns an iterator pointing at a location with a given path.
     *
     * Returns an iterator that points to a location in the scene graph that is
     * identified by the given forward slash-separated scene graph location
     * path.
     *
     * @param path The forward slash-separated path of the scene graph location
     *     for which to return a scene graph iterator.
     * @param evict Flag that controls whether the Geolib3 Runtime will flush
     *     cached scene data, while protecting data that is deemed potentially
     *     relevant to the scene graph location with the given path, for
     *     example that of ancestor locations.
     */
    FnScenegraphIterator getByPath(const std::string &path, bool evict=false) const;


    /**
     * @brief Get a specific attribute from the location pointed by the iterator.
     *
     * Returns an existing Attribute from the current Location if no location is
     * specified. If there isn't any Attribute with the given name on the given
     * Location then an invalid FnAttribute is returned (which can be checked
     * with the function FnAttribute::isValid()).
     *
     * @param name The name of the attribute - can be a child of a GroupAttribute
     *              (ex: "geometry.point.P")
     * @param global If false then the global attributes (inherited from parent
     *               Locations) will be ignored. In this case if the requested
     *               Attribute is global, then an invalid attribute is returned.
     *
     */
    FnAttribute::Attribute getAttribute(const std::string &name, bool global=false) const;

    /**
     * @brief Return the names of all the attributes on the location pointed by
     * the iterator.
     *
     * @return A valid StringAttribute
     */
    FnAttribute::StringAttribute getAttributeNames() const;

    /**
     * @return a GroupAttribute that represents the global transform for this
     *     location.
     */
    FnAttribute::GroupAttribute getGlobalXFormGroup() const;

    /**
     * Notifies the Geolib Runtime that the renderer plug-in will not interact
     * with it again, and therefore resources can be deallocated (e.g. cached
     * scene data, internal threads, etc.).
     *
     * To be used (just) after the scene has been entirely traversed, and
     * typically before the renderer starts rendering.
     *
     * This function can take time. It may be desirable to invoke this function
     * from a worker thread to avoid blocking the actual rendering.
     *
     * @note All iterators associated with the same runtime instance will be
     * invalidated, as no more interaction is expected from them.
     */
    void finalizeRuntime() const;

    ///@cond FN_INTERNAL_DEV

    static FnPlugStatus setHost(FnPluginHost *host);

    static FnSgIteratorHostSuite_v3 *_suite;

    FnSgIteratorHandle getHandle() const
    {
        return _handle;
    }

    /**
     * Returns a Group attribute representing the description of the Op tree
     * stored in the file with the given |filename|.
     */
    static FnAttribute::GroupAttribute getOpTreeDescriptionFromFile(
            const std::string & opTreeFileName);

    /**
     * Returns a FnScenegraphIterator object pointing to the root location of
     * the scene graph built evaluating the Op tree stored in the file with the
     * given |filename|.
     */
    static FnScenegraphIterator getIteratorFromFile(
            const std::string & opTreeFileName);

    /**
     * Returns a FnScenegraphIterator object pointing to the root location of
     * the scene graph built evaluating the Op tree encoded in the given Group
     * attribute.
     */
    static FnScenegraphIterator getIteratorFromOpTreeDescription(
            const FnAttribute::GroupAttribute & opTreeDescription);

protected:
    void acceptHandle(const FnScenegraphIterator &rhs);

private:

    FnSgIteratorHandle _handle;

    ///@endcond
};



/// @}

}
FNSCENEGRAPHITERATOR_NAMESPACE_EXIT

#endif  // KATANA_PLUGINAPIS_FNSCENEGRAPHITERATOR_FNSCENEGRAPHITERATOR_H_
