#ifndef KATANA_PLUGINAPI_FNPLATFORM_STRINGVIEW_H_
#define KATANA_PLUGINAPI_FNPLATFORM_STRINGVIEW_H_
#include <algorithm>
#include <cassert>
#include <cstddef>
#include <iterator>
#include <limits>
#include <ostream>
#include <stdexcept>
#include <string>
#include <utility>

#include <FnPlatform/internal/Portability.h>
#include <FnPlatform/internal/TypeTraits.h>
#include <FnPlatform/ns.h>

FNPLATFORM_NAMESPACE_ENTER
{
/// \class StringView
///
/// Wraps an existing string without taking ownership. StringView objects are
/// cheap to construct and cheap to copy. The implementation consists of two
/// data members: a pointer to the start of the string and a length.
///
/// This is a simplified implementation of the C++17 \c string_view class
/// derived from the libc++ standard library as of LLVM 4.0.
///
/// See: <http://en.cppreference.com/w/cpp/string/basic_string_view>.
class StringView
{
public:
    // types
    typedef std::char_traits<char> traits_type;
    typedef char value_type;
    typedef char* pointer;
    typedef const char* const_pointer;
    typedef char& reference;
    typedef const char& const_reference;
    typedef const_pointer const_iterator;  // See [string.view.iterators]
    typedef const_iterator iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
    typedef const_reverse_iterator reverse_iterator;
    typedef std::size_t size_type;
    typedef std::ptrdiff_t difference_type;
    static FNKAT_CONSTEXPR_OR_CONST size_type npos = static_cast<size_type>(-1);

    // [string.view.cons], construct/copy
    FNKAT_CONSTEXPR StringView() FNKAT_NOEXCEPT : data_(NULL), size_(0) {}

    FNKAT_CONSTEXPR StringView(const StringView& rhs) FNKAT_NOEXCEPT
        : data_(rhs.data_), size_(rhs.size_)
    {
    }

    // NOLINTNEXTLINE(runtime/explicit)
    /* implicit */ StringView(const std::string& rhs) FNKAT_NOEXCEPT
        : data_(rhs.data()),
          size_(rhs.size())
    {
    }

    StringView& operator=(const StringView& rhs) FNKAT_NOEXCEPT
    {
        data_ = rhs.data_;
        size_ = rhs.size_;
        return *this;
    }

    FNKAT_CONSTEXPR StringView(const char* s, size_type len)
        : data_(s), size_(len)
    {
    }

    // NOLINTNEXTLINE(runtime/explicit)
    /* implicit */ StringView(const char* s)
        : data_(s), size_(traits_type::length(s))
    {
    }

    template <typename StringT>
    /* implicit */ StringView(
        const StringT& s,
        typename internal::EnableIf<
            internal::IsStringContainer<StringT>::value>::type* = 0)
        : data_(s.data()), size_(s.size())
    {
    }

    // [string.view.iterators], iterators
    FNKAT_CONSTEXPR const_iterator begin() const FNKAT_NOEXCEPT
    {
        return cbegin();
    }

    FNKAT_CONSTEXPR const_iterator end() const FNKAT_NOEXCEPT { return cend(); }

    FNKAT_CONSTEXPR const_iterator cbegin() const FNKAT_NOEXCEPT
    {
        return data_;
    }

    FNKAT_CONSTEXPR const_iterator cend() const FNKAT_NOEXCEPT
    {
        return data_ + size_;
    }

    const_reverse_iterator rbegin() const FNKAT_NOEXCEPT
    {
        return const_reverse_iterator(cend());
    }

    const_reverse_iterator rend() const FNKAT_NOEXCEPT
    {
        return const_reverse_iterator(cbegin());
    }

    const_reverse_iterator crbegin() const FNKAT_NOEXCEPT
    {
        return const_reverse_iterator(cend());
    }

    const_reverse_iterator crend() const FNKAT_NOEXCEPT
    {
        return const_reverse_iterator(cbegin());
    }

    // [string.view.capacity], capacity
    FNKAT_CONSTEXPR size_type size() const FNKAT_NOEXCEPT { return size_; }

    FNKAT_CONSTEXPR size_type length() const FNKAT_NOEXCEPT { return size_; }

    FNKAT_CONSTEXPR size_type max_size() const FNKAT_NOEXCEPT
    {
        return std::numeric_limits<size_type>::max();
    }

    FNKAT_CONSTEXPR bool empty() const FNKAT_NOEXCEPT { return size_ == 0; }

    // [string.view.access], element access
    FNKAT_CONSTEXPR
    const_reference operator[](size_type pos) const FNKAT_NOEXCEPT
    {
        return data_[pos];
    }

    FNKAT_CONSTEXPR const_reference at(size_type pos) const
    {
        return pos >= size() ? (throw_out_of_range("StringView::at"), data_[0])
                             : data_[pos];
    }

    FNKAT_CONSTEXPR const_reference front() const
    {
#ifndef NDEBUG
        return empty() ? (throw_out_of_range("StringView::front()"), data_[0])
                       : data_[0];
#else
        return data_[0];
#endif  // NDEBUG
    }

    FNKAT_CONSTEXPR const_reference back() const
    {
#ifndef NDEBUG
        return empty() ? (throw_out_of_range("StringView::back()"),
                          data_[size_ - 1])
                       : data_[size_ - 1];
#else
        return data_[size_ - 1];
#endif  // NDEBUG
    }

    FNKAT_CONSTEXPR const_pointer data() const FNKAT_NOEXCEPT { return data_; }

    // [string.view.modifiers], modifiers:
    void remove_prefix(size_type n) FNKAT_NOEXCEPT
    {
        assert(n <= size() && "remove_prefix() can't remove more than size()");
        data_ += n;
        size_ -= n;
    }

    void remove_suffix(size_type n) FNKAT_NOEXCEPT
    {
        assert(n <= size() && "remove_suffix() can't remove more than size()");
        size_ -= n;
    }

    void swap(StringView& other) FNKAT_NOEXCEPT
    {
        std::swap(data_, other.data_);
        std::swap(size_, other.size_);
    }

#if __cplusplus >= 201103L || _MSVC_LANG >= 201103L
    explicit operator std::string() const
    {
        return std::string(begin(), end());
    }
#else
    // C++98 -- no explicit conversions.
    operator std::string() const { return std::string(begin(), end()); }
#endif

    size_type copy(char* s, size_type n, size_type pos = 0) const
    {
        if (pos > size())
            throw_out_of_range("StringView::copy");
        size_type rlen = std::min(n, size() - pos);
        traits_type::copy(s, data() + pos, rlen);
        return rlen;
    }

    StringView substr(size_type pos = 0, size_type n = npos) const
    {
        return pos > size()
                   ? (throw_out_of_range("StringView::substr"), StringView())
                   : StringView(data() + pos, std::min(n, size() - pos));
    }

    int compare(StringView sv) const FNKAT_NOEXCEPT
    {
        size_type rlen = std::min(size(), sv.size());
        int retval = traits_type::compare(data(), sv.data(), rlen);
        if (retval == 0)  // first rlen chars matched
            retval = size() == sv.size() ? 0 : (size() < sv.size() ? -1 : 1);
        return retval;
    }

    int compare(size_type pos1, size_type n1, StringView sv) const
    {
        return substr(pos1, n1).compare(sv);
    }

    int compare(size_type pos1,
                size_type n1,
                StringView sv,
                size_type pos2,
                size_type n2) const
    {
        return substr(pos1, n1).compare(sv.substr(pos2, n2));
    }

    int compare(const char* s) const FNKAT_NOEXCEPT
    {
        return compare(StringView(s));
    }

    int compare(size_type pos1, size_type n1, const char* s) const
    {
        return substr(pos1, n1).compare(StringView(s));
    }

    int compare(size_type pos1, size_type n1, const char* s, size_type n2) const
    {
        return substr(pos1, n1).compare(StringView(s, n2));
    }

    // find
    size_type find(StringView s, size_type pos = 0) const FNKAT_NOEXCEPT
    {
        assert((s.size() == 0 || s.data() != NULL) &&
               "StringView::find(): received NULL");
        return str_find(data(), size(), s.data(), pos, s.size());
    }

    size_type find(char c, size_type pos = 0) const FNKAT_NOEXCEPT
    {
        return str_find(data(), size(), c, pos);
    }

    size_type find(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) && "StringView::find(): received NULL");
        return str_find(data(), size(), s, pos, n);
    }

    size_type find(const char* s, size_type pos = 0) const
    {
        assert(s != NULL && "StringView::find(): received NULL");
        return str_find(data(), size(), s, pos, traits_type::length(s));
    }

    // rfind
    size_type rfind(StringView s, size_type pos = npos) const FNKAT_NOEXCEPT
    {
        assert((s.size() == 0 || s.data() != NULL) &&
               "StringView::find(): received NULL");
        return str_rfind(data(), size(), s.data(), pos, s.size());
    }

    size_type rfind(char c, size_type pos = npos) const FNKAT_NOEXCEPT
    {
        return str_rfind(data(), size(), c, pos);
    }

    size_type rfind(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) && "StringView::rfind(): received NULL");
        return str_rfind(data(), size(), s, pos, n);
    }

    size_type rfind(const char* s, size_type pos = npos) const
    {
        assert(s != NULL && "StringView::rfind(): received NULL");
        return str_rfind(data(), size(), s, pos, traits_type::length(s));
    }

    // find_first_of
    size_type find_first_of(StringView s,
                            size_type pos = 0) const FNKAT_NOEXCEPT
    {
        assert(((s.size() == 0 || s.data() != NULL)) &&
               "StringView::find_first_of(): received NULL");
        return str_find_first_of(data(), size(), s.data(), pos, s.size());
    }

    size_type find_first_of(char c, size_type pos = 0) const FNKAT_NOEXCEPT
    {
        return find(c, pos);
    }

    size_type find_first_of(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) &&
               "StringView::find_first_of(): received NULL");
        return str_find_first_of(data(), size(), s, pos, n);
    }

    size_type find_first_of(const char* s, size_type pos = 0) const
    {
        assert(s != NULL && "StringView::find_first_of(): received NULL");
        return str_find_first_of(data(), size(), s, pos,
                                 traits_type::length(s));
    }

    // find_last_of
    size_type find_last_of(StringView s,
                           size_type pos = npos) const FNKAT_NOEXCEPT
    {
        assert((s.size() == 0 || s.data() != NULL) &&
               "StringView::find_last_of(): received NULL");
        return str_find_last_of(data(), size(), s.data(), pos, s.size());
    }

    size_type find_last_of(char c, size_type pos = npos) const FNKAT_NOEXCEPT
    {
        return rfind(c, pos);
    }

    size_type find_last_of(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) &&
               "StringView::find_last_of(): received NULL");
        return str_find_last_of(data(), size(), s, pos, n);
    }

    size_type find_last_of(const char* s, size_type pos = npos) const
    {
        assert(s != NULL && "StringView::find_last_of(): received NULL");
        return str_find_last_of(data(), size(), s, pos, traits_type::length(s));
    }

    // find_first_not_of
    size_type find_first_not_of(StringView s,
                                size_type pos = 0) const FNKAT_NOEXCEPT
    {
        assert((s.size() == 0 || s.data() != NULL) &&
               "StringView::find_first_not_of(): received NULL");
        return str_find_first_not_of(data(), size(), s.data(), pos, s.size());
    }

    size_type find_first_not_of(char c, size_type pos = 0) const FNKAT_NOEXCEPT
    {
        return str_find_first_not_of(data(), size(), c, pos);
    }

    size_type find_first_not_of(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) &&
               "StringView::find_first_not_of(): received NULL");
        return str_find_first_not_of(data(), size(), s, pos, n);
    }

    size_type find_first_not_of(const char* s, size_type pos = 0) const
    {
        assert(s != NULL && "StringView::find_first_not_of(): received NULL");
        return str_find_first_not_of(data(), size(), s, pos,
                                     traits_type::length(s));
    }

    // find_last_not_of
    size_type find_last_not_of(StringView s,
                               size_type pos = npos) const FNKAT_NOEXCEPT
    {
        assert((s.size() == 0 || s.data() != NULL) &&
               "StringView::find_last_not_of(): received NULL");
        return str_find_last_not_of(data(), size(), s.data(), pos, s.size());
    }

    size_type find_last_not_of(char c,
                               size_type pos = npos) const FNKAT_NOEXCEPT
    {
        return str_find_last_not_of(data(), size(), c, pos);
    }

    size_type find_last_not_of(const char* s, size_type pos, size_type n) const
    {
        assert((n == 0 || s != NULL) &&
               "StringView::find_last_not_of(): received NULL");
        return str_find_last_not_of(data(), size(), s, pos, n);
    }

    size_type find_last_not_of(const char* s, size_type pos = npos) const
    {
        assert(s != NULL && "StringView::find_last_not_of(): received NULL");
        return str_find_last_not_of(data(), size(), s, pos,
                                    traits_type::length(s));
    }

    // [string.view.comparison]
    friend bool operator==(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        if (lhs.size() != rhs.size())
            return false;
        return lhs.compare(rhs) == 0;
    }

    friend bool operator!=(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        if (lhs.size() != rhs.size())
            return true;
        return lhs.compare(rhs) != 0;
    }

    friend bool operator<(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        return lhs.compare(rhs) < 0;
    }

    friend bool operator>(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        return lhs.compare(rhs) > 0;
    }

    friend bool operator<=(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        return lhs.compare(rhs) <= 0;
    }

    friend bool operator>=(StringView lhs, StringView rhs) FNKAT_NOEXCEPT
    {
        return lhs.compare(rhs) >= 0;
    }

    friend std::ostream& operator<<(std::ostream& os, StringView sv)
    {
        // Implementation derived from:
        // https://github.com/llvm-mirror/libcxx/blob/release_40/. See:
        //
        // - include/locale: __pad_and_output().
        // - include/ostream: __put_character_sequence().

        std::ostream::sentry sentry(os);
        if (!sentry)
            return os;

        std::streambuf& buffer = *os.rdbuf();

        const char* const str = sv.data();
        const std::streamsize len = static_cast<std::streamsize>(sv.size());

        const bool leftAligned =
            (os.flags() & std::ios_base::adjustfield) == std::ios_base::left;
        const char fillChar = os.fill();
        const std::streamsize numFillChars =
            std::max(static_cast<std::streamsize>(0), os.width() - len);

        if (leftAligned)
        {
            if (buffer.sputn(str, len) != len)
            {
                os.setstate(std::ios_base::badbit | std::ios_base::failbit);
                return os;
            }
        }

        if (numFillChars)
        {
            char fillBuf[32];
            std::fill(fillBuf, fillBuf + sizeof(fillBuf), fillChar);

            std::streamsize remaining = numFillChars;
            while (remaining)
            {
                const std::streamsize n = std::min(
                    static_cast<std::streamsize>(sizeof(fillBuf)), remaining);
                if (buffer.sputn(fillBuf, n) != n)
                {
                    os.setstate(std::ios_base::badbit | std::ios_base::failbit);
                    return os;
                }
                remaining -= n;
            }
        }

        if (!leftAligned)
        {
            if (buffer.sputn(str, len) != len)
            {
                os.setstate(std::ios_base::badbit | std::ios_base::failbit);
                return os;
            }
        }

        os.width(0);
        return os;
    }

private:
    static void throw_out_of_range(const char* msg)
    {
        throw std::out_of_range(msg);
    }

    // -------------------------------------------------------------------------
    // String search routines derived from libc++. See:
    // <https://github.com/llvm-mirror/libcxx/blob/release_40/> --
    // include/__string.

    // str_find
    static size_type str_find(const char* p,
                              size_type sz,
                              char c,
                              size_type pos) FNKAT_NOEXCEPT
    {
        if (pos >= sz)
            return npos;
        const char* r = traits_type::find(p + pos, sz - pos, c);
        if (r == 0)
            return npos;
        return static_cast<size_type>(r - p);
    }

    static const char* search_substring(const char* first1,
                                        const char* last1,
                                        const char* first2,
                                        const char* last2)
    {
        // Take advantage of knowing source and pattern lengths.
        // Stop short when source is smaller than pattern.
        const std::ptrdiff_t len2 = last2 - first2;
        if (len2 == 0)
            return first1;

        std::ptrdiff_t len1 = last1 - first1;
        if (len1 < len2)
            return last1;

        // First element of first2 is loop invariant.
        char f2 = *first2;
        while (true)
        {
            len1 = last1 - first1;
            // Check whether first1 still has at least len2 bytes.
            if (len1 < len2)
                return last1;

            // Find f2 the first byte matching in first1.
            first1 = traits_type::find(
                first1, static_cast<std::size_t>(len1 - len2 + 1), f2);
            if (first1 == 0)
                return last1;

            // It is faster to compare from the first byte of first1 even if we
            // already know that it matches the first byte of first2: this is
            // because first2 is most likely aligned, as it is user's "pattern"
            // string, and first1 + 1 is most likely not aligned, as the match
            // is in the middle of the string.
            if (traits_type::compare(first1, first2,
                                     static_cast<std::size_t>(len2)) == 0)
            {
                return first1;
            }

            ++first1;
        }
    }

    static size_type str_find(const char* p,
                              size_type sz,
                              const char* s,
                              size_type pos,
                              size_type n) FNKAT_NOEXCEPT
    {
        if (pos > sz)
            return npos;

        if (n == 0)  // There is nothing to search, just return pos.
            return pos;

        const char* r = search_substring(p + pos, p + sz, s, s + n);

        if (r == p + sz)
            return npos;
        return static_cast<size_type>(r - p);
    }

    // str_rfind
    static size_type str_rfind(const char* p,
                               size_type sz,
                               char c,
                               size_type pos) FNKAT_NOEXCEPT
    {
        if (sz < 1)
            return npos;
        if (pos < sz)
            ++pos;
        else
            pos = sz;
        for (const char* ps = p + pos; ps != p;)
        {
            if (traits_type::eq(*--ps, c))
                return static_cast<size_type>(ps - p);
        }
        return npos;
    }

    static size_type str_rfind(const char* p,
                               size_type sz,
                               const char* s,
                               size_type pos,
                               size_type n) FNKAT_NOEXCEPT
    {
        pos = std::min(pos, sz);
        if (n < sz - pos)
            pos += n;
        else
            pos = sz;
        const char* r = std::find_end(p, p + pos, s, s + n, traits_type::eq);
        if (n > 0 && r == p + pos)
            return npos;
        return static_cast<size_type>(r - p);
    }

    // str_find_first_of
    static size_type str_find_first_of(const char* p,
                                       size_type sz,
                                       const char* s,
                                       size_type pos,
                                       size_type n) FNKAT_NOEXCEPT
    {
        if (pos >= sz || n == 0)
            return npos;
        const char* r =
            std::find_first_of(p + pos, p + sz, s, s + n, traits_type::eq);
        if (r == p + sz)
            return npos;
        return static_cast<size_type>(r - p);
    }

    // str_find_last_of
    static size_type str_find_last_of(const char* p,
                                      size_type sz,
                                      const char* s,
                                      size_type pos,
                                      size_type n) FNKAT_NOEXCEPT
    {
        if (n != 0)
        {
            if (pos < sz)
                ++pos;
            else
                pos = sz;
            for (const char* ps = p + pos; ps != p;)
            {
                const char* r = traits_type::find(s, n, *--ps);
                if (r)
                    return static_cast<size_type>(ps - p);
            }
        }
        return npos;
    }

    // str_find_first_not_of
    static size_type str_find_first_not_of(const char* p,
                                           size_type sz,
                                           const char* s,
                                           size_type pos,
                                           size_type n) FNKAT_NOEXCEPT
    {
        if (pos < sz)
        {
            const char* pe = p + sz;
            for (const char* ps = p + pos; ps != pe; ++ps)
                if (traits_type::find(s, n, *ps) == 0)
                    return static_cast<size_type>(ps - p);
        }
        return npos;
    }

    static size_type str_find_first_not_of(const char* p,
                                           size_type sz,
                                           char c,
                                           size_type pos) FNKAT_NOEXCEPT
    {
        if (pos < sz)
        {
            const char* pe = p + sz;
            for (const char* ps = p + pos; ps != pe; ++ps)
                if (!traits_type::eq(*ps, c))
                    return static_cast<size_type>(ps - p);
        }
        return npos;
    }

    // str_find_last_not_of
    static size_type str_find_last_not_of(const char* p,
                                          size_type sz,
                                          const char* s,
                                          size_type pos,
                                          size_type n) FNKAT_NOEXCEPT
    {
        if (pos < sz)
            ++pos;
        else
            pos = sz;
        for (const char* ps = p + pos; ps != p;)
            if (traits_type::find(s, n, *--ps) == 0)
                return static_cast<size_type>(ps - p);
        return npos;
    }

    static size_type str_find_last_not_of(const char* p,
                                          size_type sz,
                                          char c,
                                          size_type pos) FNKAT_NOEXCEPT
    {
        if (pos < sz)
            ++pos;
        else
            pos = sz;
        for (const char* ps = p + pos; ps != p;)
            if (!traits_type::eq(*--ps, c))
                return static_cast<size_type>(ps - p);
        return npos;
    }

    const char* data_;
    size_type size_;
};
}
FNPLATFORM_NAMESPACE_EXIT

/*
This header uses code derived from the libc++ project, part of LLVM. The
license text appears below, sourced from:

- https://llvm.org/svn/llvm-project/libcxx/tags/RELEASE_400/final/LICENSE.TXT
- https://llvm.org/svn/llvm-project/libcxx/tags/RELEASE_400/final/CREDITS.TXT

==============================================================================
libc++ License
==============================================================================

The libc++ library is dual licensed under both the University of Illinois
"BSD-Like" license and the MIT license.  As a user of this code you may choose
to use it under either license.  As a contributor, you agree to allow your code
to be used under both.

Full text of the relevant licenses is included below.

==============================================================================

University of Illinois/NCSA
Open Source License

Copyright (c) 2009-2017 by the contributors listed in CREDITS.TXT

All rights reserved.

Developed by:

    LLVM Team

    University of Illinois at Urbana-Champaign

    http://llvm.org

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal with
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimers.

    * Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimers in the
      documentation and/or other materials provided with the distribution.

    * Neither the names of the LLVM Team, University of Illinois at
      Urbana-Champaign, nor the names of its contributors may be used to
      endorse or promote products derived from this Software without specific
      prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS WITH THE
SOFTWARE.

==============================================================================

Copyright (c) 2009-2014 by the contributors listed in CREDITS.TXT

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

==============================================================================
CREDITS.TXT
==============================================================================

This file is a partial list of people who have contributed to the LLVM/libc++
project.  If you have contributed a patch or made some other contribution to
LLVM/libc++, please submit a patch to this file to add yourself, and it will be
done!

The list is sorted by surname and formatted to allow easy grepping and
beautification by scripts.  The fields are: name (N), email (E), web-address
(W), PGP key ID and fingerprint (P), description (D), and snail-mail address
(S).

N: Saleem Abdulrasool
E: compnerd@compnerd.org
D: Minor patches and Linux fixes.

N: Dan Albert
E: danalbert@google.com
D: Android support and test runner improvements.

N: Dimitry Andric
E: dimitry@andric.com
D: Visibility fixes, minor FreeBSD portability patches.

N: Holger Arnold
E: holgerar@gmail.com
D: Minor fix.

N: Ruben Van Boxem
E: vanboxem dot ruben at gmail dot com
D: Initial Windows patches.

N: David Chisnall
E: theraven at theravensnest dot org
D: FreeBSD and Solaris ports, libcxxrt support, some atomics work.

N: Marshall Clow
E: mclow.lists@gmail.com
E: marshall@idio.com
D: C++14 support, patches and bug fixes.

N: Jonathan B Coe
E: jbcoe@me.com
D: Implementation of propagate_const.

N: Eric Fiselier
E: eric@efcs.ca
D: LFTS support, patches and bug fixes.

N: Bill Fisher
E: william.w.fisher@gmail.com
D: Regex bug fixes.

N: Matthew Dempsky
E: matthew@dempsky.org
D: Minor patches and bug fixes.

N: Google Inc.
D: Copyright owner and contributor of the CityHash algorithm

N: Howard Hinnant
E: hhinnant@apple.com
D: Architect and primary author of libc++

N: Hyeon-bin Jeong
E: tuhertz@gmail.com
D: Minor patches and bug fixes.

N: Argyrios Kyrtzidis
E: kyrtzidis@apple.com
D: Bug fixes.

N: Bruce Mitchener, Jr.
E: bruce.mitchener@gmail.com
D: Emscripten-related changes.

N: Michel Morin
E: mimomorin@gmail.com
D: Minor patches to is_convertible.

N: Andrew Morrow
E: andrew.c.morrow@gmail.com
D: Minor patches and Linux fixes.

N: Michael Park
E: mcypark@gmail.com
D: Implementation of <variant>.

N: Arvid Picciani
E: aep at exys dot org
D: Minor patches and musl port.

N: Bjorn Reese
E: breese@users.sourceforge.net
D: Initial regex prototype

N: Nico Rieck
E: nico.rieck@gmail.com
D: Windows fixes

N: Jon Roelofs
E: jonathan@codesourcery.com
D: Remote testing, Newlib port, baremetal/single-threaded support.

N: Jonathan Sauer
D: Minor patches, mostly related to constexpr

N: Craig Silverstein
E: csilvers@google.com
D: Implemented Cityhash as the string hash function on 64-bit machines

N: Richard Smith
D: Minor patches.

N: Joerg Sonnenberger
E: joerg@NetBSD.org
D: NetBSD port.

N: Stephan Tolksdorf
E: st@quanttec.com
D: Minor <atomic> fix

N: Michael van der Westhuizen
E: r1mikey at gmail dot com

N: Larisse Voufo
D: Minor patches.

N: Klaas de Vries
E: klaas at klaasgaaf dot nl
D: Minor bug fix.

N: Zhang Xiongpang
E: zhangxiongpang@gmail.com
D: Minor patches and bug fixes.

N: Xing Xue
E: xingxue@ca.ibm.com
D: AIX port

N: Zhihao Yuan
E: lichray@gmail.com
D: Standard compatibility fixes.

N: Jeffrey Yasskin
E: jyasskin@gmail.com
E: jyasskin@google.com
D: Linux fixes.
*/
#endif  // KATANA_PLUGINAPI_FNPLATFORM_STRINGVIEW_H_
