#ifndef KATANA_PLUGINAPI_FNPLATFORM_ARRAYVIEW_H_
#define KATANA_PLUGINAPI_FNPLATFORM_ARRAYVIEW_H_
#include <algorithm>
#include <cassert>
#include <cstddef>
#include <iterator>
#include <stdexcept>
#include <vector>

#include "FnPlatform/internal/Portability.h"

#ifdef FNKAT_CXX11
#include <array>
#include <initializer_list>
#endif  // FNKAT_CXX11

#include "FnPlatform/internal/TypeTraits.h"
#include "FnPlatform/ns.h"

FNPLATFORM_NAMESPACE_ENTER
{
/// \class ArrayView
///
/// ArrayView - Represents a constant reference to an array (0 or more elements
/// consecutively in memory). ArrayView objects are cheap to construct and
/// cheap to copy. The implementation consists of two data members: a pointer
/// to the start of the string and a size.
///
/// This is a simplified implementation of LLVM's ArrayRef. See:
/// <https://llvm.org/svn/llvm-project/llvm/tags/RELEASE_400/final/include/llvm/ADT/ArrayRef.h>.
template <typename T>
class ArrayView
{
    template <typename From, typename To>
    struct EnableIfConvertible;

public:
    typedef T value_type;
    typedef const value_type* pointer;
    typedef const value_type* const_pointer;
    typedef const value_type& reference;
    typedef const value_type& const_reference;
    typedef const value_type* iterator;
    typedef const value_type* const_iterator;
    typedef std::size_t size_type;
    typedef std::ptrdiff_t difference_type;
    typedef std::reverse_iterator<iterator> reverse_iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    /// Construct an empty ArrayView.
    ArrayView() : data_(), size_() {}

    template <typename U>
    ArrayView(const ArrayView<U>& other,
              typename EnableIfConvertible<U, T>::type* = 0)
        : data_(other.data()), size_(other.size())
    {
    }

    /// Construct an ArrayView from a pointer and size.
    ArrayView(const T* data, size_type size)
        : data_(data), size_(size)
    {
    }

    /// Construct an ArrayView from a range.
    ArrayView(const T* begin, const T* end) : data_(begin), size_(end - begin)
    {
    }

    /// Construct an ArrayView from a std::vector.
    template <typename U, typename Allocator>
    /*implicit*/ ArrayView(const std::vector<U, Allocator>& vec,
                           typename EnableIfConvertible<U, T>::type* = 0)
        : data_(vec.data()), size_(vec.size())
    {
    }

    /// Construct an ArrayView from a C array.
    template <typename U, size_type N>
    /*implicit*/ ArrayView(U (&arr)[N],
                           typename EnableIfConvertible<U, T>::type* = 0)
        : data_(arr), size_(N)
    {
    }

#ifdef FNKAT_CXX11
    /// Construct an ArrayView from a std::array.
    template <typename U, size_t N>
    /*implicit*/ ArrayView(const std::array<U, N>& vec,
                           typename EnableIfConvertible<U, T>::type* = 0)
        : data_(vec.data()), size_(vec.size())
    {
    }

    /// Construct an ArrayView from a std::initializer_list.
    template <typename U>
    /*implicit*/ ArrayView(const std::initializer_list<U>& vec,
                           typename EnableIfConvertible<U, T>::type* = 0)
        : data_(vec.begin()), size_(vec.size())
    {
    }

    // Disallow assignment from temporaries.
    template <typename U, size_t N>
    ArrayView& operator=(std::array<U, N>&&) = delete;

    template <typename U, typename Allocator>
    ArrayView& operator=(std::vector<U, Allocator>&&) = delete;

    template <typename U>
    ArrayView& operator=(std::initializer_list<U>&&) = delete;

    template <typename U, size_type N>
    ArrayView& operator=(U (&&)[N]) = delete;
#endif  // FNKAT_CXX11

    iterator begin() const { return data_; }
    iterator end() const { return data_ + size_; }

    reverse_iterator rbegin() const { return reverse_iterator(end()); }
    reverse_iterator rend() const { return reverse_iterator(begin()); }

    bool empty() const { return size_ == 0; }
    const T* data() const { return data_; }
    size_type size() const { return size_; }

    const T& front() const
    {
        assert(!empty());
        return data_[0];
    }
    const T& back() const
    {
        assert(!empty());
        return data_[size_ - 1];
    }

    const T& operator[](size_type n) const
    {
        assert(n < size_);
        return data_[n];
    }
    const T& at(size_type i) const
    {
        if (i >= size_)
            throw std::out_of_range("ArrayView::at");
        return begin()[i];
    }

private:
    // SFINAE helper to facilitate `T*` -> `const T*` conversions.
    template <typename From, typename To>
    struct EnableIfConvertible
        : FnPlatform::internal::EnableIf<
              FnPlatform::internal::IsConvertible<const From*,
                                                  const To*>::value>
    {
    };

    /// The start of the array, in an external buffer.
    const T* data_;

    /// The number of elements.
    size_type size_;
};
}
FNPLATFORM_NAMESPACE_EXIT
#endif  // KATANA_PLUGINAPI_FNPLATFORM_ARRAYVIEW_H_
