// Copyright (c) 2012 The Foundry Visionmongers Ltd. All Rights Reserved.
#ifndef KATANA_PLUGINAPI_FNLOGGING_FNLOGGING_H_
#define KATANA_PLUGINAPI_FNLOGGING_FNLOGGING_H_
#include <sstream>
#include <string>
#include <vector>

#include "FnPlatform/internal/Portability.h"
#include "FnPluginSystem/FnPluginSystem.h"

#include "FnLogging/FnLoggingAPI.h"
#include "FnLogging/ns.h"
#include "FnLogging/suite/FnLoggingSuite.h"

FNLOGGING_NAMESPACE_ENTER
{
namespace FnLogging
{
FNLOGGING_API
FnPlugStatus setHost(FnPluginHost* host);
}  // namespace FnLogging

class FNLOGGING_API FnLog
{
public:
    explicit FnLog(const std::string& module = "");
    ~FnLog();

    void log(const std::string& message, FnLoggingSeverity severity) const;

    void debug(const std::string& message) const
    {
        log(message, kFnLoggingSeverityDebug);
    }

    void info(const std::string& message) const
    {
        log(message, kFnLoggingSeverityInfo);
    }

    void warning(const std::string& message) const
    {
        log(message, kFnLoggingSeverityWarning);
    }

    void error(const std::string& message) const
    {
        log(message, kFnLoggingSeverityError);
    }

    void critical(const std::string& message) const
    {
        log(message, kFnLoggingSeverityCritical);
    }

    bool isSeverityEnabled(FnLoggingSeverity severity) const;

    static FnPlugStatus setHost(FnPluginHost* host);
    static void setSuite(FnLoggingHostSuite_v1* suite);
    static const FnLoggingHostSuite_v1* getSuite();

private:
    // no copy/assign
    FnLog(const FnLog& rhs);
    FnLog& operator=(const FnLog& rhs);

    std::string _module;

    static const FnLoggingHostSuite_v1* _loggingSuite;
};
namespace FnLogging
{
using FNLOGGING_NAMESPACE::FnLog;  // compatibility alias
}  // namespace FnLogging
}
FNLOGGING_NAMESPACE_EXIT

// Some definitions used to set up logging in a plugin source file and
// to compose log messages

#if defined(__clang__)
#define FnLogSetup(name)                                             \
    _Pragma("clang diagnostic push");                                \
    _Pragma("clang diagnostic ignored \"-Wexit-time-destructors\""); \
    static ::FnLogging::FnLog _fnLog(name);                          \
    _Pragma("clang diagnostic pop");
#else
#define FnLogSetup(name) static ::FnLogging::FnLog _fnLog(name);
#endif  // __clang__

// The following macros build up string buffers automagically behind
// the scenes, reducing the amount of effort required to use logging
// first, a generalised internal version
#define FnLogInternal(logEvent, severity)     \
    do                                        \
    {                                         \
        std::ostringstream _log_buf;          \
        _log_buf << logEvent;                 \
        _fnLog.log(_log_buf.str(), severity); \
    } while (0);

// and now, wrappers for all the levels

#define FnLogDebug(logEvent) FnLogInternal(logEvent, kFnLoggingSeverityDebug)
#define FnLogInfo(logEvent) FnLogInternal(logEvent, kFnLoggingSeverityInfo)
#define FnLogWarn(logEvent) FnLogInternal(logEvent, kFnLoggingSeverityWarning)
#define FnLogError(logEvent) FnLogInternal(logEvent, kFnLoggingSeverityError)
#define FnLogCritical(logEvent) \
    FnLogInternal(logEvent, kFnLoggingSeverityCritical)

////////////////////////////////////////////////////////////////////////////////

FNLOGGING_NAMESPACE_ENTER
{
struct FnLogQueueEntry
{
    std::string message;
    unsigned int severity;
    std::string module;
};

// This class is a historical curiosity that has never been implemented.
class FNLOGGING_API FnLogQueue
{
public:
    FNKAT_DEPRECATED explicit FnLogQueue(unsigned int severityFilter,
                                         const char* moduleFilter = 0x0);
    FNKAT_DEPRECATED FnLogQueue();

    FNKAT_DEPRECATED ~FnLogQueue();

    FNKAT_DEPRECATED void clear();

    FNKAT_DEPRECATED const std::vector<FnLogQueueEntry>& getEntries() const
    {
        return _entries;
    }

    FNKAT_DEPRECATED std::string getEntriesAsString() const;

    FNKAT_DEPRECATED static void handler(const char* message,
                                         unsigned int severity,
                                         const char* module,
                                         void* userData);

    FNKAT_DEPRECATED static FnPlugStatus setHost(FnPluginHost* host);
    FNKAT_DEPRECATED static void setSuite(FnLoggingHostSuite_v1* suite);

private:
    // no copy/assign
    FnLogQueue(const FnLogQueue& rhs);
    FnLogQueue& operator=(const FnLogQueue& rhs);

    void init(unsigned int severityFilter, const char* moduleFilter);

    void _appendEntry(const char* message,
                      unsigned int severity,
                      const char* module);

    std::vector<FnLogQueueEntry> _entries;

    static const FnLoggingHostSuite_v1* _loggingSuite;
};

#if defined(FNGEOLIB_INTERNAL_NAMESPACE)
using FNLOGGING_NAMESPACE::FnLogging::setHost;
#else
namespace FnLogging
{
using FNLOGGING_NAMESPACE::FnLogQueueEntry;  // compatibility alias
using FNLOGGING_NAMESPACE::FnLogQueue;       // compatibility alias
}  // namespace FnLogging
#endif  // defined(FNGEOLIB_INTERNAL_NAMESPACE)
}
FNLOGGING_NAMESPACE_EXIT

#endif  // KATANA_PLUGINAPI_FNLOGGING_FNLOGGING_H_
