#ifndef FnGeolibServicesXformUtil_H
#define FnGeolibServicesXformUtil_H

#include <stdint.h>
#include <limits>
#include <string>
#include <utility>

#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnGroupBuilder.h>

#include <FnGeolibServices/FnGeolibServicesAPI.h>
#include <FnGeolibServices/ns.h>
#include <FnGeolibServices/suite/FnXFormUtilSuite.h>

#include <FnPluginSystem/FnPluginSystem.h>

FNGEOLIBSERVICES_NAMESPACE_ENTER
{
    /**
     * \defgroup FnXFormUtil XFormUtil API
     * @{
     */

    /**
     * Provides a number of matrix transform utilities.
     *
     * Matrix ordering matches the IMath convention:
     *
     * \code
     * FnAttribute::DoubleAttribute::array_type m = attr.getNearestSample(0.f);
     * Imath::M44d(m[0],  m[1],  m[2],  m[3],
     *             m[4],  m[5],  m[6],  m[7],
     *             m[8],  m[9],  m[10], m[11],
     *             m[12], m[13], m[14], m[15]);
     * \endcode
     *
     * or,
     *
     * \code
     * Imath::M44d( (double(*)[4]) m.data());
     * \endcode
     *
     */
    class FNGEOLIBSERVICES_API FnXFormUtil
    {
    public:
        /**
         * @param groupAttr the FnAttribute::GroupAttribute to calculate the
         *  transform from.
         * @param time      the sample time at which to calculate the transform.
         * @return (4x4 matrix attr, isAbsolute)
         */
        static std::pair<FnAttribute::DoubleAttribute, bool>
            CalcTransformMatrixAtTime(
                const FnAttribute::GroupAttribute & groupAttr,
                float time);

        /**
         * @note This will do xform interpolation as needed
         * @param groupAttr  FnAttribute::GroupAttribute containing the
         *  transform
         * @param time       sample time to calculate the transform at.
         * @param numSamples number of time samples in time argument.
         * @return return (4x4 matrix attr, isAbsolute)
         */
        static std::pair<FnAttribute::DoubleAttribute, bool>
            CalcTransformMatrixAtTimes(
                const FnAttribute::GroupAttribute& groupAttr,
                const float* time, int numSamples);

        /**
         * @note This will do xform interpolation as needed
         * @param groupAttr  FnAttribute::GroupAttribute containing the
         *  transform
         * @return return (4x4 matrix attr, isAbsolute)
         */
        static std::pair<FnAttribute::DoubleAttribute, bool>
            CalcTransformMatrixAtExistingTimes(
                const FnAttribute::GroupAttribute & groupAttr);

        //
        // Bounds handling
        //

        /**
         * Creates a bounds attribute with the following properties:
         *
         * - Tuple Size: 2
         * - Length: 6,
         * - Values: [xMin, xMax, yMin, yMax, zMin, zMax]
         *
         * @param  xMin x min
         * @param  xMax x max
         * @param  yMin y min
         * @param  yMax y max
         * @param  zMin z max
         * @param  zMax z max
         * @return      an FnAttribute::DoubleAttribute instance.
         */
        static FnAttribute::DoubleAttribute CreateBoundsAttr(
            double xMin, double xMax,
            double yMin, double yMax,
            double zMin, double zMax);

        /**
         * Merge the two bounds attributes.
         * If the boundsAttrs are multi-sampled, the output bounds will be
         * up-sampled to the union of incoming sample times. Bounds data at
         * sample-times not provided (but needed) will be synthesized using
         * fillInterpSample()
         *
         * @param  boundAttr1 first bounds attribute
         * @param  boundAttr2 second bounds attribute
         * @return an instance of FnAttribute::DoubleAttribute containing the
         *  merged attributes.
         */
        static FnAttribute::DoubleAttribute MergeBounds(
            const FnAttribute::DoubleAttribute & boundAttr1,
            const FnAttribute::DoubleAttribute & boundAttr2);


        /**
         * Calculates a new bounds attribute after baking in the specified
         * coordinate system.
         *
         * The input transform can either be specified as a 4 x 4
         * FnAttribute::DoubleAttribute (matrix) or as an unbaked
         * FnAttribute::GroupAttribute \c xform hierarchy. All other types
         * will result in failure an invalid FnAttribute instance being
         * returned.
         *
         * The computed bounds attribute will contain time samples at the union
         * of existing samples in the incoming \c xform and \c bound
         * attributes.
         *
         * If a user requires a non-multisampled bound attribute, this call
         * should be followed up with a call to CollapseBoundTimeSamples()
         *
         * @param xform         the coordinate system
         * @param boundAttr     the bound attribute to transform
         * @return              the new transformed bound attribute.
         */
        static FnAttribute::DoubleAttribute
            CalcTransformedBoundsAtExistingTimes(
                const FnAttribute::Attribute& xform,
                const FnAttribute::DoubleAttribute& boundAttr);

        /**
         * Creates a single sampled attribute with the merged union of the
         * \b mutlisampled input attribute.
         *
         * @param  boundAttr Multisampled bound attribute
         * @return           Single sample merged union of the input bound
         *  attribute
         */
        static FnAttribute::DoubleAttribute CollapseBoundsTimeSamples(
            const FnAttribute::DoubleAttribute & boundAttr);


        /**
         * This group of function accepting FnAttribute::GroupBuilder references
         * are used to simplify the construction of \c xform attributes.
         *
         * @note All children must be either instances of
         * FnAttribute::DoubleAttributes or FnAttribute::GroupAttribute.
         *
         * For the specified GroupBuilder, set \c group \c inherit to \c false
         * (as all xforms should be) and then do a shallow update with the
         * specified attribute, if provided.
         *
         * Note that if one's command sequence utilizes only:
         * \code
         * Init(gb, ...)
         * PushXXX(...)
         * PushXXX(...)
         * PushXXX(...)
         * etc...
         * \endcode
         *
         * It's OK to construct the group builder with the
         * GroupBuilder::BuilderModeStrict argument.
         *
         * @param gb An instance of FnAttribute::GroupBuilder
         */
        static void InitXForm(FnAttribute::GroupBuilder & gb);

        static void InitXForm(FnAttribute::GroupBuilder & gb,
            const FnAttribute::Attribute & xform);

        static void PushRotateAttr(FnAttribute::GroupBuilder & gb,
            double angle, double x, double y, double z);

        static FnAttribute::Attribute PushRotateAttr(
            const FnAttribute::Attribute & xform,
            double angle, double x, double y, double z);

        static void PushTranslateAttr(FnAttribute::GroupBuilder & gb,
            double x, double y, double z);

        static FnAttribute::Attribute PushTranslateAttr(
            const FnAttribute::Attribute & xform,
            double x, double y, double z);

        static void PushScaleAttr(
            FnAttribute::GroupBuilder & gb,
            double x, double y, double z);

        static FnAttribute::Attribute PushScaleAttr(
            const FnAttribute::Attribute & xform,
            double x, double y, double z);

        static void PushMatrixAttr(
            FnAttribute::GroupBuilder & gb,
            const double * mtx16);

        static FnAttribute::Attribute PushMatrixAttr(
            const FnAttribute::Attribute & xform,
            const double * mtx16);

        static void PushOriginAttr(FnAttribute::GroupBuilder & gb);

        static FnAttribute::Attribute PushOriginAttr(
            const FnAttribute::Attribute & xform);

    private:
        FnXFormUtil();

        static const FnXFormUtilHostSuite_v1 *_getSuite();
    };
    /** @} */
}
FNGEOLIBSERVICES_NAMESPACE_EXIT


#endif // FnGeolibServicesXformUtil_H
