// Copyright (c) 2013 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FnGeolibCookInterfaceUtilsService_H
#define FnGeolibCookInterfaceUtilsService_H

#include <stdint.h>
#include <string>

#include "FnAttribute/FnAttribute.h"
#include "FnGeolib/op/FnGeolibCookInterface.h"
#include "FnPlatform/StringView.h"
#include "FnPlatform/internal/Portability.h"
#include "FnPluginSystem/FnPluginSystem.h"

#include "FnGeolibServices/FnGeolibServicesAPI.h"
#include "FnGeolibServices/ns.h"
#include "FnGeolibServices/suite/FnGeolibCookInterfaceUtilsSuite.h"

FNGEOLIBSERVICES_NAMESPACE_ENTER
{
    /**
     * \defgroup FnGeolibCookInterfaceUtilsService \
     *     GeolibCookInterfaceUtilsService API
     * @{
     */

    class BaseGlobalAttributeProvider;
    class BaseGlobalAttributeProviderLegacy;

    /**
     * Provides a number of utility functions commonly used by Ops.
     */
    class FNGEOLIBSERVICES_API FnGeolibCookInterfaceUtils
    {
    public:
        struct MatchesCELInfo
        {
            bool matches;
            bool canMatchChildren;
        };

        /**
         * Evaluates the CEL expression against the input location path and
         * updates \c matchesCELInfo based on the results.
         *
         * @param matchesCELInfo    a instance of MatchesCELInfo
         * @param interface         the GeolibCookInterface passed to your
         *  \c cook()() function.
         * @param cel               a StringAttribute instance containing the
         *  CEL expression to be evaluated
         * @param inputLocationPath the input location path the CEL expression
         *  will be evaluated against.
         * @param inputIndex        the input index on the incoming scene to be
         *  queried (if for example attributes are reference in the CEL
         *  expression)
         */
        static void matchesCEL(MatchesCELInfo & matchesCELInfo,
                const FNGEOLIBOP_NAMESPACE::GeolibCookInterface & interface,
                const FnAttribute::StringAttribute & cel,
                const std::string & inputLocationPath=std::string(),
                int inputIndex=kFnKatGeolibDefaultInput);

        /**
         * Invoke the Default Attribute Producers (DAPs) and return the result.
         *
         * @param interface          the GeolibCookInterface passed to your
         *  \c cook()() function.
         * @param  attrRoot          the attribute from which DAPs will be
         *  invoked from, e.g. \c materials, will only invoke the DAPs for
         *  attributes beneath \c materials.
         * @param  inputLocationPath the input location path
         * @param  inputIndex        the input index to query the incoming scene
         * @param  cookOrderAttr     an optional FnAttribute::StringAttribute
         *  describing the order in which DAPs should be evaluated.
         * @return                   an FnAttribute::GroupAttribute instance
         *  containing the result of evaluating the DAPs.
         *
         * @ingroup DAPSGroup
         * @see DefaultAttributeProducer
         */
        static FnAttribute::GroupAttribute cookDaps(
                const FNGEOLIBOP_NAMESPACE::GeolibCookInterface & interface,
                const std::string & attrRoot,
                const std::string & inputLocationPath=std::string(),
                int inputIndex=kFnKatGeolibDefaultInput,
                const FnAttribute::Attribute & cookOrderAttr=FnAttribute::Attribute());

        struct BuildLocalsAndGlobalsInfo
        {
            FnAttribute::GroupAttribute locals;
            FnAttribute::GroupAttribute globals;
        };

        static BuildLocalsAndGlobalsInfo buildLocalsAndGlobals(
                const FNGEOLIBOP_NAMESPACE::GeolibCookInterface & interface,
                const std::string & attrRoot);

        static FnAttribute::Attribute getDefaultDapCookOrder();

        static FnAttribute::GroupAttribute resolveFileNamesWithinGroup(
                const FNGEOLIBOP_NAMESPACE::GeolibCookInterface & interface,
                const FnAttribute::GroupAttribute & inputGroup,
                const FnAttribute::StringAttribute & celAttr,
                int frameNumber);

        static FnAttribute::StringAttribute resolveFileNameInString(
                const FNGEOLIBOP_NAMESPACE::GeolibCookInterface & interface,
                const FnAttribute::StringAttribute & inputString,
                int frameNumber);

        static FnAttribute::Attribute getGlobalAttrGeneric(
            BaseGlobalAttributeProvider& provider,
            FnPlatform::StringView name,
            FnPlatform::StringView locationPath,
            bool* didAbortPtr = NULL);

        /**
         * This function is deprecated. Please use \c getGlobalAttrGeneric()
         * with the \c BaseGlobalAttributeProvider type for the \c provider
         * parameter instead.
         *
         * @deprecated: Please use the variant with BaseGlobalAttributeProvider
         *     instead.
         */
        static FNKAT_DEPRECATED FnAttribute::Attribute getGlobalAttrGeneric(
            BaseGlobalAttributeProviderLegacy& provider,
            const std::string& name,
            const std::string& locationPath,
            bool* didAbortPtr = NULL);

    private:
        FnGeolibCookInterfaceUtils();

        static const FnGeolibCookInterfaceUtilsHostSuite_v2 * _getSuite();
    };

    /// Base abstract class representing a generic interface to query global
    /// attributes starting from a specified scene graph location.
    ///
    /// A global attribute provider walks towards the root location, its
    /// \c getAttr() implementation returning the attribute corresponding to
    /// \c attrPath on each invocation.
    class FNGEOLIBSERVICES_API BaseGlobalAttributeProvider
    {
    public:
        virtual ~BaseGlobalAttributeProvider() {}

        virtual FnGeolibGlobalAttributeProviderStatus getAttr(
            FnPlatform::StringView attrPath,
            FnAttribute::Attribute* attributeOut) = 0;
    };

    /**
     * This class is deprecated. Please use \c BaseGlobalAttributeProvider
     * instead.
     *
     * @deprecated: Please use BaseGlobalAttributeProvider instead.
     */
    class FNGEOLIBSERVICES_API BaseGlobalAttributeProviderLegacy
    {
    public:
        virtual ~BaseGlobalAttributeProviderLegacy() {}

        FNKAT_DEPRECATED virtual FnGeolibGlobalAttributeProviderStatus
        provideAttribute(const std::string& attrPath,
                         const std::string& leafLocationPath,
                         int32_t relativeDepth,
                         FnAttribute::Attribute* attributeOut) = 0;
    };
    /** @} */
}
FNGEOLIBSERVICES_NAMESPACE_EXIT

#endif // FnGeolibServicesCookInterfaceUtilsService_H
