#ifndef FnGeolibServicesExpressionMath_H
#define FnGeolibServicesExpressionMath_H

#include <string>

#include <FnGeolibServices/FnGeolibServicesAPI.h>
#include <FnGeolibServices/ns.h>
#include <FnGeolibServices/suite/FnExpressionMathSuite.h>

#include <FnPluginSystem/FnPluginSystem.h>

FNGEOLIBSERVICES_NAMESPACE_ENTER
{
    /**
     * @defgroup FnExpressionMath ExpressionMath API
     * @{
     */

    /**
     * Class providing range common utility functions used in expression math.
     */
    class FNGEOLIBSERVICES_API FnExpressionMath
    {
    public:
        /**
         * Clamp the value between \c a and \c b. This function works for a>b
         * and a<b.
         *
         * @param  value the value to be clamped
         * @param  a     lower bound to clamp \c value to
         * @param  b     upper bound to clamp \c value to.
         * @return       \c a if \c value is less than \c a, \c b if \c value is
         *  greater than \c value else \c value.
         */
        static int    clamp(int value, int a, int b);
        static float  clamp(float value, float a, float b);
        static double clamp(double value, double a, double b);

        /**
         * Linearly interpolate between \c a and \c b using the specified mix
         * Mix values outside of [0,1] are not clamped.
         * Mix value of zero returns 'a'.
         * This argument order is such to make it similar to fit and cfit
         *
         * @return     the value between a and b, using the specified mix
         */
        static float  lerp(float mix, float a, float b);
        static double lerp(double mix, double a, double b);

        /**
         * Compute a smoothstep (ease in, ease out) version of t: [0,1]
         * This will clamp the output to between 0 and 1
         */
        static float  smoothstep(float t);
        static double smoothstep(double t);

        /**
         * Example:
         * \code
         * fit(0.8, 0, 1, 0, 255) = 204.
         * fit(3,1,4,5,20) = 15
         * \endcode
         *
         * @return a number between \c newmin and \c newmax, which is relative
         *  to \c val in the range between \c oldmin and \c oldmax.
         */
        static float  fit(float value, float oldmin, float oldmax,
                float newmin, float newmax);
        static double fit(double value, double oldmin, double oldmax,
                double newmin, double newmax);

        /**
         * Same as fit(), but clamps to new borders, works on both increasing /
         * decreasing segments
         * @see fit()
         */
        static float  cfit(float value, float oldmin, float oldmax,
                float newmin, float newmax);
        static double cfit(double value, double oldmin, double oldmax,
                double newmin, double newmax);

        /**
         * Like regular cfit(), only softer, specifically, uses SmoothStep to
         * ease in and out of the fit.
         * @see cfit() fit()
         */
        static float  softcfit(float value, float oldmin, float oldmax,
                float newmin, float newmax);
        static double softcfit(double value, double oldmin, double oldmax,
                double newmin, double newmax);

        enum RetimeHoldMode
        {
            // Hold the first/last frame of the sequence. 1111 1234 4444
            RETIME_FREEZE = kFnKatExpressionMathRetimeFreeze,

            // Repeat the sequence.                       1234 1234 1234
            RETIME_REPEAT = kFnKatExpressionMathRetimeRepeat,

            // Mirror the sequence;                       3432 1234 3212
            RETIME_MIRROR = kFnKatExpressionMathRetimeMirror
        };

        static double retime(double frame, double start, double end,
                RetimeHoldMode inMode, RetimeHoldMode outMode);

        /**
         * @return      a random value between \c [min,max]
         */
        static float  randval(float min, float max, int seed);
        static double randval(double min, double max, int seed);

        /**
         * Improved Perlin noise (Siggraph 2002)
         * Ranges from \c [0,1]
         */
        static float noise(float x);
        static float noise(float x, float y);
        static float noise(float x, float y, float z);
        static float noise(float x, float y, float z, float w);

        /**
         * Signed Improved Perlin noise (Siggraph 2002)
         * Ranges from \c [-1,1]
         * Results are just a rescaled version of noise
         */
        static float snoise(float x);
        static float snoise(float x, float y);
        static float snoise(float x, float y, float z);
        static float snoise(float x, float y, float z, float w);

        /**
         * We always return a 32-bit integer, no matter which architecture
         * we're on.
         *
         * @note Uses MD5 internally.
         */
        static int32_t stablehash(const std::string &cacheID);

    private:
        FnExpressionMath();

        static const FnExpressionMathHostSuite_v1 *_getSuite();
    };
    /** @} */
}
FNGEOLIBSERVICES_NAMESPACE_EXIT


#endif // FnGeolibServicesExpressionMath_H
