// Copyright (c) 2011 The Foundry Visionmongers Ltd. All Rights Reserved.
#ifndef INCLUDED_FNGEOLIBUTIL_PATH_H
#define INCLUDED_FNGEOLIBUTIL_PATH_H

#include <set>
#include <string>
#include <vector>

#include <FnGeolib/FnGeolibAPI.h>
#include <FnGeolib/util/ns.h>
#include <FnPlatform/StringView.h>

FNGEOLIBUTIL_NAMESPACE_ENTER
{

namespace Path
{
    /**
     * \defgroup Geolib3PathUtils Geolib3PathUtils
     *  All of these functions (which take as input an absolute scene graph
     *  location path) assume a properly normalized scene graph path:
     *      - no trailing slash '/'
     *      - no redundent internal separator (A//B, A/B/, A/./B and A/foo/../B all become A/B.)
     *
     *  Examples of normalized locations:
     *
     *  \code
     *      /root
     *      /root/world/geo
     *      /root/world/taco/a/b
     *  \endcode
     *
     * Examples of un-normalised locations:
     *
     * \code
     *      /root/
     *      /root/world/
     *      /root/world/../a
     *      /root/world/taco//a/b
     * \endcode
     * @{
     */


    FNGEOLIB_API
    std::string NormalizeAbsPath(const std::string & locationPath);

    /**
     * @return the parent scene graph location path for the given input e.g.
     *
     * \code
     *  GetLocationParent('/root/world/geo') => '/root/world'
     *  GetLocationParent('/root') => ''
     * \endcode
     *
     */
    FNGEOLIB_API
    std::string GetLocationParent(FnPlatform::StringView locationPath);

    /**
     * @return the leaf for the given input e.g.
     *
     * \code
     * GetLeafName('/a/b/c') => 'c'
     * \endcode
     *
     */
    FNGEOLIB_API
    std::string GetLeafName(FnPlatform::StringView locationPath);

    FNGEOLIB_API
    void GetLeafAndParent(std::string& parent,
                          std::string& leaf,
                          FnPlatform::StringView locationPath);

    /**
     * Example:
     *
     * \code
     * GetLocationStack('/a/b/c') => ['/a','/a/b','/a/b/c']
     * GetLocationStack('/a/b/c/d/e', root='/a/b/c') => ['/a/b/c', '/a/b/c/d', '/a/b/c/d/e']
     * GetLocationStack('/a/b/c', root='/a/b/c') => ['/a/b/c']
     * \endcode
     *
     * @param returnStack  container that will be populated with the return
     *  stack
     * @param locationPath the location path to be processed
     * @param rootPath     an optional 'root' location from which to begin.
     */
    FNGEOLIB_API
    void GetLocationStack(std::vector<std::string> & returnStack,
        const std::string & locationPath, const std::string & rootPath=std::string());

    /**
     * Example:
     * \code
     * IsAncestorOrEqual('/root/a','/root/a/b/c') => true
     * IsAncestorOrEqual('/root/a','/root/a') => true
     * \endcode
     *
     * @param  locA scene graph location A
     * @param  locB scene graph location B
     * @return true if location A is an ancestor or equal of location B
     */
    FNGEOLIB_API
    bool IsAncestorOrEqual(FnPlatform::StringView locA,
                           FnPlatform::StringView locB);

    /**
     * Example
     *
     * \code
     * IsAncestor('/root/a','/root/a/b/c') => true
     * IsAncestor('/root/a','/root/a') => false
     * \endcode
     *
     * @param  locA scene graph location A
     * @param  locB scene graph location B
     * @return true if location A is an ancestor of location B
     */
    FNGEOLIB_API
    bool IsAncestor(FnPlatform::StringView locA, FnPlatform::StringView locB);

    FNGEOLIB_API
    std::string Join(FnPlatform::StringView locA, FnPlatform::StringView locB);

    FNGEOLIB_API
    bool IsRelativePath(const std::string & path);

    /**
     * Example:
     * \code
     * RelativeToAbsPath('/root/world','../') => '/root'
     * \endcode
     * @param  rootPath the path from which to generate the new absolute path
     * @param  path     the relative path specifier.
     * @return a new absolute path given the \c rootPath and \c path.
     */
    FNGEOLIB_API
    std::string RelativeToAbsPath(const std::string & rootPath, const std::string & path);

    /**
     * Given a \c rootpath, and either a relative or absolute path, create a
     * normalized relative path.
     *
     * @note This function will throw an exception if root is not ancestor or
     * equal to path.
     *
     * Example:
     * \code
     * NormalizedRelativePath('/root','/root/world') -> 'world'
     * NormalizedRelativePath('/root','/root/world/geo') -> 'world/geo'
     * NormalizedRelativePath('/root','/root') -> ''
     * NormalizedRelativePath('/root','/notroot') -> EXCEPTION
     * NormalizedRelativePath('/root','a/b/c') -> 'a/b/c'
     * \endcode
     *
     * @param  rootpath the root path
     * @param  path     the full path from which to generate the relative path.
     * @return          a normalized relative path
     */
    FNGEOLIB_API
    std::string NormalizedRelativePath(const std::string & rootpath,
        const std::string & path);

    /**
     * Given two absolute paths, create a relative path from rootPath to path
     * even if rootPath is not an ancestor of path).
     *
     * Example:
     * \code
     * RelativePath('/root/world/geo', '/root/world/geo/a') -> 'a'
     * RelativePath('/root/world/geo/a', '/root/world/geo') -> '..'
     * RelativePath('/root/world/geo/a', '/root/world/geo/b') -> '../b'
     * RelativePath('/root/world/geo/a', '/root/world/geo/a') -> ''
     * RelativePath('/root/world/geo/a', '/root/world/cam/a') -> '../../cam/a'
     * \endcode
     *
     * @param  rootPath the root path
     * @param  path     path to generate the relative path to.
     * @return a relative path from rootPath to path
     */
    FNGEOLIB_API
    std::string RelativePath(const std::string & rootPath,
        const std::string & path);

    // both

    struct FnMatchInfo
    {
        bool match;
        bool canMatchChildren;
    };

    /**
     * @note \c testpath and pattern should be absolute scene graph paths,
     * normalized in the style of \c pystring::normpath_posix
     *
     * @param matchInfo instance of match info
     * @param testpath  scene graph path to be tested
     * @param pattern   pattern to match against.
     */
    FNGEOLIB_API
    void FnMatch(FnMatchInfo& matchInfo,
                 FnPlatform::StringView testpath,
                 FnPlatform::StringView pattern);

    /**
     * @note \c testpath and pattern should be absolute scene graph paths,
     * normalized in the style of \c pystring::normpath_posix
     *
     * @param matchInfo instance of match info
     * @param testpath  scene graph path to be tested
     * @param pattern   pattern to match against.
     */
    FNGEOLIB_API
    void FnMatchIncludingAncestors(FnMatchInfo& matchInfo,
                                   FnPlatform::StringView testpath,
                                   FnPlatform::StringView pattern);

    /**
     * @note testpath and pattern should be absolute scene graph paths,
     * normalized in the style of \c pystring::normpath_posix
     *
     * @param matchInfo instance of match info
     * @param testpath  scene graph path to be tested
     * @param pattern   pattern to match against.
     */
    FNGEOLIB_API
    void ExactMatch(FnMatchInfo& matchInfo,
                    FnPlatform::StringView testpath,
                    FnPlatform::StringView pattern);

    FNGEOLIB_API
    std::string MakeUniqueName(FnPlatform::StringView baseName,
                               const std::set<std::string>& existingNames);

    FNGEOLIB_API
    std::string MakeSafeIdentifier(FnPlatform::StringView identifier);

    /**
     * @brief Compares two scene graph location paths.
     *
     * Each path component of \c pathA is compared lexicographically with the
     * corresponding component of \c pathB. The function assumes normalized,
     * absolute paths (except for superfluous trailing slashes, which are
     * ignored).
     *
     * @param  pathA scene graph location path A
     * @param  pathB scene graph location path A
     * @return a negative value if \c pathA sorts before \c pathB, zero if both
     *  paths compare equal, and a positive value if \c pathA sorts after
     *  \c pathB.
     */
    FNGEOLIB_API
    int Compare(FnPlatform::StringView pathA, FnPlatform::StringView pathB);

    /** @} */

}  // namespace Path
}
FNGEOLIBUTIL_NAMESPACE_EXIT

#endif // INCLUDED_FNGEOLIBUTIL_PATH_H
