#ifndef FnGeolibRuntimeSuite_H
#define FnGeolibRuntimeSuite_H

#include <stdint.h>
#include <FnGeolib/suite/FnGeolibOpSuite.h>

extern "C" {

/** @brief Blind declaration of runtime handle
*/
typedef struct FnGeolibRuntimeStruct * FnGeolibRuntimeHandle;

/** @brief Blind declaration of transaction handle
*/
typedef struct FnGeolibTransactionStruct * FnGeolibTransactionHandle;

/** @brief Id for referring to runtime op instance.

An op id can stored in an attribute (see getOpInputs).
*/
typedef int32_t FnGeolibOpId;

#define kFnKatGeolibNullOpId 0
#define kFnKatUUIDStringReprLength 36ul

/** @brief Id for referring to runtime client instance.
*/
typedef int32_t FnGeolibClientId;

/** @brief Id for referring to commit IDs.
*/
typedef int32_t FnGeolibCommitId;

#define kFnKatGeolibNoOp "no-op"

typedef int64_t (*FnGeolibWriteToStream)(void *stream, const void *buf, int64_t size);
typedef int64_t (*FnGeolibReadFromStream)(void *stream, void *buf, int64_t size);

#define FnGeolibRuntimeSuite_version 2

enum {
    kFnGeolibStatus_OK = 0,
    kFnGeolibStatus_InvalidArgument,
};
typedef int FnGeolibStatus;

typedef void (*FnGeolibCompletionHandler)(FnGeolibStatus status,
                                          const void* response,
                                          int64_t length,
                                          void* context);

/// Registered plugin names of Geolib.
#define kFnKatGeoSyncRuntime "runtime-sync"
#define kFnKatGeoSyncWithProfilingRuntime "runtime-sync-prof"
#define kFnKatGeoAsyncRuntime "runtime-async"
#define kFnKatGeoAsyncWithProfilingRuntime "runtime-async-prof"
#define kFnKatGeoConcurrentRuntime "runtime-concurrent"
#define kFnKatGeoConcurrentExperimentalRuntime "runtime-concurrent-experimental"

/** @brief Runtime suite

This suite provides TODO
*/
struct FnGeolibRuntimeSuite_v2
{
    FnGeolibRuntimeHandle (*createRuntime)();
    void (*destroyRuntime)(FnGeolibRuntimeHandle handle);
    void (*registerOp)(FnGeolibRuntimeHandle handle, const char *opType,
        FnGeolibOpSuite_v1 *opSuite);

    FnGeolibTransactionHandle (*createTransaction)(
        FnGeolibRuntimeHandle handle);
    void (*destroyTransaction)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn);

    FnGeolibOpId (*createOp)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn);
    void (*retainOp)(FnGeolibRuntimeHandle handle, FnGeolibOpId op);
    void (*releaseOp)(FnGeolibRuntimeHandle handle, FnGeolibOpId op);
    void (*setOpArgs)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn,
        FnGeolibOpId op, const char *opType, FnAttributeHandle args);
    void (*setOpInputs)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn, FnGeolibOpId op,
        FnGeolibOpId *inputs, int32_t numInputs);

    void (*getOpArgs)(FnGeolibRuntimeHandle handle, FnGeolibOpId op,
        const char **opType, FnAttributeHandle *args);

    // This returns an int attr, which are PRE-INCREMENTED FnGeolibClientId(s)
    // The caller MUST release all returned clients.
    FnAttributeHandle (*getOpInputs)(FnGeolibRuntimeHandle handle,
        FnGeolibOpId op);
    uint8_t (*isValidOp)(FnGeolibRuntimeHandle handle, FnGeolibOpId op);

    FnGeolibClientId (*createClient)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn);
    void (*retainClient)(FnGeolibRuntimeHandle handle, FnGeolibClientId client);
    void (*releaseClient)(FnGeolibRuntimeHandle handle, FnGeolibClientId client);
    void (*setClientOp)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn,
        FnGeolibClientId client, FnGeolibOpId op);

    /* returns a new reference to the Op, which you are reponsible to release */
    FnGeolibOpId (*getClientOp)(FnGeolibRuntimeHandle handle, FnGeolibClientId client);

    FnGeolibCommitId (*commit)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle *transactions, int32_t numTransactions);

    FnGeolibCommitId (*getLatestCommitId)(FnGeolibRuntimeHandle handle);

    void (*setLocationsOpen)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths);
    void (*setLocationsClosed)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths);
    void (*setLocationsOpenRecursive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths,
        FnAttributeHandle stopTypes);
    void (*setLocationsClosedRecursive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths);
    void (*interruptOpenRecursive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client);
    void (*setLocationsActive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths);
    void (*setLocationsInactive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, FnAttributeHandle locationPaths);
    FnAttributeHandle (*getOpenLocations)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *rootLocationPath);
    FnAttributeHandle (*getActiveLocations)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *rootLocationPath);
    uint8_t (*isLocationOpen)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *locationPath);
    uint8_t (*isLocationActive)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *locationPath);

    void (*setAllOpen)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, uint8_t allOpen);
    uint8_t (*isAllOpen)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client);

    uint8_t (*hasLocationEvents)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client);
    FnAttributeHandle (*getLocationEvents)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, int32_t maxEvents);
    void (*setEventCachingEnabled)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, uint8_t enabled);
    uint8_t (*isEventCachingEnabled)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client);
    FnAttributeHandle (*getCachedLocation)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *locationPath);
    uint8_t (*isLocationDataCurrent)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *locationPath,
        FnGeolibCommitId commitId);

    const char *(*getRootLocationPath)(FnGeolibRuntimeHandle handle);

    // location data
    FnAttributeHandle (*cookLocation)(FnGeolibRuntimeHandle handle,
        FnGeolibClientId client, const char *locationPath);

    FnAttributeHandle (*getOptions)(FnGeolibRuntimeHandle handle);

    void (*setOptions)(FnGeolibRuntimeHandle handle,
        FnAttributeHandle options);


    // op tree serialization
    uint8_t (*serializeOps)(FnGeolibRuntimeHandle handle,
        FnGeolibOpId op,
        void *writeStream, FnGeolibWriteToStream writeFunc,
        FnAttributeHandle *errorMessage);

    /** On success, returns a new reference to the root op, which you are
         reponsible to release.  Ops are not available for use until txn
         is committed.
        On failure, returns 0 (NoOpId) */
    FnGeolibOpId (*deserializeOps)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn,
        void *readStream, FnGeolibReadFromStream readFunc,
        FnAttributeHandle *errorMessage);

    uint8_t (*isProcessing)(FnGeolibRuntimeHandle handle);

    void (*flushCaches)(FnGeolibRuntimeHandle handle);

    void (*evict)(FnGeolibRuntimeHandle handle, const char *primaryPathToKeep);

    FnAttributeHandle (*describeOp)(FnGeolibRuntimeHandle handle,
        const char* opType);

    FnAttributeHandle (*getRegisteredOpTypes)(FnGeolibRuntimeHandle handle);

    void (*_sendMessageInternal)(FnGeolibRuntimeHandle handle,
                                 const void* payload,
                                 int64_t length,
                                 void* context,
                                 FnGeolibCompletionHandler handler);

    void (*retainRuntime)(FnGeolibRuntimeHandle handle);
    void (*releaseRuntime)(FnGeolibRuntimeHandle handle);

    /// Sets the name of the Katana Node that created the specified Op.
    void (*setNodeName)(FnGeolibRuntimeHandle handle,
                        FnGeolibTransactionHandle txn,
                        FnGeolibOpId op,
                        const char* nodeName);

    /// Returns the name of the Katana Node that created this Op. Will be an
    /// empty string if it wasn't created by the node graph or if it was
    /// appended as a terminal Op.
    void (*getNodeName)(FnGeolibRuntimeHandle handle,
                        FnGeolibOpId op,
                        const char** nodeName);

    void (*finalize)(FnGeolibRuntimeHandle handle);

    /// Sets the type of the Katana Node that created the specified Op.
    void (*setNodeType)(FnGeolibRuntimeHandle handle,
                        FnGeolibTransactionHandle txn,
                        FnGeolibOpId op,
                        const char* nodeType);

    /// Returns the type of the Katana Node that created this Op.  Will be an
    /// empty string if it wasn't created by the node graph or if it was
    /// appended appended as a terminal Op.
    void (*getNodeType)(FnGeolibRuntimeHandle handle,
                        FnGeolibOpId op,
                        const char** nodeType);

    /// Sets a tag for this Op.
    void (*setTag)(FnGeolibRuntimeHandle handle,
        FnGeolibTransactionHandle txn,
        FnGeolibOpId op,
        const char* tag);

    /// Returns the tag for the specified Op
    void (*getTag)(FnGeolibRuntimeHandle handle, FnGeolibOpId op, char* tag);
};
}
#endif /* FnGeolibRuntimeSuite_H */
