// Copyright (c) 2011 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FnAttributeSuite_H
#define FnAttributeSuite_H

#include <stdint.h>

extern "C" {

/** @brief Blind declaration of an attribute handle
*/
typedef struct FnAttributeStruct * FnAttributeHandle;

/** @brief Blind declaration of a group builder handle
*/
typedef struct FnGroupBuilderStruct * FnGroupBuilderHandle;

typedef int32_t FnKatAttributeType;
#define kFnKatAttributeTypeNull 0
#define kFnKatAttributeTypeInt 1
#define kFnKatAttributeTypeFloat 2
#define kFnKatAttributeTypeDouble 3
#define kFnKatAttributeTypeString 4
#define kFnKatAttributeTypeGroup 5
#define kFnKatAttributeTypeError -1

typedef int32_t BuilderModeType;
#define kFnKatGroupBuilderModeNormal 0
#define kFnKatGroupBuilderModeStrict 1

typedef int32_t BuilderBuildMode;
#define kFnKatGroupBuilderBuildAndFlush  0
#define kFnKatGroupBuilderBuildAndRetain 1

typedef int32_t FnKatStreamType;
#define kFnKatStreamTypeBinary 0
#define kFnKatStreamTypeXML 1

struct FnAttributeHash
{
    uint64_t hash1;
    uint64_t hash2;
};

/** @brief Pointer to function for writing 'size' bytes from 'buf' to 'stream'.
Returns the number of bytes actually written.
*/
typedef int64_t (*FnAttributeWriteStreamFunc)(void *stream, const void *buf, int64_t size);

/** @brief Pointer to function for reading 'size' bytes from 'stream' into 'buf'.
Returns the number of bytes actually read.
*/
typedef int64_t (*FnAttributeReadStreamFunc)(void *stream, void *buf, int64_t size);

/** @brief Pointer to a function for freeing memory attribute has taken
 * ownership of.
 */
typedef void (*FnAttributeFreeOwnedDataFunc)(void* context);

enum
{
    /**
     * Option that permits \c createAccessor to create a heap-allocated
     * accessor if necessary, rather than using the inline storage supplied by
     * the caller.
     *
     * This is required to guarantee stability of pointers to attribute data if
     * the accessor struct is copied to another location in memory.
     */
    kFnKatAttributeSampleAccessorRefCounted = 0x1u,

    /**
     * Indicates that the data from all samples is stored in a single
     * tightly-packed array. The caller must read from the \c values member of
     * \c FnSampleAccessor.
     *
     * @note Not valid as an option to \c createAccessor().
     */
    kFnKatAttributeSampleAccessorContiguous = 0x10u,

    /**
     * Indicates that data for each sample is stored in separate buffers.
     * The caller must read sample data from the \c samples member of
     * \c FnSampleAccessor.
     *
     * @note Not valid as an option to \c createAccessor().
     */
    kFnKatAttributeSampleAccessorIndirected = 0x20u,
};
typedef uint32_t FnKatAttributeSampleAccessorOptions;

/** @brief Object that mediates access to the sample data of an FnAttribute.
 */
typedef struct FnAttributeSampleAccessorStorage
{
    /** The version of this struct. Pass 0. */
    int version;
    /** Bitset of options. */
    FnKatAttributeSampleAccessorOptions options;
    /** The number of time samples in an attribute */
    int64_t sampleCount;
    /** The number of values per time sample */
    int64_t valueCount;
    /** A pointer to a buffer of size \c sampleCount containing the time samples
     */
    const float* sampleTimes;
    union
    {
        /**
         * A pointer to a buffer of size \c sampleCount*valueCount containing
         * the sample data. Valid if \c kFnKatAttributeSampleAccessorContiguous
         * is set in \c options.
         */
        const void* values;

        /**
         * A pointer to an buffer of \c sampleCount buffers, each with
         * \c valueCount elements. Valid if
         * \c kFnKatAttributeSampleAccessorIndirected
         * is set in \c options.
         */
        const void* const* samples;
    };
    /** Opaque data that may not be modified by the caller. */
    uintptr_t opaque[3];
} FnAttributeSampleAccessor;


/** @brief Attribute suite

This suite provides functions needed by a plugin for creating and querying
attributes.

Any FnAttributeHandle instance returned from a function in this suite will
already have a single reference.  It is the responsibility of the caller to
release the attribute by calling releaseAttr().
*/
struct FnAttributeHostSuite
{
    // All Attrs
    // TODO remove streamType and auto determine it when we can
    FnAttributeHandle (*readAttributeStream)(void *stream,
        FnAttributeReadStreamFunc func,
        FnAttributeHandle *errorMessage,
        FnKatStreamType streamType);

    uint8_t (*writeAttributeStream)(FnAttributeHandle handle,
        void *stream, FnAttributeWriteStreamFunc func,
        FnAttributeHandle *errorMessage,
        FnKatStreamType streamType);

    FnKatAttributeType (*getType)(FnAttributeHandle handle);
    void (*retainAttr)(FnAttributeHandle handle);
    void (*releaseAttr)(FnAttributeHandle handle);
    uint8_t (*isEqual)(FnAttributeHandle handle, FnAttributeHandle otherHandle);
    FnAttributeHash (*getHash)(FnAttributeHandle handle);

    // DataAttrs
    int64_t (*getTupleSize)(FnAttributeHandle handle);
    int64_t (*getNumberOfValues)(FnAttributeHandle handle);
    int64_t (*getNumberOfTuples)(FnAttributeHandle handle);
    int64_t (*getNumberOfTimeSamples)(FnAttributeHandle handle);
    float (*getSampleTime)(FnAttributeHandle handle, int64_t index);
    uint8_t (*getBoundingSampleTimes)(FnAttributeHandle handle,
        float *ltime, float *rtime, float time);

    // NullAttr
    FnAttributeHandle (*createNullAttr)();

    // IntAttr
    FnAttributeHandle (*createIntAttr0)();
    FnAttributeHandle (*createIntAttr1)(int32_t value);
    FnAttributeHandle (*createIntAttr2)(const int32_t *values, int64_t valueCount,
        int64_t tupleSize);
    FnAttributeHandle (*createIntAttr3)(const float *times, int64_t timeCount,
        const int32_t **values, int64_t valueCount, int64_t tupleSize);
    const int32_t *(*getIntNearestSample)(FnAttributeHandle handle, float time,
        int64_t *valueCount);

    // FloatAttr
    FnAttributeHandle (*createFloatAttr0)();
    FnAttributeHandle (*createFloatAttr1)(float value);
    FnAttributeHandle (*createFloatAttr2)(const float *values, int64_t valueCount,
        int64_t tupleSize);
    FnAttributeHandle (*createFloatAttr3)(const float *times, int64_t timeCount,
        const float **values, int64_t valueCount, int64_t tupleSize);
    const float *(*getFloatNearestSample)(FnAttributeHandle handle, float time,
        int64_t *valueCount);

    // DoubleAttr
    FnAttributeHandle (*createDoubleAttr0)();
    FnAttributeHandle (*createDoubleAttr1)(double value);
    FnAttributeHandle (*createDoubleAttr2)(const double *values, int64_t valueCount,
        int64_t tupleSize);
    FnAttributeHandle (*createDoubleAttr3)(const float *times, int64_t timeCount,
        const double **values, int64_t valueCount, int64_t tupleSize);
    const double *(*getDoubleNearestSample)(FnAttributeHandle handle,
        float time, int64_t *valueCount);

    // StringAttr
    FnAttributeHandle (*createStringAttr0)();
    FnAttributeHandle (*createStringAttr1)(const char *value);
    FnAttributeHandle (*createStringAttr2)(const char **values, int64_t valueCount,
        int64_t tupleSize);
    FnAttributeHandle (*createStringAttr3)(const float *times, int64_t timeCount,
        const char ***values, int64_t valueCount, int64_t tupleSize);
    const char **(*getStringNearestSample)(FnAttributeHandle handle, float time,
        int64_t *valueCount);

    // GroupAttr
    FnAttributeHandle (*createGroupAttr)(const char **names,
        FnAttributeHandle *children, int64_t childCount, uint8_t groupInherit);
    int64_t (*getNumberOfChildren)(FnAttributeHandle handle);
    const char *(*getChildNameByIndex)(FnAttributeHandle, int64_t index,
        int32_t * nameSize);
    FnAttributeHandle (*getChildByIndex)(FnAttributeHandle handle,
        int64_t index);
    FnAttributeHandle (*getChildByName)(FnAttributeHandle handle,
        const char *name, int32_t stringlen);
    uint8_t (*getGroupInherit)(FnAttributeHandle handle);

    // GroupBuilder
    FnGroupBuilderHandle (*createGroupBuilder)();
    FnGroupBuilderHandle (*createGroupBuilder2)(BuilderModeType builderMode);
    void (*releaseGroupBuilder)(FnGroupBuilderHandle handle);
    void (*setGroupBuilder)(FnGroupBuilderHandle handle, const char *path,
        int32_t stringlen, FnAttributeHandle attr, uint8_t groupInherit);
    void (*setGroupBuilderUnique)(FnGroupBuilderHandle handle, const char *path,
        int32_t stringlen, FnAttributeHandle attr, uint8_t groupInherit);
    void (*delGroupBuilder)(FnGroupBuilderHandle handle, const char *path,
        int32_t stringlen);
    void (*updateGroupBuilder)(FnGroupBuilderHandle handle,
        FnAttributeHandle attr);
    void (*deepUpdateGroupBuilder)(FnGroupBuilderHandle handle,
        FnAttributeHandle attr);
    void (*reserveGroupBuilder)(FnGroupBuilderHandle handle, int64_t n);
    FnAttributeHandle (*buildGroupBuilder)(FnGroupBuilderHandle handle,
        BuilderBuildMode buildMode);
    void (*setGroupBuilderInherit)(FnGroupBuilderHandle handle,
        uint8_t groupInherit);
    void (*sortGroupBuilder)(FnGroupBuilderHandle handle);

    // Name Utils
    // Currently swaps '.' and '/' to non-ambiguous alternatives
    void (*delimiterEncode)(char * str, uint64_t len);
    void (*delimiterDecode)(char * str, uint64_t len);

    // GetValue
    uint8_t (*getIntValue)(FnAttributeHandle* phandle, float time,
        int64_t index, int32_t* valueOut);
    uint8_t (*getFloatValue)(FnAttributeHandle* phandle, float time,
        int64_t index, float* valueOut);
    uint8_t (*getDoubleValue)(FnAttributeHandle* phandle, float time,
        int64_t index, double* valueOut);
    uint8_t (*getStringValue)(FnAttributeHandle* phandle, float time,
        int64_t index, const char** valueOut);

    /**
     * Disables reference counting for the specified FnAttribute.
     *
     * Certain attributes are pre-allocated at boot for performance reasons. For
     * these attributes (and other attributes with a similar allocation
     * strategy) reference counting is not applicable and can be disabled.
     *
     * This function must be called prior to any calls to retainAttr or
     * releaseAttr for a given attribute. This function is not thread safe and
     * is intended to be called during serial sections of code during attribute
     * initialization.
     *
     * Note: This function is not recommended for general usage.
     *
     * @param handle handle to the attribute to disable reference counting.
     */
    void (*disableRefCounting)(FnAttributeHandle handle);

    /**
     *  @return True if the specified attribute is reference counted otherwise
     *          false.
     */
    uint8_t (*isRefCounted)(FnAttributeHandle handle);

    /**
     *  Deallocate the specified handle.
     *
     *  Once this function returns the attribute pointed to by handle is no
     *  longer valid for use.
     *
     *  @param handle the handle to delete.
     */
    void (*dispose)(FnAttributeHandle handle);

    /**
     * Creates a <em>Sample Accessor</em> object to access the sample data of
     * an FnAttribute.
     *
     * @param attr handle to the attribute whose data is to be accessed.
     * @param type type of attribute whose sample data is to be accessed.
     * @param options bitset of accessor creation options.
     * @param storage pointer to caller-owned memory to be used for auxiliary
     *     storage.
     * @returns a pointer to a Sample Accessor object, or \c NULL on error.
     */
    FnAttributeSampleAccessor* (*createAccessor)(
        FnAttributeHandle attr,
        FnKatAttributeType type,
        FnKatAttributeSampleAccessorOptions options,
        struct FnAttributeSampleAccessorStorage* storage);

    /**
     * Increments the reference count for the given sample accessor.
     *
     * @note Has no effect if \a accessor is \c NULL, or if \c accessor->options
     * does not contain \c kFnKatAttributeSampleAccessorRefCounted.
     */
    void (*retainAccessor)(FnAttributeSampleAccessor* accessor);

    /**
     * Decrements the reference count for the given sample accessor.
     *
     * @note Has no effect if \a accessor is \c NULL, or if \c accessor->options
     * does not contain \c kFnKatAttributeSampleAccessorRefCounted.
     */
    void (*releaseAccessor)(FnAttributeSampleAccessor* accessor);

    /**
     *  Creates and returns instances of IntAttributes.
     *
     *  This function can be used to create IntAttributes where the caller has
     *  an array or arrays of values available.
     *
     *  If the time sample array (times) is set to FnAttributeSuite::kOne then
     *  the factory will provide a default single time sample at 0.0f and
     *  interpret values as a const int* rather than const int**.
     *
     *  If context and freeOwnedDataFunc are supplied then the factory will
     *  return a 'zero copy' attribute instance. Zero copy attributes take
     *  ownership of the data pointed to by the values array. It does not take
     *  ownership of the times or values arrays themselves, just the data
     *  pointed to by values. When the attribute is deallocated
     *  freeOwnedDataFunc will be called passing context as an argument. This
     *  function should free any memory that was allocated to store the data
     *  pointed to by the values array.
     *
     *  @param times array of floating point values speciying the sample times
     *  defined in values.
     *  @param timeCount the number of float values in in the times array.
     *  @param values an array of timeCount pointers which point to the values
     *  for each corresponding time sample.
     *  @param valueCount the number of values in each array pointed to by the
     *  pointers in the values array.
     *  @param tupleSize the number of data values per tuple.
     *  @param context if specified a pointer that will be passed to the
     *  freeOwnedDataFunc when the the attribute is deallocated.
     *  @param freeOwnedDataFunc pointer to function that will be called when
     *  the attribute is to be deallocated.
     */
    FnAttributeHandle (*createIntAttrFactory)(
        const float* times,
        int64_t timeCount,
        const int32_t** values,
        int64_t valueCount,
        int64_t tupleSize,
        void* context,
        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc);

    /**
     *  Creates and returns instances of FloatAttributes.
     *
     *  This function can be used to create FloatAttribute where the caller has
     *  an array or arrays of values available.
     *
     *  If the time sample array (times) is set to FnAttributeSuite::kOne then
     *  the factory will provide a default single time sample at 0.0f and
     *  interpret values as a const float* rather than const float**.
     *
     *  If context and freeOwnedDataFunc are supplied then the factory will
     *  return a 'zero copy' attribute instance. Zero copy attributes take
     *  ownership of the data pointed to by the values array. It does not take
     *  ownership of the times or values arrays themselves, just the data
     *  pointed to by values. When the attribute is deallocated
     *  freeOwnedDataFunc will be called passing context as an argument. This
     *  function should free any memory that was allocated to store the data
     *  pointed to by the values array.
     *
     *  @param times array of floating point values speciying the sample times
     *  defined in values.
     *  @param timeCount the number of float values in in the times array.
     *  @param values an array of timeCount pointers which point to the values
     *  for each corresponding time sample.
     *  @param valueCount the number of values in each array pointed to by the
     *  pointers in the values array.
     *  @param tupleSize the number of data values per tuple.
     *  @param context if specified a pointer that will be passed to the
     *  freeOwnedDataFunc when the the attribute is deallocated.
     *  @param freeOwnedDataFunc pointer to function that will be called when
     *  the attribute is to be deallocated.
     */
    FnAttributeHandle (*createFloatAttrFactory)(
        const float* times,
        int64_t timeCount,
        const float** values,
        int64_t valueCount,
        int64_t tupleSize,
        void* context,
        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc);

    /**
     *  Creates and returns instances of DoubleAttributes.
     *
     *  This function can be used to create DoubleAttributes where the caller
     *  has an array or arrays of values available.
     *
     *  If the time sample array (times) is set to FnAttributeSuite::kOne then
     *  the factory will provide a default single time sample at 0.0f and
     *  interpret values as a const double* rather than const double**.
     *
     *  If context and freeOwnedDataFunc are supplied then the factory will
     *  return a 'zero copy' attribute instance. Zero copy attributes take
     *  ownership of the data pointed to by the values array. It does not take
     *  ownership of the times or values arrays themselves, just the data
     *  pointed to by values. When the attribute is deallocated
     *  freeOwnedDataFunc will be called passing context as an argument. This
     *  function should free any memory that was allocated to store the data
     *  pointed to by the values array.
     *
     *  @param times array of floating point values speciying the sample times
     *  defined in values.
     *  @param timeCount the number of float values in in the times array.
     *  @param values an array of timeCount pointers which point to the values
     *  for each corresponding time sample.
     *  @param valueCount the number of values in each array pointed to by the
     *  pointers in the values array.
     *  @param tupleSize the number of data values per tuple.
     *  @param context if specified a pointer that will be passed to the
     *  freeOwnedDataFunc when the the attribute is deallocated.
     *  @param freeOwnedDataFunc pointer to function that will be called when
     *  the attribute is to be deallocated.
     */
    FnAttributeHandle (*createDoubleAttrFactory)(
        const float* times,
        int64_t timeCount,
        const double** values,
        int64_t valueCount,
        int64_t tupleSize,
        void* context,
        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc);

    /**
     *  Creates and returns instances of StringAttributes.
     *
     *  This function can be used to create StringAttributes where the caller
     *  has an array or arrays of values available.
     *
     *  If the time sample array (times) is set to FnAttributeSuite::kOne then
     *  the factory will provide a default single time sample at 0.0f and
     *  interpret values as a const char** rather than const char***.
     *
     *  If context and freeOwnedDataFunc are supplied then the factory will
     *  return a 'zero copy' attribute instance. Zero copy attributes take
     *  ownership of the data pointed to by the values array. It does not take
     *  ownership of the times or values arrays themselves, just the data
     *  pointed to by values. When the attribute is deallocated
     *  freeOwnedDataFunc will be called passing context as an argument. This
     *  function should free any memory that was allocated to store the data
     *  pointed to by the values array.
     *
     *  @param times array of floating point values speciying the sample times
     *  defined in values.
     *  @param timeCount the number of float values in in the times array.
     *  @param values an array of timeCount pointers which point to the values
     *  for each corresponding time sample.
     *  @param valueCount the number of values in each array pointed to by the
     *  pointers in the values array.
     *  @param tupleSize the number of data values per tuple.
     *  @param context if specified a pointer that will be passed to the
     *  freeOwnedDataFunc when the the attribute is deallocated.
     *  @param freeOwnedDataFunc pointer to function that will be called when
     *  the attribute is to be deallocated.
     */
    FnAttributeHandle (*createStringAttrFactory)(
        const float* times,
        int64_t timeCount,
        const char*** values,
        int64_t valueCount,
        int64_t tupleSize,
        void* context,
        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc);

    /**
     * Returns the total memory in bytes that was allocated when the attribute
     * was created.
     *
     * For \c GroupAttribute, this value includes, apart from the allocated size
     * of the group internals, also the size of all its children. If an
     * attribute is included more than once in the group, the size of said
     * attribute will be added just as many times as it appears.
     *
     * Releasing the last reference to the attribute will potentially free as
     * much data as returned by this function.
     *
     * @note Katana may optimize the storage of certain attributes by caching
     * small, frequently-used values internally. In these cases, it will not be
     * possible to release the last reference.
     *
     * @param handle Handle to the attribute.
     * @return The total allocated memory of the attribute in bytes.
     */
    uint64_t (*getSize)(FnAttributeHandle handle);

    /**
     * Returns the total memory currently allocated by all FnAttributes.
     */
    uint64_t (*getTotalSize)();
};

// Declaration of the default suite version.
#define FnAttributeHostSuite_version                      5
#define FnGetAttributeHostSuite  FnGetAttributeHostSuite_v5

}
#endif // FnAttributeSuite_H
