#ifndef KATANA_PLUGINAPI_FNATTRIBUTE_FNSAMPLEACCESSOR_H_
#define KATANA_PLUGINAPI_FNATTRIBUTE_FNSAMPLEACCESSOR_H_
#include <stdint.h>
#include <cmath>
#include <cstring>
#include <iterator>
#include <stdexcept>
#include <string>
#include <utility>
#include <vector>

#include "FnPlatform/ArrayView.h"
#include "FnPlatform/StringView.h"
#include "FnPlatform/internal/Portability.h"
#include "FnPlatform/internal/TypeTraits.h"
#include "FnPluginSystem/FnPluginSystem.h"

#include "FnAttribute/FnAttributeAPI.h"
#include "FnAttribute/FnAttributeBase.h"
#include "FnAttribute/ns.h"
#include "FnAttribute/suite/FnAttributeSuite.h"

#if FNKAT_CXX11
#include "FnPlatform/internal/IndexSequence.h"
#endif

FNATTRIBUTE_NAMESPACE_ENTER
{
/**
 * \addtogroup FnAttribute_Utility_Classes FnAttribute
 * @{
 */
template <typename ValueT>
class SampleIterator;

template <typename ValueT>
class SampleAccessorBase
{
public:
    SampleAccessorBase() : _accessor(NULL), _inlineStorage() {}

    /**
     * Constructor for an accessor that shares ownership of \c attr.
     */ explicit SampleAccessorBase(const Attribute& attr)
        : _accessor(), _inlineStorage()
    {
        _accessor = Attribute::getSuite()->createAccessor(
            attr.getHandle(), GetAttributeType<ValueT>(),
            kFnKatAttributeSampleAccessorRefCounted, &_inlineStorage);
        _attr = attr;
    }

    /**
     * Constructor for an accessor that does not take ownership of \c attr.
     */
    explicit SampleAccessorBase(const Attribute* attr)
        : _accessor(), _inlineStorage()
    {
        _accessor = Attribute::getSuite()->createAccessor(
            attr->getHandle(), GetAttributeType<ValueT>(), 0, &_inlineStorage);
    }

#ifdef FNKAT_CXX11
    /** Constructor for an accessor that takes ownership of \c attr. */
    explicit SampleAccessorBase(Attribute&& attr)
        : _accessor(), _inlineStorage()
    {
        _accessor = Attribute::getSuite()->createAccessor(
            attr.getHandle(), GetAttributeType<ValueT>(),
            kFnKatAttributeSampleAccessorRefCounted, &_inlineStorage);
        _attr = std::move(attr);
    }
#endif  // FNKAT_CXX11

    ~SampleAccessorBase() { release(); }

    SampleAccessorBase(const SampleAccessorBase& rhs)
    {
        if (rhs.usesInlineStorage())
        {
            _accessor = &_inlineStorage;
            _inlineStorage = rhs._inlineStorage;
        }
        else
        {
            _accessor = rhs._accessor;
            std::memset(&_inlineStorage, 0, sizeof(_inlineStorage));
        }
        retain();
        _attr = rhs._attr;
    }

#ifdef FNKAT_CXX11
    SampleAccessorBase(SampleAccessorBase&& rhs)
    {
        if (rhs.usesInlineStorage())
        {
            _accessor = &_inlineStorage;
            _inlineStorage = rhs._inlineStorage;
            rhs._accessor = nullptr;
        }
        else
        {
            _accessor = rhs._accessor;
            rhs._accessor = nullptr;
            std::memset(&_inlineStorage, 0, sizeof(_inlineStorage));
        }
        _attr = std::move(rhs._attr);
    }
#endif  // FNKAT_CXX11

    SampleAccessorBase& operator=(const SampleAccessorBase& rhs)
    {
        if (_accessor != rhs._accessor)
        {
            release();
            if (rhs.usesInlineStorage())
            {
                _accessor = &_inlineStorage;
                _inlineStorage = rhs._inlineStorage;
            }
            else
            {
                _accessor = rhs._accessor;
                std::memset(&_inlineStorage, 0, sizeof(_inlineStorage));
            }
            retain();
            _attr = rhs._attr;
        }
        return *this;
    }

#ifdef FNKAT_CXX11
    SampleAccessorBase& operator=(SampleAccessorBase&& rhs)
    {
        if (_accessor != rhs._accessor)
        {
            release();
            if (rhs.usesInlineStorage())
            {
                _accessor = &_inlineStorage;
                _inlineStorage = rhs._inlineStorage;
                rhs._accessor = nullptr;
            }
            else
            {
                _accessor = rhs._accessor;
                rhs._accessor = nullptr;
                std::memset(&_inlineStorage, 0, sizeof(_inlineStorage));
            }
            _attr = std::move(rhs._attr);
        }
        return *this;
    }
#endif  // FNKAT_CXX11

    /**
     * Returns \c true if this is a valid accessor -- an accessor that
     * was created from a valid FnAttribute and not via the default constructor.
     */
    bool isValid() const { return _accessor != NULL; }

    /**
     * Returns the number of samples in the attribute.
     */
    int64_t getNumberOfTimeSamples() const
    {
        return isValid() ? _accessor->sampleCount : 0;
    }

    /**
     * Returns the number of values per sample.
     */
    int64_t getNumberOfValues() const
    {
        return isValid() ? _accessor->valueCount : 0;
    }

    /**
     * Returns the number of samples in the attribute.
     */
    size_t size() const
    {
        return static_cast<size_t>(getNumberOfTimeSamples());
    }

    /**
     * Returns \c true if the attribute has no time samples (or the accessor is
     * invalid).
     */
    bool empty() const { return getNumberOfTimeSamples() == 0; }

    /** @cond FN_INTERNAL_DEV */
    /** Returns a pointer to the CAPI object backing this accessor.  */
    const FnAttributeSampleAccessor* getRawAccessor() const
    {
        return _accessor;
    }
    /** @endcond */

private:
    // Maps a DataAttribute's raw value type to its FnKatAttributeType.
    template <typename T>
    static FnKatAttributeType GetAttributeType()
    {
        return GetAttributeTypeImpl(Identity<T>());
    }

    // Used for tag-dispatching in GetAttributeTypeImpl.
    template <typename T>
    struct Identity
    {
        typedef T type;
    };

    static FnKatAttributeType GetAttributeTypeImpl(Identity<int>)
    {
        return kFnKatAttributeTypeInt;
    }
    static FnKatAttributeType GetAttributeTypeImpl(Identity<float>)
    {
        return kFnKatAttributeTypeFloat;
    }
    static FnKatAttributeType GetAttributeTypeImpl(Identity<double>)
    {
        return kFnKatAttributeTypeDouble;
    }
    static FnKatAttributeType GetAttributeTypeImpl(Identity<const char*>)
    {
        return kFnKatAttributeTypeString;
    }
    template <typename T>
    static FnKatAttributeType GetAttributeTypeImpl(Identity<T>)
    {
        return kFnKatAttributeTypeError;
    }

    bool usesInlineStorage() const { return _accessor == &_inlineStorage; }

    void retain()
    {
        if (_accessor &&
            (_accessor->options & kFnKatAttributeSampleAccessorRefCounted))
        {
            Attribute::getSuite()->retainAccessor(_accessor);
        }
    }

    void release()
    {
        if (_accessor &&
            (_accessor->options & kFnKatAttributeSampleAccessorRefCounted))
        {
            Attribute::getSuite()->releaseAccessor(_accessor);
        }
    }

    FnAttributeSampleAccessor* _accessor;
    FnAttributeSampleAccessorStorage _inlineStorage;
    // When not borrowing the caller's reference, ensures attribute data remains
    // valid for the lifetime of the accessor.
    Attribute _attr;
};

/**
 * @class SampleAccessor
 *
 * A \c SampleAccessor provides access to the sample buffers contained in a
 * \c DataAttribute. Create one by calling the \c getSamples() method of any
 * \c DataAttribute.
 *
 * \note Pointers and references returned by \c SampleAccessor are invalidated
 * when the accessor is destructed.
 *
 * C++11 Example:
 * \code
 * auto accessor = attr.getSamples();
 * for (const auto& sample : accessor)
 * {
 *     int64_t idx = sample.getSampleIndex();
 *     float t = sample.getSampleTime();
 *     const int* data = sample.data();
 *     for (int val : sample)
 *     {
 *         // ...
 *     }
 * }
 * \endcode
 *
 * C++98 Example:
 * \code
 * IntAttribute::accessor_type accessor = attr.getSamples();
 * for (IntAttribute::accessor_type::const_iterator it = accessor.begin();
 *      it != accessor.end();
 *      ++it)
 * {
 *     int64_t idx = it->getSampleIndex();
 *     float t = it->getSampleTime();
 *     const int* data = it->data();
 *
 *     for (IntAttribute::sample_type::const_iterator valIt = it->begin();
 *          valueIt != sampleIt->end();
 *          ++valueIt)
 *     {
 *         int val = *it;
 *     }
 * }
 * \endcode
 */
template <typename ValueT>
class SampleAccessor : public SampleAccessorBase<ValueT>
{
public:
    typedef SampleIterator<ValueT> iterator;
    typedef iterator const_iterator;
    typedef size_t size_type;
    typedef typename std::iterator_traits<iterator>::difference_type
        difference_type;
    typedef typename std::iterator_traits<iterator>::value_type value_type;
    typedef typename std::iterator_traits<iterator>::reference reference;
    typedef reference const_reference;

    /**
     * Constructs a SampleAccessor to access data of the given attribute. You
     * should call \c getSamples() on a \c DataAttribute instead.
     *
     * @param attr The attribute whose data is to be accessed.
     */
    explicit SampleAccessor(const Attribute& attr)
        : SampleAccessorBase<ValueT>(attr)
    {
    }

    /**
     * Constructor for an accessor that does not take ownership of \c attr.
     */
    explicit SampleAccessor(const Attribute* attr)
        : SampleAccessorBase<ValueT>(attr)
    {
    }

#ifdef FNKAT_CXX11
    /**
     * Constructor for an accessor that takes ownership of \c attr.
     */
    explicit SampleAccessor(Attribute&& attr)
        : SampleAccessorBase<ValueT>(std::move(attr))
    {
    }
#endif  // FNKAT_CXX11

    /**
     * Constructs an invalid SampleAccessor, that is, \c isValid() will return
     * \c false.
     */
    SampleAccessor() {}  // = default;

    // NOLINTNEXTLINE(runtime/explicit)
    /* implicit */ SampleAccessor(const SampleAccessorBase<ValueT>& other)
        : SampleAccessorBase<ValueT>(other)
    {
    }

    /**
     * Returns an iterator to the first sample.
     */
    const_iterator begin() const FNKAT_LVALUE_REF_QUALIFIER
    {
        return const_iterator(*this);
    }

    /**
     * Returns an iterator to the past-the-end sample.
     */
    const_iterator end() const FNKAT_LVALUE_REF_QUALIFIER
    {
        return const_iterator(*this, this->getNumberOfTimeSamples());
    }

    /**
     * Returns a reference to the first sample.
     */
    const_reference front() const FNKAT_LVALUE_REF_QUALIFIER
    {
        assert(!this->empty());
        return *begin();
    }

    /**
     * Returns a reference to the last sample.
     */
    const_reference back() const FNKAT_LVALUE_REF_QUALIFIER
    {
        assert(!this->empty());
        return *(--end());
    }

    /**
     * Returns a reference to the ith sample.
     */
    const_reference operator[](difference_type i) const
        FNKAT_LVALUE_REF_QUALIFIER
    {
        assert(0 <= i && static_cast<size_t>(i) < this->size());
        return begin()[i];
    }

    /**
     * Returns a reference to the ith sample.
     *
     * @throw std::out_of_range if \c i is not a valid index.
     */
    const_reference at(difference_type i) const FNKAT_LVALUE_REF_QUALIFIER
    {
        if (!(0 <= i && static_cast<size_t>(i) < this->size()))
        {
            throw std::out_of_range("SampleAccessor::at");
        }
        return begin()[i];
    }

    /**
     * Returns a view of the samples times present in this attribute.
     */
    FnPlatform::ArrayView<float> getSampleTimes() const
        FNKAT_LVALUE_REF_QUALIFIER
    {
        if (!this->isValid())
            return FnPlatform::ArrayView<float>();

        const FnAttributeSampleAccessor* acc = this->getRawAccessor();
        return FnPlatform::ArrayView<float>(
            acc->sampleTimes, static_cast<size_t>(acc->sampleCount));
    }

    /**
     * Returns a reference to the sample closest to \a t, or a reference to a
     * default-constructed \c Sample if the accessor contains no time samples.
     */
    const_reference getNearestSample(float t) const FNKAT_LVALUE_REF_QUALIFIER
    {
        return !this->empty() ? (*this)[this->getNearestSampleIndex(t)]
                              : const_reference();
    }

    /**
     * Returns the index of the sample closest to \a t, or \c 0 if the accessor
     * contains no time samples.
     */
    int64_t getNearestSampleIndex(float t) const
    {
        if (!this->isValid())
            return 0;

        const FnAttributeSampleAccessor* acc = this->getRawAccessor();
        FnPlatform::ArrayView<float> sampleTimes(
            acc->sampleTimes, static_cast<size_t>(acc->sampleCount));

        if (sampleTimes.size() <= 1)
            return 0;

        size_t currentIndex = 0;
        float currentDelta = std::fabs(sampleTimes[0] - t);

        for (size_t i = 1; i < sampleTimes.size(); ++i)
        {
            float localDelta = std::fabs(sampleTimes[i] - t);
            if (localDelta < currentDelta)
            {
                currentDelta = localDelta;
                currentIndex = i;
            }
        }
        return static_cast<int64_t>(currentIndex);
    }

#ifdef FNKAT_CXX11
    /// @cond FN_INTERNAL_DEV
    const_iterator begin() && = delete;
    const_iterator end() && = delete;

    const_reference front() && = delete;
    const_reference back() && = delete;

    const_reference operator[](difference_type i) && = delete;
    const_reference at(difference_type i) && = delete;

    FnPlatform::ArrayView<float> getSampleTimes() && = delete;
    const_reference getNearestSample(float t) && = delete;
    /// @endcond
#endif  // FNKAT_CXX11
};

/**
 * Reference to a single time sample.
 *
 * @note \c Sample objects do not own any data; any pointers or references
 *     obtained from a \c Sample object will be invalidated if the backing
 *     \c SampleAccessor object is destructed.
 */
template <typename ValueT>
class Sample : public FnPlatform::ArrayView<ValueT>
{
public:
    Sample(const ValueT* data,
           int64_t valueCount,
           int64_t sampleIndex,
           float time)
        : FnPlatform::ArrayView<ValueT>(data, static_cast<size_t>(valueCount)),
          _sampleIndex(sampleIndex),
          _sampleTime(time)
    {
    }

    Sample() : _sampleIndex(), _sampleTime() {}

    float getSampleTime() const { return _sampleTime; }
    int64_t getSampleIndex() const { return _sampleIndex; }
    int64_t getNumberOfValues() const
    {
        return static_cast<int64_t>(this->size());
    }

    /**
     * Convenience function that creates an object of type \a T by passing
     * the first \a N values from the sample.
     *
     * @note The C++98 implementation supports constructors that take at most
     *     16 arguments.
     *
     * @tparam T The type of object to construct.
     * @tparam N The number of values to pass to \a T's constructor.
     * @param defValue The value to return if the sample contains less than
     *     \a N values.
     * @return An object of type \a T, constructed with the first \a N
     *     values from the sample.
     */
    template <typename T, size_t N>
    T getAs(const T& defValue = T()) const
    {
        if (this->size() < N)
            return defValue;
#if FNKAT_CXX11
        using FnPlatform::internal::MakeIndexSequence;
        return getAsImpl<T>(this->data(), MakeIndexSequence<N>());
#else
        using FnPlatform::internal::IntegralConstant;
        return getAsImpl<T>(this->data(), IntegralConstant<size_t, N>());
#endif  // FNKAT_CXX11
    }

private:
#if FNKAT_CXX11
#ifdef __GNUC__
// Suppress C++11 warning about aggregate initialization requiring two sets
// of braces until C++14.
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wmissing-braces"
#endif  // __GNUC__
    template <typename T, size_t... I>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IndexSequence<I...>)
    {
        (void)p;  // Suppress unused variable warnings.
        return T{p[I]...};
    }
#ifdef __GNUC__
#pragma GCC diagnostic pop
#endif  // __GNUC__
#else
    // C++98 Path.
    template <typename T>
    static T getAsImpl(const ValueT*,
                       FnPlatform::internal::IntegralConstant<size_t, 0>)
    {
        return T();
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 1>)
    {
        return T(p[0]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 2>)
    {
        return T(p[0], p[1]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 3>)
    {
        return T(p[0], p[1], p[2]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 4>)
    {
        return T(p[0], p[1], p[2], p[3]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 5>)
    {
        return T(p[0], p[1], p[2], p[3], p[4]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 6>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 7>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 8>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 9>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 10>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 11>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 12>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10], p[11]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 13>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10], p[11], p[12]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 14>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10], p[11], p[12], p[13]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 15>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10], p[11], p[12], p[13], p[14]);
    }

    template <typename T>
    static T getAsImpl(const ValueT* p,
                       FnPlatform::internal::IntegralConstant<size_t, 16>)
    {
        return T(p[0], p[1], p[2], p[3], p[4], p[5], p[6], p[7], p[8], p[9],
                 p[10], p[11], p[12], p[13], p[14], p[15]);
    }
#endif  // FNKAT_CXX11

    int64_t _sampleIndex;
    float _sampleTime;
};

template <typename ValueT>
class SampleIterator
{
    class Pointer;

public:
    typedef std::ptrdiff_t difference_type;
    typedef Sample<ValueT> value_type;
    typedef Pointer pointer;
    typedef Sample<ValueT> reference;
    typedef std::random_access_iterator_tag iterator_category;

    SampleIterator() : _accessor(), _sampleIdx() {}

    explicit SampleIterator(const SampleAccessor<ValueT>& acc,
                            int64_t sampleIndex = 0)
        : _accessor(acc.getRawAccessor()), _sampleIdx(sampleIndex)
    {
    }

    reference operator*() const
    {
        assert((_accessor->options & kFnKatAttributeSampleAccessorContiguous) ^
               (_accessor->options & kFnKatAttributeSampleAccessorIndirected));
        assert(_sampleIdx < _accessor->sampleCount);

        const ValueT* data;
        if (_accessor->options & kFnKatAttributeSampleAccessorContiguous)
        {
            data = static_cast<const ValueT*>(_accessor->values) +
                   _sampleIdx * _accessor->valueCount;
        }
        else
        {
            data = static_cast<const ValueT*>(_accessor->samples[_sampleIdx]);
        }
        return reference(data, _accessor->valueCount, _sampleIdx,
                         _accessor->sampleTimes[_sampleIdx]);
    }

    pointer operator->() const { return pointer(**this); }

    reference operator[](difference_type n) const { return *(*this + n); }

    friend bool operator==(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) == 0;
    }
    friend bool operator!=(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) != 0;
    }
    friend bool operator<(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) < 0;
    }
    friend bool operator>(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) > 0;
    }
    friend bool operator<=(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) <= 0;
    }
    friend bool operator>=(SampleIterator lhs, SampleIterator rhs)
    {
        return operator-(lhs, rhs) >= 0;
    }

    SampleIterator& operator++() { return *this += 1; }
    SampleIterator& operator--() { return *this -= 1; }

    SampleIterator operator++(int)
    {
        SampleIterator ret(*this);
        operator++();
        return ret;
    }
    SampleIterator operator--(int)
    {
        SampleIterator ret(*this);
        operator--();
        return ret;
    }

    SampleIterator& operator+=(difference_type n) { return advance(n); }
    SampleIterator& operator-=(difference_type n) { return advance(-n); }

    friend SampleIterator operator+(SampleIterator lhs, difference_type n)
    {
        return lhs.operator+=(n);
    }
    friend SampleIterator operator+(difference_type n, SampleIterator rhs)
    {
        return rhs.operator+=(n);
    }

    friend SampleIterator operator-(SampleIterator lhs, difference_type n)
    {
        return lhs.operator-=(n);
    }

    friend difference_type operator-(SampleIterator lhs, SampleIterator rhs)
    {
        return lhs._sampleIdx - rhs._sampleIdx;
    }

private:
    // Proxy object to support operator->().
    class Pointer
    {
    public:
        explicit Pointer(const Sample<ValueT>& s) : _sample(s) {}
        const Sample<ValueT>* operator->() const { return &_sample; }

    private:
        Sample<ValueT> _sample;
    };

    // Advances the iterator by \a n samples.
    SampleIterator& advance(difference_type n)
    {
        _sampleIdx += n;
        assert(_sampleIdx <= _accessor->sampleCount);
        return *this;
    }

    const FnAttributeSampleAccessor* _accessor;
    int64_t _sampleIdx;  // index of the current sample
};
/**
 * @}
 */
}
FNATTRIBUTE_NAMESPACE_EXIT

#endif  // KATANA_PLUGINAPI_FNATTRIBUTE_FNSAMPLEACCESSOR_H_
