// Copyright (c) 2011 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FoundryKatanaGroupBuilder_H
#define FoundryKatanaGroupBuilder_H

#include <FnAttribute/FnAttribute.h>
#include <FnAttribute/FnAttributeAPI.h>
#include <FnAttribute/ns.h>
#include <FnPlatform/StringView.h>

FNATTRIBUTE_NAMESPACE_ENTER
{

    /**
     * @brief A factory class for constructing \c GroupAttribute objects.
     *
     * Typical usage involves creating a \c GroupBuilder, adding attributes to
     * it with the \c set() method, and, when finished, retrieving a newly
     * constructed \c GroupAttribute using the \c GroupBuilder's \c build()
     * method.
     *
     * \warning There is currently no way to inspect the contents of the
     * builder, other than by calling \c build() and inspecting the generated
     * \c GroupAttribute. Note that by default \c build() clears the contents
     * of the builder; to override this behaviour pass \c
     * GroupBuilder::BuildAndRetain to \c build().
     *
     * As a convenience, \c GroupBuilder has support for creating arbitrarily
     * nested \c GroupAttribute structures by passing a dot-delimited string to
     * \c set(), which is referred to as a "path". Note that this implies that
     * the '.' character is <em>not</em> a valid attribute name!
     *
     * Example:
     * \code
     * GroupBuilder gb;
     * gb.set("my.nested.attribute", IntAttribute(2));
     * gb.set("myTopLevelAttribute", StringAttribute("taco"));
     * gb.set("myOtherTopLevelAttribute", FloatAttribute(4.0f));
     *
     * GroupAttribute groupAttribute = gb.build();
     *
     * // Following the call to build(), |gb| is empty and |groupAttribute| has
     * // the following structure:
     * //
     * // {
     * //   "my": {
     * //     "nested": {
     * //       "attribute": IntAttribute(2)
     * //      }
     * //   },
     * //   "myTopLevelAttribute": StringAttribute("taco"),
     * //   "myOtherTopLevelAttribute": FloatAttribute(4.0f)
     * // }
     * \endcode
     */
    class FNATTRIBUTE_API GroupBuilder
    {
    public:
        /**
         * Creates a new, empty \c GroupBuilder object.
         *
         * \remark The builder is created with \c
         *     GroupBuilder::BuilderModeNormal.
         */
        GroupBuilder() : _handle(0) {}

        /**
         * An enumeration of available modes in which \c GroupBuilder objects
         * can build their resulting \c GroupAttribute objects.
         *
         * The builder mode is set when constructing a \c GroupBuilder object.
         */
        enum BuilderMode
        {
            /**
             * The "normal" build mode, which allows the full suite of
             * \c GroupBuilder functionality. This is the default.
             */
            BuilderModeNormal = kFnKatGroupBuilderModeNormal,

            /**
             * An advanced option that enables a more restrictive but higher
             * performance mode of operation.
             *
             * When working in this mode:
             * - callers must not pass dot-delimited paths to the \c set()
             *   method
             * - callers must not make multiple calls to \c set() using the
             *   same path
             * - deletions are disallowed: the \c del() method becomes a no-op.
             */
            BuilderModeStrict = kFnKatGroupBuilderModeStrict
        };

        // @cond FN_INTERNAL_DEV
        typedef enum BuilderMode BuilderMode;
        // @endcond

        /**
         * Creates a new, empty \c GroupBuilder that uses the specified
         * \link Foundry::Katana::GroupBuilder::BuilderMode \c \
         * BuilderMode \endlink.
         */
        GroupBuilder(BuilderMode builderMode) : _handle(getSuite()->createGroupBuilder2((BuilderModeType) builderMode)) {}

        ~GroupBuilder() {
            if (_handle != 0x0)
                getSuite()->releaseGroupBuilder(_handle);
        }

        void reset()
        {
            if (_handle != 0x0) {
                getSuite()->releaseGroupBuilder(_handle);
                _handle = NULL;
            }
        }

        bool isValid() const {
            return _handle != 0x0;
        }

        /**
         * Sets the value for the attribute identified by the given path.
         *
         * If \p path refers to an existing attribute in the builder and the
         * builder was created with \c GroupBuilder::BuilderModeNormal, the
         * attribute is replaced.
         *
         * @param path The path of the attribute whose value to set.
         * @param attr The attribute object to set for the given path.
         * @param groupInherit If \p path is a dot-delimited path, specifies
         *     the \e groupInherit flag for any new groups added by this \c
         *     set() call.
         *
         * \remark If the builder was created with \c
         * GroupBuilder::BuilderModeStrict, note the caveats (documented above)
         * for working in this mode.
         *
         * \remark The empty string ("") is not a valid path component.
         */

        GroupBuilder& set(FnPlatform::StringView path,
                          const Attribute& attr,
                          bool groupInherit = true)
        {
            return set(path.data(), static_cast<int32_t>(path.size()),
                       attr.getHandle(), groupInherit);
        }

        GroupBuilder& set(const char* path,
                          int32_t length,
                          const Attribute& attr,
                          bool groupInherit = true)
        {
            return set(path, length, attr.getHandle(), groupInherit);
        }

        GroupBuilder& set(const char* path,
                          int32_t length,
                          FnAttributeHandle attr,
                          bool groupInherit = true)
        {
            lazyCreate();
            getSuite()->setGroupBuilder(_handle, path, length, attr,
                                        static_cast<uint8_t>(groupInherit));
            return *this;
        }

        /**
         * Sets the value for an attribute identified using a unique name
         * derived from the given path.
         *
         * If no attribute exists for the given path, this is equivalent to
         * calling \c set(). Otherwise, \c setWithUniqueName() chooses a new
         * path by suffixing the given path with an integer.
         *
         * @param path The base path of the attribute whose value to set.
         * @param attr The attribute object to set for the resulting path.
         * @param groupInherit If \p path is a dot-delimited path, specifies
         *     the \e groupInherit flag for any new groups added by this \c
         *     set() call.
         */

        GroupBuilder& setWithUniqueName(FnPlatform::StringView path,
                                        const Attribute& attr,
                                        bool groupInherit = true)
        {
            lazyCreate();
            getSuite()->setGroupBuilderUnique(
                _handle, path.data(), static_cast<int32_t>(path.size()),
                attr.getHandle(), uint8_t(groupInherit));
            return (*this);
        }

        /**
         * Deletes the attribute of the builder specified with the given \p
         * path.
         *
         * \remark This has no effect if the builder was created with
         * \c GroupBuilder::BuilderModeStrict.
         */
        GroupBuilder& del(FnPlatform::StringView path)
        {
            lazyCreate();
            getSuite()->delGroupBuilder(_handle, path.data(),
                                        static_cast<int32_t>(path.size()));
            return (*this);
        }


        /**
         * Updates the contents of the builder with the attributes
         * from the given GroupAttribute.
         *
         * Any new attributes with the same names as existing attributes
         * replace the old ones. Existing attributes not matching new
         * attributes are left intact. (This is analogous to the Python
         * dictionary's update method.)
         *
         * If \c setGroupInherit() has not been previously called, the builder
         * will also adopt the incoming GroupAttribute's \e groupInherit flag.
         *
         * @param attr  A GroupAttribute containing new attributes to add
         */
        GroupBuilder & update(const GroupAttribute &attr)
        {
            if (attr.isValid()) {
                lazyCreate();
                getSuite()->updateGroupBuilder(_handle, attr.getHandle());
            }
            return (*this);
        }

        /**
         * Recursively updates the contents of the builder with the attributes
         * held within the provided group attribute.
         *
         * Groups are traversed until set operations can be applied at the
         * leaves which are <em>not</em> \c GroupAttribute objects themselves.
         *
         * If \c setGroupInherit() has not been previously called, the builder
         * will also adopt the incoming GroupAttribute's \e groupInherit flag.
         */
        GroupBuilder & deepUpdate(const GroupAttribute &attr)
        {
            if (attr.isValid()) {
                lazyCreate();
                getSuite()->deepUpdateGroupBuilder(_handle, attr.getHandle());
            }
            return (*this);
        }

        /**
         * Reserves space for \p n attributes.
         *
         * This is an optimisation only. Calling \c reserve() before adding
         * attributes will avoid having to reallocate internal data structures.
         *
         * @param n The number of attributes the group attribute is to have.
         */
        GroupBuilder & reserve(int64_t n)
        {
            if (n > 0) {
                lazyCreate();
                getSuite()->reserveGroupBuilder(_handle, n);
            }
            return (*this);
        }

        /**
         * Sets a special flag on the builder that determines the value
         * returned by GroupAttribute::getGroupInherit() for the top-level
         * GroupAttribute returned by \c build().
         *
         * This \p groupInherit flag is <em>sticky</em>, so once it's been set
         * -- either through an explicit call to setGroupInherit(), or
         * indirectly via a call to \c update()/\c deepUpdate() -- further
         * calls to \c setGroupInherit() will have no effect.
         */
        GroupBuilder & setGroupInherit(bool groupInherit)
        {
            if (_handle || !groupInherit) {
                lazyCreate();
                getSuite()->setGroupBuilderInherit(_handle, (uint8_t) groupInherit);
            }
            return (*this);
        }

        /**
         * Sorts the top-level attributes of the builder by name.
         *
         * \remark \c sort() uses bytewise lexicographic ordering.
         */
        GroupBuilder & sort()
        {
            if (_handle)
                getSuite()->sortGroupBuilder(_handle);
            return (*this);
        }

        /**
         * An enumeration of flags to control the behaviour of the \c build()
         * method.
         */
        enum BuilderBuildMode
        {
            /**
             * Specifies that the builder's contents are cleared following a
             * call to \c build(). This is the default.
             */
            BuildAndFlush = kFnKatGroupBuilderBuildAndFlush,

            /**
             * Specifies that the builder's contents are retained following a
             * call to \c build().
             */
            BuildAndRetain = kFnKatGroupBuilderBuildAndRetain
        };

        // @cond FN_INTERNAL_DEV
        typedef enum BuilderBuildMode BuilderBuildMode;
        // @endcond

        /**
         * Returns a newly created group attribute with the contents of the
         * builder.
         *
         * @param builderMode An optional argument specifying whether the
         * builder's contents are retained.
         *
         * @warning \c GroupAttribute supports up to 2^28 children. An invalid
         * attribute will be returned if this limit is breached.
         *
         * @warning By default, the contents of the builder are cleared
         * following a call to \c build(). To reuse the builder while retaining
         * its contents, pass \c GroupBuilder::BuildAndRetain for the optional
         * \p builderMode argument.
         */
        GroupAttribute build(BuilderBuildMode builderMode = BuildAndFlush)
        {
            if (_handle)
                return Attribute::CreateAndSteal(getSuite()->buildGroupBuilder(
                    _handle, (int32_t) builderMode));
            else
                return GroupAttribute(true);
        }

        ///@cond FN_INTERNAL_DEV

        FnGroupBuilderHandle getHandle() {
            // this function is used to hand over GroupBuilders through C APIs
            // so we need to make sure we do not return NULL
            lazyCreate();
            return _handle;
        }

        static const FnAttributeHostSuite *getSuite()
        {
            return _attrSuite;
        }

        static void setSuite(const FnAttributeHostSuite *suite);

        static FnPlugStatus setHost(FnPluginHost *host);


    private:
        // no copy/assign
        GroupBuilder(const GroupBuilder& rhs);
        GroupBuilder& operator=(const GroupBuilder& rhs);

        void lazyCreate() {
            if (_handle == 0x0)
                _handle = getSuite()->createGroupBuilder();
        }

        static const FnAttributeHostSuite *_attrSuite;

        FnGroupBuilderHandle _handle;

        //@endcond
    };
}
FNATTRIBUTE_NAMESPACE_EXIT

#endif // FoundryKatanaGroupBuilder_H
