// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.
#ifndef KATANA_PLUGINAPI_FNATTRIBUTE_FNCONSTVECTOR_H_
#define KATANA_PLUGINAPI_FNATTRIBUTE_FNCONSTVECTOR_H_

#include <stdint.h>
#include <cstddef>
#include <utility>

#include "FnAttribute/FnAttributeBase.h"
#include "FnAttribute/FnSampleAccessor.h"
#include "FnAttribute/ns.h"
#include "FnAttribute/suite/FnAttributeSuite.h"
#include "FnPlatform/internal/Portability.h"

FNATTRIBUTE_NAMESPACE_ENTER
{
    /**
     * \defgroup FnAttribute_Utility_Classes FnAttribute
     * @{
     */

    /**
     *  @brief Helper class used to wrap raw attribute data contained
     *  on the host side.
     *
     *  There are several convenience typedefs of this class for the
     *  different Attribute types, which remove the
     *  template from ConstVectorBase:
     *  <b>IntConstVectorBase</b>, <b>FloatConstVectorBase</b>,
     *  <b>DoubleConstVectorBase</b>, <b>StringConstVectorBase</b>
     */
    template<class T>
    class ConstVectorBase
    {
    public:
        /**
         * Basic type of the data in ConstVectorBase
         */
        typedef T              value_type;
        /**
         * Typedef of the sizetype
         */
        typedef std::size_t    size_type;
        typedef std::ptrdiff_t difference_type;

        /**
         * Constructor for a \c ConstVectorBase that shares ownership of
         * \c attr.
         */
        ConstVectorBase(const Attribute& attr, float t)
            : _accessor(attr), _elems(_accessor.getNearestSample(t).data())
        {
        }

#ifdef FNKAT_CXX11
        /** Constructor for a \c ConstVectorBase that takes ownership of
         * \c attr.
         */
        ConstVectorBase(Attribute&& attr, float t)
            : _accessor(std::move(attr)),
              _elems(_accessor.getNearestSample(t).data())
        {
        }
#endif  // FNKAT_CXX11

        ConstVectorBase() : _accessor(), _elems() {}

        /**
         * Returns the number of data elements
         */
        size_type size() const
        {
            return static_cast<size_type>(_accessor.getNumberOfValues());
        }

        /**
         * Returns false if the ConstVectorBase contains any data
         */
        bool empty() const { return size() == 0; }

    protected:
        SampleAccessor<T> _accessor;
        const T* _elems;
    };

    typedef ConstVectorBase<int32_t> IntConstVectorBase;
    typedef ConstVectorBase<float> FloatConstVectorBase;
    typedef ConstVectorBase<double> DoubleConstVectorBase;
    typedef ConstVectorBase<const char*> StringConstVectorBase;

    /**
     *  @brief Helper class used to wrap raw attribute data contained
     *  on the host side, with a stl::vector style interface.
     *
     *  The data is immutable so only const references are available.
     *
     *  There are several convenience typedefs of this class for the
     *  different Attribute types, which remove the template from ConstVector:
     *  <b>IntConstVector</b>, <b>FloatConstVector</b>,
     *  <b>DoubleConstVector</b>, <b>StringConstVector</b>
     */

    template <class T>
    class ConstVector : public ConstVectorBase<T>
    {
    public:
        /**
         * Constructor for a \c ConstVector that shares ownership of \c attr.
         */
        ConstVector(const Attribute& attr, float t)
            : ConstVectorBase<T>(attr, t)
        {
        }

#ifdef FNKAT_CXX11
        /** Constructor for a \c ConstVector that takes ownership of \c attr. */
        ConstVector(Attribute&& attr, float t)
            : ConstVectorBase<T>(std::move(attr), t)
        {
        }
#endif  // FNKAT_CXX11

        // NOLINTNEXTLINE(runtime/explicit)
        /* implicit */ ConstVector(const ConstVectorBase<T>& other)
            : ConstVectorBase<T>(other)
        {
        }

        ConstVector() {}

        /**
         * Basic type of the data in ConstVectorBase
         */
        typedef T value_type;
        /**
         * Const iterator for the data type
         */
        typedef const value_type* const_iterator;
        /**
         * Const reference for the data type
         */
        typedef const value_type& const_reference;
        /**
         * Typedef of the sizetype
         */
        typedef std::size_t    size_type;
        typedef std::ptrdiff_t difference_type;

        /**
         * Returns the number of data elements
         */
        size_type size() const
        {
            return static_cast<size_type>(this->_accessor.getNumberOfValues());
        }

        /**
         * Returns false if the ConstVector contains any data
         */
        bool empty() const { return size() == 0; }

        /**
         * Returns an iterator at the beginning of the data
         */
        const_iterator begin() const FNKAT_LVALUE_REF_QUALIFIER
        {
            return this->_elems;
        }

        /**
         * Returns an iterator at the end of the data
         */
        const_iterator end() const FNKAT_LVALUE_REF_QUALIFIER
        {
            return this->_elems + size();
        }

        /**
         * Individual element access read only
         */
        const_reference operator[](size_type i) const FNKAT_LVALUE_REF_QUALIFIER
        {
            assert(i < size());
            return this->_elems[i];
        }

        /**
         * Individual element access read only
         */
        const_reference at(size_type i) const FNKAT_LVALUE_REF_QUALIFIER
        {
            if (i >= size())
            {
                throw std::out_of_range("ConstVector::at");
            }
            return this->_elems[i];
        }

        /**
         * Returns a const reference to the first element
         */
        const_reference front() const FNKAT_LVALUE_REF_QUALIFIER
        {
            assert(!empty());
            return this->_elems[0];
        }

        /**
         * Returns a const reference to the last element
         */
        const_reference back() const FNKAT_LVALUE_REF_QUALIFIER
        {
            assert(!empty());
            return this->_elems[size() - 1];
        }

        /**
         * Pointer to the raw data (read-only)
         */
        const value_type* data() const FNKAT_LVALUE_REF_QUALIFIER
        {
            return this->_elems;
        }

#ifdef FNKAT_CXX11
        /// @cond FN_INTERNAL_DEV
        const_iterator begin() && = delete;
        const_iterator end() && = delete;
        const_reference operator[](size_type i) && = delete;
        const_reference at(size_type i) && = delete;
        const_reference front() && = delete;
        const_reference back() && = delete;
        const value_type* data() && = delete;
        /// @endcond
#endif  // FNKAT_CXX11
    };

    typedef ConstVector<int32_t>     IntConstVector;
    typedef ConstVector<float>       FloatConstVector;
    typedef ConstVector<double>      DoubleConstVector;
    typedef ConstVector<const char*> StringConstVector;

    /** @} */
}
FNATTRIBUTE_NAMESPACE_EXIT
#endif  // KATANA_PLUGINAPI_FNATTRIBUTE_FNCONSTVECTOR_H_
