// Copyright (c) 2017 The Foundry Visionmongers Ltd. All Rights Reserved.
#ifndef KATANA_PLUGINAPI_FNATTRIBUTE_FNATTRIBUTEBASE_H_
#define KATANA_PLUGINAPI_FNATTRIBUTE_FNATTRIBUTEBASE_H_
#include <stdint.h>
#include <cstddef>
#include <string>
#include <utility>
#include <vector>

#include "FnAttribute/FnAttributeAPI.h"
#include "FnAttribute/ns.h"
#include "FnAttribute/suite/FnAttributeSuite.h"
#include "FnPlatform/internal/Portability.h"
#include "FnPluginSystem/FnPluginSystem.h"

#if FNKAT_CXX11
#include <type_traits>
#endif

FNATTRIBUTE_NAMESPACE_ENTER
{
    /** \defgroup FnAttribute Attribute API
     *
     * @brief API that allows to manipulate Katana Attributes.
     *
     *
     * Attributes are the basic data storage classes used by Katana's scenegraph.
     * Data types are int, float, double, string, and each attribute contains a
     * map of time samples, with each sample being an array of the base data type.
     * Group attributes contain a list of named child attributes.
     *
     * This API should used in every plug-in that reads/manipulates Katana attributes.
     *
     * All attributes, once constructed, are immutable.  Attributes can be (and
     * often are) referenced in many locations.  Their lifetime is managed internally by
     * ref-counted smart pointers.
     *
     * The API provides several utility classes to build attributes in more friendly way
     *  (DataBuilder, and its attribute type specific typedefs: <b>IntBuilder</b>,
     *  <b>FloatBuilder</b>, <b>DoubleBuilder</b> and <b>StringBuilder</b>).
     *
     * \internal
     * All attributes created via the C API are allocated by the host.  The C++
     * wrappers manage references to the attributes via the retainAttr()/releaseAttr()
     * C API calls.  Copy/assignment of a C++ wrapper instance simply retains a new
     * reference to the attribute handle, so is a very lightweight operation.
     * Copy/assignment also checks the attribute type and will only accept a reference
     * to the handle if it is the proper type (copy/assignment, followed by isValid()
     * can be used for RTTI).
     * \endinternal
     */
    class Attribute;

    struct FNATTRIBUTE_API Hash
    {
        uint64_t hash1;
        uint64_t hash2;

        std::string str() const;
        uint64_t uint64() const {
            /* Per spooky hash docs,
             * It can produce 64-bit and 32-bit hash values too, at the same speed,
             * just use the bottom n bits.
             */
            return hash1;
        }

        Attribute attr() const;

        Hash(): hash1(0), hash2(0)
        { }

        Hash(uint64_t hash1_, uint64_t hash2_): hash1(hash1_), hash2(hash2_)
        { }

        Hash(FnAttributeHash h) : hash1(h.hash1), hash2(h.hash2)
        { }

        bool operator==(const Hash & rhs) const
        {
            return ((hash1 == rhs.hash1) && (hash2 == rhs.hash2));
        }

        bool operator!=(const Hash & rhs) const
        {
            return ((hash1 != rhs.hash1) || (hash2 != rhs.hash2));
        }

        bool operator<(const Hash & rhs) const
        {
            return (hash1 == rhs.hash1) ? (hash2 < rhs.hash2) : (hash1 < rhs.hash1);
        }
    };

    /**
     *  @brief The base class of all Attributes.
     */
    class FNATTRIBUTE_API Attribute
    {
    public:
        /**
         * Create empty attribute class (isValid() == false).
         */
        Attribute() : _handle()
        {
#if FNKAT_CXX11
            static_assert(std::is_standard_layout<Attribute>::value,
                          "Attribute should be standard layout");
            static_assert(offsetof(FnAttribute::Attribute, _handle) == 0,
                          "Attribute handle should be first member");
            static_assert(
                std::is_same<FnAttributeHandle, decltype(_handle)>::value,
                "Attribute handle should be first member");
#endif
        }

        ~Attribute() {
            if (_handle)
                getSuite()->releaseAttr(_handle);
        }

        /**
         * Returns true if this attribute is valid.
         */
        bool isValid() const {return _handle != 0x0;}

        /**
         * Returns the total memory in bytes that was allocated when the
         * attribute was created.
         *
         * For \c GroupAttribute, this value includes, apart from the allocated
         * size of the group internals, also the size of all its children. If an
         * attribute is included more than once in the group, the size of said
         * attribute will be added just as many times as it appears.
         *
         * Releasing the last reference to the attribute will potentially
         * free as much data as returned by this function.
         *
         * @note Katana may optimize the storage of certain attributes by
         * caching small, frequently-used values internally. In these cases, it
         * will not be possible to release the last reference.
         *
         * @return The total allocated memory of the attribute in bytes.
         */
        uint64_t getSize() const { return getSuite()->getSize(getHandle()); }

        /**
         * Returns the Type of Attribute
         */
        FnKatAttributeType getType() const {
            return getSuite()->getType(getHandle());
        }

        /**
         * Returns an xml representation of the attribute.
         */
        std::string getXML() const;

        /**
         * Returns an Attribute from an XML representation.
         */
        static Attribute parseXML(const char * xml);

         /**
         * Returns an binary representation of the attribute.
         */
        void getBinary(std::vector<char> * buffer) const;

        /**
         * Returns an Attribute from a binary representation.
         */
        static Attribute parseBinary(const char * buffer, size_t size);

        /**
         * Writes an Attribute to the stream provided.
         * Returns true for success, false for failure.
         *  @param stream Opaque pointer to stream object.
         *  @param func Function to write bytes to stream object.
         *  @param streamType How to write the stream either as XML or binary.
         */
        bool writeAttributeStream(void *stream, FnAttributeWriteStreamFunc func, FnKatStreamType streamType) const;

        /**
         * Returns an Attribute read from the stream provided.
         *  @param stream Opaque pointer to stream object.
         *  @param func Function to read bytes from stream object.
         *  @param streamType How to read the stream either as XML or binary.
         */
        static Attribute readAttributeStream(void *stream, FnAttributeReadStreamFunc func, FnKatStreamType streamType);

        Hash getHash() const {
            return Hash(getSuite()->getHash(getHandle()));
        }

        Attribute(const Attribute& rhs) : _handle(0x0)
        {
            acceptHandle(rhs);
        }

        Attribute& operator=(const Attribute& rhs)
        {
            acceptHandle(rhs);
            return *this;
        }

#if FNKAT_CXX11
        Attribute(Attribute&& rhs) : _handle(nullptr)
        {
            moveHandle(std::move(rhs));
        }

        Attribute& operator=(Attribute&& rhs)
        {
            moveHandle(std::move(rhs));
            return *this;
        }
#endif  // FNKAT_CXX11

        friend bool operator==(const Attribute & lhs, const Attribute & rhs);
        friend bool operator!=(const Attribute & lhs, const Attribute & rhs);

        ///@cond FN_INTERNAL_DEV

        const FnAttributeHandle& getHandle() const { return _handle; }

        FnAttributeHandle getRetainedHandle() const FNKAT_LVALUE_REF_QUALIFIER
        {
            if (_handle!=0x0) getSuite()->retainAttr(_handle);
            return _handle;
        }

#if FNKAT_CXX11
        FnAttributeHandle getRetainedHandle() &&
        {
            FnAttributeHandle h = _handle;
            _handle = nullptr;
            return h;
        }
#endif  // FNKAT_CXX11

        static const FnAttributeHostSuite *getSuite() {
            return _attrSuite;
        }

        static FnPlugStatus setHost(FnPluginHost *host);
        static void setSuite(const FnAttributeHostSuite *suite);

        static Attribute CreateAndSteal(FnAttributeHandle handle) {
            return Attribute(handle);
        }

        static Attribute CreateAndRetain(FnAttributeHandle handle) {
            if(handle)
                getSuite()->retainAttr(handle);
            return Attribute(handle);
        }

    protected:
        Attribute(FnAttributeHandle handle) : _handle(handle) {}

        // set internal _handle and grab a reference
        void acceptHandle(const Attribute &rhs)
        {
            getSuite()->retainAttr(rhs._handle);
            if (_handle!=0x0) getSuite()->releaseAttr(_handle);
            _handle = rhs._handle;
        }

        // steal the specified handle, and do not increment it.
        void stealHandle(const FnAttributeHandle & handle)
        {
            if (_handle!=0x0) getSuite()->releaseAttr(_handle);
            _handle = handle;
        }

        void clearHandle()
        {
            if (_handle!=0x0) getSuite()->releaseAttr(_handle);
            _handle = 0x0;
        }
        void checkAndAcceptHandle(const Attribute &rhs, FnKatAttributeType type)
        {
            if (rhs.isValid() && rhs.getType() == type)
            {
                acceptHandle(rhs);
            }
            else
            {
                clearHandle();
            }
        }
#if FNKAT_CXX11
        void moveHandle(Attribute&& rhs)
        {
            if (_handle) { getSuite()->releaseAttr(_handle); }
            _handle = rhs._handle;
            rhs._handle = nullptr;
        }
        void checkAndMoveHandle(Attribute&& rhs, FnKatAttributeType type)
        {
            if (rhs.isValid() && rhs.getType() == type)
            {
                moveHandle(std::move(rhs));
            }
            else
            {
                clearHandle();
            }
        }
#endif  // FNKAT_CXX11

    private:
        static const FnAttributeHostSuite *_attrSuite;
        FnAttributeHandle _handle;

        /// @endcond
    };

    inline bool operator==(const Attribute & lhs, const Attribute & rhs)
    {
        return (0 != lhs.getSuite()->isEqual(lhs.getHandle(), rhs.getHandle()));
    }

    inline bool operator!=(const Attribute & lhs, const Attribute & rhs)
    {
        return (!(lhs==rhs));
    }
}
FNATTRIBUTE_NAMESPACE_EXIT

#endif  // KATANA_PLUGINAPI_FNATTRIBUTE_FNATTRIBUTEBASE_H_
