// Copyright (c) 2011 The Foundry Visionmongers Ltd. All Rights Reserved.

#ifndef FoundryKatanaAttribute_H
#define FoundryKatanaAttribute_H

#include <stdint.h>
#include <cstring>
#include <stdexcept>
#include <string>
#include <utility>
#include <vector>

#include "FnPlatform/StringView.h"
#include "FnPlatform/internal/Portability.h"
#include "FnPluginSystem/FnPluginSystem.h"

#include "FnAttribute/FnAttributeAPI.h"
#include "FnAttribute/FnAttributeBase.h"
#include "FnAttribute/FnConstVector.h"
#include "FnAttribute/FnSampleAccessor.h"
#include "FnAttribute/ns.h"
#include "FnAttribute/suite/FnAttributeSuite.h"

FNATTRIBUTE_NAMESPACE_ENTER
{
    /** \addtogroup FnAttribute
     * @{
     */

    /**
     * Bootstraps the API without having to link against the internal libraries
     * that implement the Attributes host. This can be useful when implementing
     * an executable that needs to use the Attributes API via the plug-in system,
     * without having to bootstrap Geolib or link against any internal library.
     *
     * Returns true if the bootstrap succeeds.
     */
    FNATTRIBUTE_API
    bool Bootstrap(const std::string& katanaPath);

    /**
     * Initializes the API with the given Attribute Host suite.
     */
    FNATTRIBUTE_API
    void Initialize(const FnAttributeHostSuite *);

    /**
     * @}
     */

    /**
     *  @brief The base class of attributes containing data, possibly at
     * multiple samples in time.
     *
     * The data are organised in tuples of a fixed size of 1 or greater per
     * attribute.
     * DataAttribute implements common, data-agnostic functionality, such as
     * querying them number and orientation of values and time samples.
     *
     * DataAttributes enforce some limits on the amount of data they can store.
     * These are as follows:
     *
     * value count (per sample): [0, 2^63)
     * time count:               [0, 2^13)
     * tuple size:               [0, 2^16)
     *
     * An invalid attribute will be returned should you attempt to create an
     * attribute with values outside of these limits.
     */
    class FNATTRIBUTE_API DataAttribute : public Attribute
    {
    public:
        /**
         * Create empty attribute class (isValid() == false).
         */
        DataAttribute() {}

        /**
         * Return the number of data values per tuple for this attribute.
         */
        int64_t getTupleSize() const
        {
            return getSuite()->getTupleSize(getHandle());
        }
        /**
         * Return the total number of data values for this attribute. This will be equal to
         * getNumberOfTuples() * getTupleSize()
         */
        int64_t getNumberOfValues() const
        {
            return getSuite()->getNumberOfValues(getHandle());
        }
        /**
         * Return the number of tuples in this attribute.
         */
        int64_t getNumberOfTuples() const
        {
            return getSuite()->getNumberOfTuples(getHandle());
        }
        /**
         * Return the number of time samples at which data is recorded in this attribute.
         */
        int64_t getNumberOfTimeSamples() const
        {
            return getSuite()->getNumberOfTimeSamples(getHandle());
        }
        /**
         * Returns a float value containing the time at a particular index
         * for this attribute.  If the index is not valid 0.0 is returned.
         */
        float getSampleTime(int64_t index) const
        {
            return getSuite()->getSampleTime(getHandle(), index);
        }

        /**
         * For the specified sampletime, return the two bounding samples.
         * (left bounds, right bounds).
         * If the sampletime exactly falls on a single sample, ltime == rtime
         * (this is also the case at extrema, or for attrs with only a single
         * sample.
         * For attrs with 0 samples, the return value is false (signifying failure)
         * For attrs >0 samples, return value is true
         */

        bool getBoundingSampleTimes(float *ltime, float *rtime, float sampletime) const
        {
            return (0 != getSuite()->getBoundingSampleTimes(getHandle(), ltime, rtime, sampletime));
        }

        DataAttribute(const Attribute& rhs) : Attribute(0x0)
        {
            checkAndAcceptDataHandle(rhs);
        }

        DataAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptDataHandle(rhs);
            return *this;
        }

#if FNKAT_CXX11
        DataAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            Attribute(nullptr)
        {
            checkAndMoveDataHandle(std::move(rhs));
        }

        DataAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveDataHandle(std::move(rhs));
            return *this;
        }
#endif  // FNKAT_CXX11

        ///@cond FN_INTERNAL_DEV

    protected:
        DataAttribute(FnAttributeHandle handle) : Attribute(handle) {}

        /**
         * Utility that represents a single time sample array at 0.0.
         */
        static const float kZero[];

    private:
        void checkAndAcceptDataHandle(const Attribute &rhs)
        {
            if (rhs.isValid())
            {
                FnKatAttributeType type = rhs.getType();
                if (type == kFnKatAttributeTypeInt    || type == kFnKatAttributeTypeFloat ||
                    type == kFnKatAttributeTypeDouble || type == kFnKatAttributeTypeString)
                {
                    acceptHandle(rhs);
                    return;
                }
            }

            clearHandle();
        }
#if FNKAT_CXX11
        void checkAndMoveDataHandle(Attribute&& rhs)
        {
            if (rhs.isValid())
            {
                const FnKatAttributeType type = rhs.getType();
                if (type == kFnKatAttributeTypeInt    ||
                    type == kFnKatAttributeTypeFloat  ||
                    type == kFnKatAttributeTypeDouble ||
                    type == kFnKatAttributeTypeString)
                {
                    moveHandle(std::move(rhs));
                    return;
                }
            }

            clearHandle();
        }
#endif  // FNKAT_CXX11
        //@endcond
    };

    /**
     * @brief A class representing a null value.
     */
    class FNATTRIBUTE_API NullAttribute : public Attribute
    {
    public:
        NullAttribute() : Attribute(getSuite()->createNullAttr()) {}
        NullAttribute(const Attribute& rhs) : Attribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeNull);
        }

        NullAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeNull);
            return *this;
        }

#if FNKAT_CXX11
        NullAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            Attribute(nullptr)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeNull);
        }

        NullAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeNull);
            return *this;
        }
#endif  // FNKAT_CXX11

        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeNull;
        }
    private:
        NullAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /**
     * @brief A class representing a data attribute containing integers.
     */
    class FNATTRIBUTE_API IntAttribute : public DataAttribute
    {
    public:
        typedef int32_t value_type;
        typedef IntConstVectorBase internal_array_type;
        typedef IntConstVector array_type;

        typedef SampleAccessor<value_type> accessor_type;
        typedef SampleAccessorBase<value_type> internal_accessor_type;
        typedef Sample<value_type> sample_type;

        /**
         *  Create empty attribute class (isValid() == false).
         */
        IntAttribute() : DataAttribute() {}

        /**
         *  Creates an integer attribute containing one value.
         *  @param value The integer value for the new attribute
         */
        IntAttribute(value_type value) : DataAttribute(getSuite()->createIntAttr1(value)) {}

        /**
         *  Creates an integer attribute containing a number of values, grouped
         *  into tuples of a given size.
         *
         *  Specifying values for context and freeOwnedDataFunc will cause the
         *  constructor to take ownership of the memory pointed to by values. It
         *  is the responsibility of freeOwnedDataFunc to deallocate this memory
         *  when called.
         *
         *  @param values An array of integers containing the values
         *  @param valueCount  The total number of values. Must be a multiple of
         *  tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        IntAttribute(const value_type* values,
                     int64_t valueCount,
                     int64_t tupleSize,
                     void* context = NULL,
                     FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createIntAttrFactory(DataAttribute::kZero,
                                                   1,
                                                   &values,
                                                   valueCount,
                                                   tupleSize,
                                                   context,
                                                   freeOwnedDataFunc))
        {
        }

        /**
         *  Creates an integer attribute containing several timed samples, each
         * containing
         *  a number of values, grouped into tuples of a given size.
         *
         *  Specifying values for context and freeOwnedDataFunc will cause the
         *  constructor to take ownership of the memory pointed to by values. It
         *  is the responsibility of freeOwnedDataFunc to deallocate this memory
         *  when called.
         *
         *  Note: when using the 'zero copy' version of this constructor copies
         *  of the pointers in times and values will be taken, only management
         *  of the memory pointed to by values will be transferred to the
         *  attribute.
         *
         *  @param times An array of floats, giving the times of the samples.
         *  @param timeCount The number of samples for which there is data
         *  @param values An array of pointers to integer arrays containing
         * values. There must be timeCount arrays, each of which has valueCount
         * values.
         *  @param valueCount  The total number of values in each sample. Must
         * be a multiple of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        IntAttribute(const float* times,
                     int64_t timeCount,
                     const value_type** values,
                     int64_t valueCount,
                     int64_t tupleSize,
                     void* context = NULL,
                     FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(getSuite()->createIntAttrFactory(times,
                                                             timeCount,
                                                             values,
                                                             valueCount,
                                                             tupleSize,
                                                             context,
                                                             freeOwnedDataFunc))
        {
        }

        /**
         * Returns the intended type of this Attribute, without needing a valid
         * handle.
         */
        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeInt;
        }

        /**
         * Returns a \c SampleAccessor object that provides a read-only view
         * into all time samples of the Attribute.
         *
         * @note Any pointers or references obtained from the \c SampleAccessor
         *     are invalidated if the \c SampleAccessor is destructed.
         */
#if FNKAT_CXX11
        accessor_type getSamples() const & { return accessor_type(*this); }

        /// @cond FN_INTERNAL_DEV
        accessor_type getSamples() &&
        {
            return accessor_type(std::move(*this));
        }
        /// @endcond
#else
        internal_accessor_type getSamples() const
        {
            return internal_accessor_type(*this);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns a \c ConstVector object that provides a read-only view into
         * values of the sample nearest to the given time.
         *
         * @note Any pointers or references obtained from the \c ConstVector are
         *   invalidated if the \c ConstVector is destructed.
         */
#if FNKAT_CXX11
        array_type getNearestSample(float time) const &
        {
            return array_type(*this, time);
        }

        /// @cond FN_INTERNAL_DEV
        array_type getNearestSample(float time) &&
        {
            return array_type(std::move(*this), time);
        }
        /// @endcond
#else
        internal_array_type getNearestSample(float time) const
        {
            return internal_array_type(*this, time);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns first value from the time sample nearest 0.0.  This is a convenience
         * for the extremely common case of an attribute that stores a single sample
         * of a single value at time 0.0.
         * By default, throws std::runtime_error if there are no time samples or
         * no values available.
         *  @param defValue  The value to return if an error occurs and throwOnError is false.
         *  @param throwOnError  When error occurs, if true, throw std::runtime_error.  If false, return defValue.
         */
        value_type getValue(const value_type defValue = 0, bool throwOnError=true) const
        {
            value_type value;
            FnAttributeHandle handle = getHandle();
            if (getSuite()->getIntValue(&handle, 0.0f, 0, &value))
            {
                return value;
            }
            if (throwOnError)
            {
                throw std::runtime_error("Error getting int value from IntAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        /**
         * Convenience function that creates an object of type \a T by passing
         * the first \a N values from time sample nearest t=0.0.
         *
         * Example:
         * \code
         * Matrix44<int> m44i = myAttr.getValuesAs<Matrix44<int>, 16>();
         * \endcode
         *
         * @note The C++98 implementation supports constructors that take at
         *     most 16 arguments.
         *
         * @tparam T The type of object to construct.
         * @tparam N The number of values to pass to \a T's constructor.
         * @param defValue The value to return if \a throwOnError is \c false
         *     and the attribute has either no time samples, or less than \a N
         *     values per sample.
         * @param throwOnError Whether to throw an exception if the attribute
         *     has no time samples or has less than \a N values per sample.
         * @return An object of type \a T, constructed with the first \a N
         *     values from the attribute's primary time sample.
         * @throw std::runtime_error if an error occurs and \a throwOnError is
         *     \c true.
         */
        template <typename T, size_t N>
        T getValuesAs(const T& defValue = T(), bool throwOnError = true) const
        {
            accessor_type accessor(this);
            if (!accessor.empty() &&
                N <= static_cast<size_t>(accessor.getNumberOfValues()))
            {
                const sample_type& sample = accessor.getNearestSample(0.0f);
                return sample.getAs<T, N>(defValue);
            }
            else if (throwOnError)
            {
                throw std::runtime_error(
                    "Error getting int values from IntAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        IntAttribute(const Attribute& rhs) : DataAttribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeInt);
        }

        IntAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeInt);
            return *this;
        }

#if FNKAT_CXX11
        IntAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            DataAttribute()
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeInt);
        }

        IntAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeInt);
            return *this;
        }
#endif  // FNKAT_CXX11

    private:
        IntAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /**
     * @brief A class representing a data attribute containing single-precision floats.
     */
    class FNATTRIBUTE_API FloatAttribute : public DataAttribute
    {
    public:
        typedef float value_type;
        typedef FloatConstVectorBase internal_array_type;
        typedef FloatConstVector array_type;

        typedef SampleAccessor<value_type> accessor_type;
        typedef SampleAccessorBase<value_type> internal_accessor_type;
        typedef Sample<value_type> sample_type;

        /**
         *  Create empty attribute class (isValid() == false).
         */
        FloatAttribute() : DataAttribute() {}
        /**
         *  Creates a float attribute containing one value.
         *  @param value The float value for the new attribute
         */
        FloatAttribute(value_type value) : DataAttribute(getSuite()->createFloatAttr1(value)) {}
        /**
         *  Creates a float attribute containing a number of values, grouped
         *  into tuples of a given size.
         *  @param values An array of floats containing the values
         *  @param valueCount  The total number of values. Must be a multiple of
         * tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        FloatAttribute(const value_type* values,
                       int64_t valueCount,
                       int64_t tupleSize,
                       void* context = NULL,
                       FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createFloatAttrFactory(DataAttribute::kZero,
                                                     1,
                                                     &values,
                                                     valueCount,
                                                     tupleSize,
                                                     context,
                                                     freeOwnedDataFunc))
        {
        }

        /**
         *  Creates a float attribute containing several timed samples, each
         * containing
         *  a number of values, grouped into tuples of a given size.
         *
         *  Specifying values for context and freeOwnedDataFunc will cause the
         *  constructor to take ownership of the memory pointed to by values. It
         *  is the responsibility of freeOwnedDataFunc to deallocate this memory
         *  when called.
         *
         *  Note: when using the 'zero copy' version of this constructor copies
         *  of the pointers in times and values will be taken, only management
         *  of the memory pointed to by values will be transferred to the
         *  attribute.
         *
         *  @param times An array of floats, giving the times of the samples.
         *  @param timeCount The number of samples for which there is data
         *  @param values An array of pointers to float arrays containing
         * values. There must be timeCount arrays, each of which has valueCount
         * values.
         *  @param valueCount  The total number of values in each sample. Must
         * be a multiple of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        FloatAttribute(const float* times,
                       int64_t timeCount,
                       const value_type** values,
                       int64_t valueCount,
                       int64_t tupleSize,
                       void* context = NULL,
                       FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createFloatAttrFactory(times,
                                                     timeCount,
                                                     values,
                                                     valueCount,
                                                     tupleSize,
                                                     context,
                                                     freeOwnedDataFunc))
        {
        }

        /**
         * Returns the intended type of this Attribute, without needing a valid
         * handle.
         */
        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeFloat;
        }

        /**
         * Returns a \c SampleAccessor object that provides a read-only view
         * into all time samples of the Attribute.
         *
         * @note Any pointers or references obtained from the \c SampleAccessor
         *     are invalidated if the \c SampleAccessor is destructed.
         */
#if FNKAT_CXX11
        accessor_type getSamples() const & { return accessor_type(*this); }

        /// @cond FN_INTERNAL_DEV
        accessor_type getSamples() &&
        {
            return accessor_type(std::move(*this));
        }
        /// @endcond
#else
        internal_accessor_type getSamples() const
        {
            return internal_accessor_type(*this);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns a \c ConstVector object that provides a read-only view into
         * values of the sample nearest to the given time.
         *
         * @note Any pointers or references obtained from the \c ConstVector are
         *   invalidated if the \c ConstVector is destructed.
         */
#if FNKAT_CXX11
        array_type getNearestSample(float time) const &
        {
            return array_type(*this, time);
        }

        /// @cond FN_INTERNAL_DEV
        array_type getNearestSample(float time) &&
        {
            return array_type(std::move(*this), time);
        }
        /// @endcond
#else
        internal_array_type getNearestSample(float time) const
        {
            return internal_array_type(*this, time);
        }
#endif  // FNKAT_CXX11

        /**
         * Fills the interpolated values for an exact time. If the time falls
         * between two samples, the values of the samples are interpolated to
         * make the result. If you wish to interpolate all values in the
         * attr, valueCount should equal attr.getNumberOfValues()
         */
        void fillInterpSample(value_type * array, int64_t valueCount,
            float sampleTime,
            const value_type defValue = 0.f, bool throwOnError=true) const;

        /**
         * Returns first value from the time sample nearest 0.0.  This is a convenience
         * for the extremely common case of an attribute that stores a single sample
         * of a single value at time 0.0.
         * By default, throws std::runtime_error if there are no time samples or
         * no values available.
         *  @param defValue  The value to return if an error occurs and throwOnError is false.
         *  @param throwOnError  When error occurs, if true, throw std::runtime_error.  If false, return defValue.
         */
        value_type getValue(const value_type defValue = 0.f, bool throwOnError=true) const
        {
            value_type value;
            FnAttributeHandle handle = getHandle();
            if (getSuite()->getFloatValue(&handle, 0.0f, 0, &value))
            {
                return value;
            }
            if (throwOnError)
            {
                throw std::runtime_error("Error getting float value from FloatAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        /**
         * Convenience function that creates an object of type \a T by passing
         * the first \a N values from time sample nearest t=0.0.
         *
         * Example:
         * \code
         * Matrix44<float> m44f = myAttr.getValuesAs<Matrix44<float>, 16>();
         * \endcode
         *
         * @note The C++98 implementation supports constructors that take at
         *     most 16 arguments.
         *
         * @tparam T The type of object to construct.
         * @tparam N The number of values to pass to \a T's constructor.
         * @param defValue The value to return if \a throwOnError is \c false
         *     and the attribute has either no time samples, or less than \a N
         *     values per sample.
         * @param throwOnError Whether to throw an exception if the attribute
         *     has no time samples or has less than \a N values per sample.
         * @return An object of type \a T, constructed with the first \a N
         *     values from the attribute's primary time sample.
         * @throw std::runtime_error if an error occurs and \a throwOnError is
         *     \c true.
         */
        template <typename T, size_t N>
        T getValuesAs(const T& defValue = T(), bool throwOnError = true) const
        {
            accessor_type accessor(this);
            if (!accessor.empty() &&
                N <= static_cast<size_t>(accessor.getNumberOfValues()))
            {
                const sample_type& sample = accessor.getNearestSample(0.0f);
                return sample.getAs<T, N>(defValue);
            }
            else if (throwOnError)
            {
                throw std::runtime_error(
                    "Error getting float values from FloatAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        FloatAttribute(const Attribute& rhs) : DataAttribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeFloat);
        }

        FloatAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeFloat);
            return *this;
        }

#if FNKAT_CXX11
        FloatAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            DataAttribute()
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeFloat);
        }

        FloatAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeFloat);
            return *this;
        }
#endif  // FNKAT_CXX11

    private:
        FloatAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /**
     * @brief A class representing a data attribute containing double-precision floats.
     */
    class FNATTRIBUTE_API DoubleAttribute : public DataAttribute
    {
    public:
        typedef double value_type;
        typedef DoubleConstVectorBase internal_array_type;
        typedef DoubleConstVector array_type;

        typedef SampleAccessor<value_type> accessor_type;
        typedef SampleAccessorBase<value_type> internal_accessor_type;
        typedef Sample<value_type> sample_type;

        /**
         *  Create empty attribute class (isValid() == false).
         */
        DoubleAttribute() : DataAttribute() {}
        /**
         *  Creates a double attribute containing one value.
         *  @param value The double value for the new attribute
         */
        DoubleAttribute(value_type value) :DataAttribute(getSuite()->createDoubleAttr1(value)) {}

        /**
         *  Creates a double attribute containing a number of values, grouped
         *  into tuples of a given size.
         *  @param values An array of doubles containing the values
         *  @param valueCount  The total number of values. Must be a multiple of
         * tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        DoubleAttribute(const value_type* values,
                        int64_t valueCount,
                        int64_t tupleSize,
                        void* context = NULL,
                        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createDoubleAttrFactory(DataAttribute::kZero,
                                                      1,
                                                      &values,
                                                      valueCount,
                                                      tupleSize,
                                                      context,
                                                      freeOwnedDataFunc))
        {
        }

        /**
         *  Creates a double attribute containing several timed samples, each
         * containing
         *  a number of values, grouped into tuples of a given size.
         *
         *  Specifying values for context and freeOwnedDataFunc will cause the
         *  constructor to take ownership of the memory pointed to by values. It
         *  is the responsibility of freeOwnedDataFunc to deallocate this memory
         *  when called.
         *
         *  Note: when using the 'zero copy' version of this constructor copies
         *  of the pointers in times and values will be taken, only management
         *  of the memory pointed to by values will be transferred to the
         *  attribute.
         *
         *  @param times An array of floats, giving the times of the samples.
         *  @param timeCount The number of samples for which there is data
         *  @param values An array of pointers to double arrays containing
         * values. There must be timeCount arrays, each of which has valueCount
         * values.
         *  @param valueCount  The total number of values in each sample. Must
         * be a multiple of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        DoubleAttribute(const float* times,
                        int64_t timeCount,
                        const value_type** values,
                        int64_t valueCount,
                        int64_t tupleSize,
                        void* context = NULL,
                        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createDoubleAttrFactory(times,
                                                      timeCount,
                                                      values,
                                                      valueCount,
                                                      tupleSize,
                                                      context,
                                                      freeOwnedDataFunc))
        {
        }

        /**
         * Returns the intended type of this Attribute, without needing a valid
         * handle.
         */
        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeDouble;
        }

        /**
         * Returns a \c SampleAccessor object that provides a read-only view
         * into all time samples of the Attribute.
         *
         * @note Any pointers or references obtained from the \c SampleAccessor
         *     are invalidated if the \c SampleAccessor is destructed.
         */
#if FNKAT_CXX11
        accessor_type getSamples() const & { return accessor_type(*this); }

        /// @cond FN_INTERNAL_DEV
        accessor_type getSamples() &&
        {
            return accessor_type(std::move(*this));
        }
        /// @endcond
#else
        internal_accessor_type getSamples() const
        {
            return internal_accessor_type(*this);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns a \c ConstVector object that provides a read-only view into
         * values of the sample nearest to the given time.
         *
         * @note Any pointers or references obtained from the \c ConstVector are
         *   invalidated if the \c ConstVector is destructed.
         */
#if FNKAT_CXX11
        array_type getNearestSample(float time) const &
        {
            return array_type(*this, time);
        }

        /// @cond FN_INTERNAL_DEV
        array_type getNearestSample(float time) &&
        {
            return array_type(std::move(*this), time);
        }
        /// @endcond
#else
        internal_array_type getNearestSample(float time) const
        {
            return internal_array_type(*this, time);
        }
#endif  // FNKAT_CXX11

        /**
         * Fills the interpolated values for an exact time. If the time falls
         * between two samples, the values of the samples are interpolated to
         * make the result. If you wish to interpolate all values in the
         * attr, valueCount should equal attr.getNumberOfValues()
         */
        void fillInterpSample(value_type * array, int64_t valueCount,
            float sampleTime,
            const value_type defValue = 0.0, bool throwOnError=true) const;

        /**
         * Returns first value from the time sample nearest 0.0.  This is a convenience
         * for the extremely common case of an attribute that stores a single sample
         * of a single value at time 0.0.
         * By default, throws std::runtime_error if there are no time samples or
         * no values available.
         *  @param defValue  The value to return if an error occurs and throwOnError is false.
         *  @param throwOnError  When error occurs, if true, throw std::runtime_error.  If false, return defValue.
         */
        value_type getValue(const value_type defValue = 0.0, bool throwOnError=true) const
        {
            value_type value;
            FnAttributeHandle handle = getHandle();
            if (getSuite()->getDoubleValue(&handle, 0.0f, 0, &value))
            {
                return value;
            }
            if (throwOnError)
            {
                throw std::runtime_error("Error getting double value from DoubleAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        /**
         * Convenience function that creates an object of type \a T by passing
         * the first \a N values from time sample nearest t=0.0.
         *
         * Example:
         * \code
         * Matrix44<double> m44d = myAttr.getValuesAs<Matrix44<double>, 16>();
         * \endcode
         *
         * @note The C++98 implementation supports constructors that take at
         *     most 16 arguments.
         *
         * @tparam T The type of object to construct.
         * @tparam N The number of values to pass to \a T's constructor.
         * @param defValue The value to return if \a throwOnError is \c false
         *     and the attribute has either no time samples, or less than \a N
         *     values per sample.
         * @param throwOnError Whether to throw an exception if the attribute
         *     has no time samples or has less than \a N values per sample.
         * @return An object of type \a T, constructed with the first \a N
         *     values from the attribute's primary time sample.
         * @throw std::runtime_error if an error occurs and \a throwOnError is
         *     \c true.
         */
        template <typename T, size_t N>
        T getValuesAs(const T& defValue = T(), bool throwOnError = true) const
        {
            accessor_type accessor(this);
            if (!accessor.empty() &&
                N <= static_cast<size_t>(accessor.getNumberOfValues()))
            {
                const sample_type& sample = accessor.getNearestSample(0.0f);
                return sample.getAs<T, N>(defValue);
            }
            else if (throwOnError)
            {
                throw std::runtime_error(
                    "Error getting double values from DoubleAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        DoubleAttribute(const Attribute& rhs) : DataAttribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeDouble);
        }

        DoubleAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeDouble);
            return *this;
        }

#if FNKAT_CXX11
        DoubleAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            DataAttribute()
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeDouble);
        }

        DoubleAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeDouble);
            return *this;
        }
#endif  // FNKAT_CXX11

    private:
        DoubleAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /**
     * @brief A class representing a data attribute containing strings.
     */
    class FNATTRIBUTE_API StringAttribute : public DataAttribute
    {
    public:
        typedef std::string value_type;
        typedef StringConstVectorBase internal_array_type;
        typedef StringConstVector array_type;

        typedef SampleAccessor<const char*> accessor_type;
        typedef SampleAccessorBase<const char*> internal_accessor_type;
        typedef Sample<const char*> sample_type;

        /**
         *  Create empty attribute class (isValid() == false).
         */
        StringAttribute() : DataAttribute() {}

        /**
         *  Creates a string attribute containing one value.
         *  @param value The value for the new attribute, in the form of a std::string.
         */
        StringAttribute(const std::string & value) : DataAttribute(getSuite()->createStringAttr1(value.c_str())) {}

        /**
         *  Creates a string attribute containing one value.
         *  @param value The value for the new attribute, in the form of a C string.
         */
        StringAttribute(const char * value) : DataAttribute(getSuite()->createStringAttr1(value)) {}

        /**
         *  Creates a string attribute containing a number of values, grouped
         *  into tuples of a given size.
         *  @param values An array of std::strings containing the values
         *  @param valueCount  The total number of values. Must be a multiple of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         */
        StringAttribute(const value_type* values,
                        int64_t valueCount,
                        int64_t tupleSize);

        /**
         *  Creates a string attribute containing a number of values, grouped
         *  into tuples of a given size.
         *  @param stringvec  A vector of std::string containing the values
         *  @param tupleSize  The size of tuples to group values into.
         */
        StringAttribute(const std::vector<std::string> & stringvec, int64_t tupleSize=1);

        /**
         *  Creates a string attribute containing a number of values, grouped
         *  into tuples of a given size.
         *  @param values An array of C strings containing the values
         *  @param valueCount  The total number of values. Must be a multiple
         *      of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         */
        StringAttribute(const char ** values, int64_t valueCount, int64_t tupleSize) :
            DataAttribute(getSuite()->createStringAttr2(values, valueCount, tupleSize)) {}

        /**
         * Returns the intended type of this Attribute, without needing a valid
         * handle.
         */
        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeString;
        }

        /**
         *  Creates a string attribute containing several timed samples, each containing
         *  a number of values, grouped into tuples of a given size.
         *
         *  Specifying values for context and freeOwnedDataFunc will cause the
         *  constructor to take ownership of the memory pointed to by values. It
         *  is the responsibility of freeOwnedDataFunc to deallocate this memory
         *  when called.
         *
         *  Note: when using the 'zero copy' version of this constructor copies
         *  of the pointers in times and values will be taken, only management
         *  of the memory pointed to by values will be transferred to the
         *  attribute.
         *
         *  @param times An array of floats, giving the times of the samples.
         *  @param timeCount The number of samples for which there is data
         *  @param values An array of pointers to C string arrays containing values. There must be timeCount arrays, each of which has valueCount values.
         *  @param valueCount  The total number of values in each sample. Must be a multiple of tupleSize
         *  @param tupleSize  The size of tuples to group values into.
         *  @param context User supplied data passed to freeOwnedDataFunc.
         *  @param freeOwnedDataFunc Pointer to function that will be called
         *  when values' memory is to be returned to the operating system.
         */
        StringAttribute(const float* times,
                        int64_t timeCount,
                        const char*** values,
                        int64_t valueCount,
                        int64_t tupleSize,
                        void* context = NULL,
                        FnAttributeFreeOwnedDataFunc freeOwnedDataFunc = NULL)
            : DataAttribute(
                  getSuite()->createStringAttrFactory(times,
                                                      timeCount,
                                                      values,
                                                      valueCount,
                                                      tupleSize,
                                                      context,
                                                      freeOwnedDataFunc))
        {
        }

        /**
         * Returns a \c SampleAccessor object that provides a read-only view
         * into all time samples of the Attribute.
         *
         * @note Any pointers or references obtained from the \c SampleAccessor
         *     are invalidated if the \c SampleAccessor is destructed.
         */
#if FNKAT_CXX11
        accessor_type getSamples() const & { return accessor_type(*this); }

        /// @cond FN_INTERNAL_DEV
        accessor_type getSamples() &&
        {
            return accessor_type(std::move(*this));
        }
        /// @endcond
#else
        internal_accessor_type getSamples() const
        {
            return internal_accessor_type(*this);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns a \c ConstVector object that provides a read-only view into
         * values of the sample nearest to the given time.
         *
         * @note Any pointers or references obtained from the \c ConstVector are
         *   invalidated if the \c ConstVector is destructed.
         */
#if FNKAT_CXX11
        array_type getNearestSample(float time) const &
        {
            return array_type(*this, time);
        }

        /// @cond FN_INTERNAL_DEV
        array_type getNearestSample(float time) &&
        {
            return array_type(std::move(*this), time);
        }
        /// @endcond
#else
        internal_array_type getNearestSample(float time) const
        {
            return internal_array_type(*this, time);
        }
#endif  // FNKAT_CXX11

        /**
         * Returns first value from the time sample nearest 0.0.  This is a convenience
         * for the extremely common case of an attribute that stores a single sample
         * of a single value at time 0.0.
         * By default, throws std::runtime_error if there are no time samples or
         * no values available.
         *  @param defValue  The value to return if an error occurs and throwOnError is false.
         *  @param throwOnError  When error occurs, if true, throw std::runtime_error.  If false, return defValue.
         */
        value_type getValue(const value_type& defValue = value_type(),
                            bool throwOnError = true) const
        {
            if (const char* value = getValueCStr(NULL, throwOnError))
            {
                return value;
            }
            else
            {
                return defValue;
            }
        }

        /**
         * Convenience function that creates an object of type \a T by passing
         * the first \a N values from time sample nearest t=0.0.
         *
         * @note The C++98 implementation supports constructors that take at
         *     most 16 arguments.
         *
         * @tparam T The type of object to construct.
         * @tparam N The number of values to pass to \a T's constructor.
         * @param defValue The value to return if \a throwOnError is \c false
         *     and the attribute has either no time samples, or less than \a N
         *     values per sample.
         * @param throwOnError Whether to throw an exception if the attribute
         *     has no time samples or has less than \a N values per sample.
         * @return An object of type \a T, constructed with the first \a N
         *     values from the attribute's primary time sample.
         * @throw std::runtime_error if an error occurs and \a throwOnError is
         *     \c true.
         */
        template <typename T, size_t N>
        T getValuesAs(const T& defValue = T(), bool throwOnError = true) const
        {
            accessor_type accessor(this);
            if (!accessor.empty() &&
                N <= static_cast<size_t>(accessor.getNumberOfValues()))
            {
                const sample_type& sample = accessor.getNearestSample(0.0f);
                return sample.getAs<T, N>(defValue);
            }
            else if (throwOnError)
            {
                throw std::runtime_error(
                    "Error getting string values from StringAttribute.");
            }
            else
            {
                return defValue;
            }
        }

        /**
         * Returns first value from the time sample nearest 0.0.  This is a convenience
         * for the extremely common case of an attribute that stores a single sample
         * of a single value at time 0.0.
         * By default, throws std::runtime_error if there are no time samples or
         * no values available.
         *  @param defValue  The value to return if an error occurs and throwOnError is false.
         *  @param throwOnError  When error occurs, if true, throw std::runtime_error.  If false, return defValue.
         *
         *  @warning The pointer returned by this function may be invalidated if
         *      \a *this is destroyed.
         */
        const char* getValueCStr(const char* defValue = "",
                                 bool throwOnError = true) const
            FNKAT_LVALUE_REF_QUALIFIER
        {
            const char* value;
            FnAttributeHandle* handlePtr =
                const_cast<FnAttributeHandle*>(&getHandle());
            if (getSuite()->getStringValue(handlePtr, 0.0f, 0, &value))
            {
                return value;
            }
            if (throwOnError)
            {
                throw std::runtime_error("Error getting string value from StringAttribute.");
            }
            else
            {
                return defValue;
            }
        }

#if FNKAT_CXX11
        /// @cond FN_INTERNAL_DEV
        const char* getValueCStr(const char* defValue = "",
                                 bool throwOnError = true) && = delete;
        /// @endcond
#endif  // FNKAT_CXX11

        bool operator==(const char* str) const {
            if (isValid()) {
                int64_t valueCount=0;
                FnAttributeHandle handle = getHandle();
                const char** data = getSuite()->getStringNearestSample(
                        handle, 0.f, &valueCount);

                // Tagged pointer then?
                const char* value[1];
                if (!data && getSuite()->getStringValue(&handle,
                        0.0f, 0, &value[0]))
                {
                    data = value;
                    valueCount = 1;
                }

                if (valueCount==1) // len>1 arrays should not equal scalars
                    return strcmp(data[0], str) == 0;
            }
            return false;
        }

        bool operator!=(const char* str) const {
            return !operator==(str);
        }

        bool operator==(const std::string& str) const {
            return this->operator==(str.c_str());
        }

        bool operator!=(const std::string& str) const {
            return !operator==(str);
        }

        StringAttribute(const Attribute& rhs) : DataAttribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeString);
        }

        StringAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeString);
            return *this;
        }

#if FNKAT_CXX11
        StringAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            DataAttribute()
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeString);
        }

        StringAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeString);
            return *this;
        }
#endif  // FNKAT_CXX11

    private:
        StringAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /**
     * @brief A class representing a group attribute, used for hierarchically encapsulating
     * other attributes.
     */
    class FNATTRIBUTE_API GroupAttribute : public Attribute
    {
    public:

        typedef std::pair<std::string, Attribute> NamedAttr_Type;
        typedef std::vector<NamedAttr_Type> NamedAttrVector_Type;

        /**
         *  Create empty attribute class (isValid() == false).
         */
        GroupAttribute() : Attribute() {}

        explicit GroupAttribute(const bool groupInherit)
            : Attribute(getSuite()->createGroupAttr(NULL, NULL, 0,
                                                    (uint8_t)groupInherit))
        {
        }

        // Note: These constructors do not support delimited child names
        // (i.e., no dots in the name). If you wish to construct
        // a GroupAttribute with a deep hierarchy, use GroupBuilder instead.

        GroupAttribute(const std::string & name1, const Attribute & attr1,
                       const bool groupInherit)  : Attribute(NULL)
        {
            const char * names[] = { name1.c_str() };
            FnAttributeHandle attrs[] = { attr1.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 1, (uint8_t) groupInherit));
        }

        GroupAttribute(const char * name1, const Attribute & attr1,
                       const bool groupInherit)  : Attribute(NULL)
        {
            const char * names[] = { name1 };
            FnAttributeHandle attrs[] = { attr1.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 1, (uint8_t) groupInherit));
        }

        GroupAttribute(const std::string & name1, const Attribute & attr1,
                       const std::string & name2, const Attribute & attr2,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1.c_str(), name2.c_str() };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 2, (uint8_t) groupInherit));
        }

        GroupAttribute(const char * name1, const Attribute & attr1,
                       const char * name2, const Attribute & attr2,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1, name2 };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 2, (uint8_t) groupInherit));
        }

        GroupAttribute(const std::string & name1, const Attribute & attr1,
                       const std::string & name2, const Attribute & attr2,
                       const std::string & name3, const Attribute & attr3,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1.c_str(), name2.c_str(),
                name3.c_str() };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 3, (uint8_t) groupInherit));
        }

        GroupAttribute(const char * name1, const Attribute & attr1,
                       const char * name2, const Attribute & attr2,
                       const char * name3, const Attribute & attr3,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1, name2, name3 };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 3, (uint8_t) groupInherit));
        }

        GroupAttribute(const std::string & name1, const Attribute & attr1,
                       const std::string & name2, const Attribute & attr2,
                       const std::string & name3, const Attribute & attr3,
                       const std::string & name4, const Attribute & attr4,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1.c_str(), name2.c_str(),
                name3.c_str(), name4.c_str() };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle(), attr4.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 4, (uint8_t) groupInherit));
        }

        GroupAttribute(const char * name1, const Attribute & attr1,
                       const char * name2, const Attribute & attr2,
                       const char * name3, const Attribute & attr3,
                       const char * name4, const Attribute & attr4,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1, name2, name3, name4 };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle(), attr4.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 4, (uint8_t) groupInherit));
        }

        GroupAttribute(const std::string & name1, const Attribute & attr1,
                       const std::string & name2, const Attribute & attr2,
                       const std::string & name3, const Attribute & attr3,
                       const std::string & name4, const Attribute & attr4,
                       const std::string & name5, const Attribute & attr5,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1.c_str(), name2.c_str(),
                name3.c_str(), name4.c_str(), name5.c_str() };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle(), attr4.getHandle(), attr5.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 5, (uint8_t) groupInherit));
        }

        GroupAttribute(const char * name1, const Attribute & attr1,
                       const char * name2, const Attribute & attr2,
                       const char * name3, const Attribute & attr3,
                       const char * name4, const Attribute & attr4,
                       const char * name5, const Attribute & attr5,
                       const bool groupInherit) : Attribute(NULL)
        {
            const char * names[] = { name1, name2, name3, name4, name5 };
            FnAttributeHandle attrs[] = { attr1.getHandle(), attr2.getHandle(),
                attr3.getHandle(), attr4.getHandle(), attr5.getHandle() };
            stealHandle(getSuite()->createGroupAttr(
                names, attrs, 5, (uint8_t) groupInherit));
        }

        GroupAttribute(const NamedAttrVector_Type &children,
                       const bool inheritChildren);

        /**
         * Returns the intended type of this Attribute, without needing a valid
         * handle.
         */
        static FnKatAttributeType getKatAttributeType()
        {
            return kFnKatAttributeTypeGroup;
        }

        /**
         * Returns the number of child attributes under the group attribute.
         */
        int64_t getNumberOfChildren() const
        {
            return getSuite()->getNumberOfChildren(getHandle());
        }

        /**
         *  Returns the name of child at index under a group attribute.  This is
         *  returned as a std::string. If index is out of range, an empty string
         *  is returned.
         *  @param index  The index of the child name to return.
         */
        std::string getChildName(int64_t index) const
        {
            int32_t nameSize = 0;
            const char *name = getSuite()->getChildNameByIndex(getHandle(), index, &nameSize);
            if (nameSize > 0)
            {
                return std::string(name, static_cast<std::size_t>(nameSize));
            }
            return std::string();
        }

        /**
         *  Returns the name of child at index under a group attribute.
         *  This is returned as a const char *.
         *  @param index  The index of the child name to return.
         */
        const char* getChildNameCStr(int64_t index) const
            FNKAT_LVALUE_REF_QUALIFIER
        {
            int32_t nameSize = 0;
            return getChildNameCStr(index, nameSize);
        }

        /**
         *  Returns the name of child at index under a group attribute.
         *  This is returned as a const char *.  The length is also returned.
         *  @param index    The index of the child name to return.
         *  @param namelen  Output parameter where the name length is returned.
         */
        const char* getChildNameCStr(int64_t index, int32_t &namelen) const
            FNKAT_LVALUE_REF_QUALIFIER
        {
            const char *name = getSuite()->getChildNameByIndex(
                getHandle(), index, &namelen);
            return name;
        }

        /**
         * Returns the name of child at index under a group attribute.  This is
         * returned as a FnPlatform::StringView, containing the data and length.
         * @param index  The index of the child name to return.
         */
        FnPlatform::StringView getChildNameStringView(int64_t index) const
            FNKAT_LVALUE_REF_QUALIFIER
        {
            int32_t namelen = 0;
            const char* name = getSuite()->getChildNameByIndex(
                getHandle(), index, &namelen);
            return FnPlatform::StringView(name, namelen);
        }

#if FNKAT_CXX11
        /// @cond FN_INTERNAL_DEV
        const char* getChildNameCStr(int64_t index) && = delete;
        const char* getChildNameCStr(int64_t index, int32_t &len) && = delete;
        FnPlatform::StringView
            getChildNameStringView(int64_t index) && = delete;
        /// @endcond
#endif  // FNKAT_CXX11

        /**
         *  Returns a child attribute by index within a group attribute.
         *  If index is out of range, an invalid Attribute object is returned
         *  (returnValue.isValid() == false).
         *  @param index  The index of the child attribute to return.
         */
        Attribute getChildByIndex(int64_t index) const
        {
            return Attribute::CreateAndSteal(getSuite()->getChildByIndex(getHandle(), index));
        }

        /**
         *  Looks up a child attribute by name, returning it
         *  If named child does not exist, an invalid Attribute object
         *  is returned  (returnValue.isValid() == false).
         *  @param name  The name of attribute to look up.
         */
        Attribute getChildByName(FnPlatform::StringView name) const
        {
            return getChildByName(name.data(),
                                  static_cast<int32_t>(name.size()));
        }

        /**
         *  Looks up a child attribute by name, returning it
         *  If named child does not exist, an invalid Attribute object
         *  is returned  (returnValue.isValid() == false).
         *  @param name  The name of attribute to look up.
         *  @param length  The length of the string to look up.
         */
        Attribute getChildByName(const char* name, int32_t length) const
        {
            return Attribute::CreateAndSteal(
                getSuite()->getChildByName(getHandle(), name, length));
        }

        bool getGroupInherit() const
        {
            return (0 != getSuite()->getGroupInherit(getHandle()));
        }

        void fillChildVector(NamedAttrVector_Type * children) const;

        GroupAttribute(const Attribute& rhs) : Attribute(0x0)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeGroup);
        }

        GroupAttribute& operator=(const Attribute& rhs)
        {
            checkAndAcceptHandle(rhs, kFnKatAttributeTypeGroup);
            return *this;
        }

#if FNKAT_CXX11
        GroupAttribute(Attribute&& rhs) :  // NOLINT(runtime/explicit)
            Attribute(nullptr)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeGroup);
        }

        GroupAttribute& operator=(Attribute&& rhs)
        {
            checkAndMoveHandle(std::move(rhs), kFnKatAttributeTypeGroup);
            return *this;
        }
#endif  // FNKAT_CXX11

    private:
        GroupAttribute(FnAttributeHandle handle); // NOTE: not implemented
    };

    /** \defgroup FnAttribute_Utility_Functions FnAttribute
     * @{
     */

    /**
     * Encodes the given name or scene graph location path for use as an
     * attribute name.
     *
     * Useful to manually work around the case where one wants to have a
     * \c FnAttribute::GroupAttribute, where the immediate children have
     * dots in their names.
     *
     * \code
     * 0x2E ('.') -> 0x09 ('\t' -- Character Tabulation)
     * 0x2F ('/') -> 0x0A ('\n' -- Line Feed)
     * \endcode
     */
    FNATTRIBUTE_API
    std::string DelimiterEncode(const std::string & str);

    /**
     * Decodes the given encoded name or scene graph location path that might
     * have been used as an attribute name.
     *
     * Useful to manually work around the case where one wants to have a
     * \c FnAttribute::GroupAttribute, where the immediate children have
     * dots in their names.
     *
     * \code
     * 0x2E ('.') <- 0x09 ('\t' -- Character Tabulation)
     * 0x2F ('/') <- 0x0A ('\n' -- Line Feed)
     * \endcode
     */
    FNATTRIBUTE_API
    std::string DelimiterDecode(const std::string & str);

    /**
     * Returns the total memory currently allocated by the FnAttribute library.
     */
    FNATTRIBUTE_API
    uint64_t GetTotalSize();

    /**
     * @}
     */
}
FNATTRIBUTE_NAMESPACE_EXIT

#endif // FoundryKatanaAttribute_H
