//
//  Copyright (c) 2018-2019, Cem Bassoy, cem.bassoy@gmail.com
//
//  Distributed under the Boost Software License, Version 1.0. (See
//  accompanying file LICENSE_1_0.txt or copy at
//  http://www.boost.org/LICENSE_1_0.txt)
//
//  The authors gratefully acknowledge the support of
//  Fraunhofer IOSB, Ettlingen, Germany
//

#ifndef BOOST_UBLAS_TENSOR_OPERATORS_ARITHMETIC_HPP
#define BOOST_UBLAS_TENSOR_OPERATORS_ARITHMETIC_HPP

#include "expression.hpp"
#include "expression_evaluation.hpp"
#include "multi_index_utility.hpp"
#include "functions.hpp"

#include <type_traits>
#include <functional>
#include <algorithm>

namespace foundryboost {} namespace boost = foundryboost; namespace foundryboost{
namespace numeric{
namespace ublas {


template<class element_type, class storage_format, class storage_type>
class tensor;

template<class E>
class matrix_expression;


template<class E>
class vector_expression;

}
}
}

#define FIRST_ORDER_OPERATOR_RIGHT(OP, EXPR_TYPE_L, EXPR_TYPE_R) \
template<class T, class L, class R> \
auto operator OP ( foundryboost::numeric::ublas:: EXPR_TYPE_L <T,L> const& lhs, foundryboost::numeric::ublas:: EXPR_TYPE_R <R> const& rhs) { \
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), \
	  [](auto const& l, auto const& r){ return l OP r; }); \
} \

FIRST_ORDER_OPERATOR_RIGHT (*, detail:: tensor_expression , vector_expression)
FIRST_ORDER_OPERATOR_RIGHT (+, detail:: tensor_expression , vector_expression)
FIRST_ORDER_OPERATOR_RIGHT (-, detail:: tensor_expression , vector_expression)
FIRST_ORDER_OPERATOR_RIGHT (/, detail:: tensor_expression , vector_expression)

FIRST_ORDER_OPERATOR_RIGHT (*, detail:: tensor_expression , matrix_expression)
FIRST_ORDER_OPERATOR_RIGHT (+, detail:: tensor_expression , matrix_expression)
FIRST_ORDER_OPERATOR_RIGHT (-, detail:: tensor_expression , matrix_expression)
FIRST_ORDER_OPERATOR_RIGHT (/, detail:: tensor_expression , matrix_expression)


#define FIRST_ORDER_OPERATOR_LEFT(OP, EXPR_TYPE_L, EXPR_TYPE_R) \
template<class T, class L, class R> \
auto operator OP ( foundryboost::numeric::ublas:: EXPR_TYPE_L <L> const& lhs, foundryboost::numeric::ublas:: EXPR_TYPE_R <T,R> const& rhs) { \
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), \
	  [](auto const& l, auto const& r){ return l OP r; }); \
} \

FIRST_ORDER_OPERATOR_LEFT (*, vector_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (+, vector_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (-, vector_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (/, vector_expression, detail:: tensor_expression)

FIRST_ORDER_OPERATOR_LEFT (*, matrix_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (+, matrix_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (-, matrix_expression, detail:: tensor_expression)
FIRST_ORDER_OPERATOR_LEFT (/, matrix_expression, detail:: tensor_expression)




template<class T, class L, class R>
auto operator+( foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), [](auto const& l, auto const& r){ return l + r; });
}
template<class T, class L, class R>
auto operator-( foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), [](auto const& l, auto const& r){ return l - r; });
//	return foundryboost::numeric::ublas::detail::make_lambda<T>([&lhs,&rhs](std::size_t i){ return lhs(i) - rhs(i);});
}
template<class T, class L, class R>
auto operator*( foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), [](auto const& l, auto const& r){ return l * r; });
}
template<class T, class L, class R>
auto operator/( foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_binary_tensor_expression<T> (lhs(), rhs(), [](auto const& l, auto const& r){ return l / r; });
}


// Overloaded Arithmetic Operators with Scalars
template<class T, class R>
auto operator+(typename T::const_reference lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (rhs(), [lhs](auto const& r){ return lhs + r; });
	//return foundryboost::numeric::ublas::detail::make_lambda<T>( [&lhs,&rhs](std::size_t i) {return lhs + rhs(i); } );
}
template<class T, class R>
auto operator-(typename T::const_reference lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (rhs(), [lhs](auto const& r){ return lhs - r; });
}
template<class T, class R>
auto operator*(typename T::const_reference lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (rhs(), [lhs](auto const& r){ return lhs * r; });
}
template<class T, class R>
auto operator/(typename T::const_reference lhs, foundryboost::numeric::ublas::detail::tensor_expression<T,R> const& rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (rhs(), [lhs](auto const& r){ return lhs / r; });
}


template<class T, class L>
auto operator+(foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, typename T::const_reference rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (lhs(), [rhs] (auto const& l) { return l + rhs; } );
}
template<class T, class L>
auto operator-(foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, typename T::const_reference rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (lhs(), [rhs] (auto const& l) { return l - rhs; } );
}
template<class T, class L>
auto operator*(foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, typename T::const_reference rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (lhs(), [rhs] (auto const& l) { return l * rhs; } );
}
template<class T, class L>
auto operator/(foundryboost::numeric::ublas::detail::tensor_expression<T,L> const& lhs, typename T::const_reference rhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (lhs(), [rhs] (auto const& l) { return l / rhs; } );
}



template<class T, class D>
auto& operator += (T& lhs, const foundryboost::numeric::ublas::detail::tensor_expression<T,D> &expr) {
	foundryboost::numeric::ublas::detail::eval(lhs, expr(), [](auto& l, auto const& r) { l+=r; } );
	return lhs;
}

template<class T, class D>
auto& operator -= (T& lhs, const foundryboost::numeric::ublas::detail::tensor_expression<T,D> &expr) {
	foundryboost::numeric::ublas::detail::eval(lhs, expr(), [](auto& l, auto const& r) { l-=r; } );
	return lhs;
}

template<class T, class D>
auto& operator *= (T& lhs, const foundryboost::numeric::ublas::detail::tensor_expression<T,D> &expr) {
	foundryboost::numeric::ublas::detail::eval(lhs, expr(), [](auto& l, auto const& r) { l*=r; } );
	return lhs;
}

template<class T, class D>
auto& operator /= (T& lhs, const foundryboost::numeric::ublas::detail::tensor_expression<T,D> &expr) {
	foundryboost::numeric::ublas::detail::eval(lhs, expr(), [](auto& l, auto const& r) { l/=r; } );
	return lhs;
}




template<class E, class F, class A>
auto& operator += (foundryboost::numeric::ublas::tensor<E,F,A>& lhs, typename foundryboost::numeric::ublas::tensor<E,F,A>::const_reference r) {
	foundryboost::numeric::ublas::detail::eval(lhs, [r](auto& l) { l+=r; } );
	return lhs;
}

template<class E, class F, class A>
auto& operator -= (foundryboost::numeric::ublas::tensor<E,F,A>& lhs, typename foundryboost::numeric::ublas::tensor<E,F,A>::const_reference r) {
	foundryboost::numeric::ublas::detail::eval(lhs, [r](auto& l) { l-=r; } );
	return lhs;
}

template<class E, class F, class A>
auto& operator *= (foundryboost::numeric::ublas::tensor<E,F,A>& lhs, typename foundryboost::numeric::ublas::tensor<E,F,A>::const_reference r) {
	foundryboost::numeric::ublas::detail::eval(lhs, [r](auto& l) { l*=r; } );
	return lhs;
}

template<class E, class F, class A>
auto& operator /= (foundryboost::numeric::ublas::tensor<E,F,A>& lhs, typename foundryboost::numeric::ublas::tensor<E,F,A>::const_reference r) {
	foundryboost::numeric::ublas::detail::eval(lhs, [r](auto& l) { l/=r; } );
	return lhs;
}






template<class T, class D>
auto const& operator +(const foundryboost::numeric::ublas::detail::tensor_expression<T,D>& lhs) {
	return lhs;
}

template<class T, class D>
auto operator -(foundryboost::numeric::ublas::detail::tensor_expression<T,D> const& lhs) {
	return foundryboost::numeric::ublas::detail::make_unary_tensor_expression<T> (lhs(), [] (auto const& l) { return -l; } );
}





/** @brief Performs a tensor contraction, not an elementwise multiplication
	*
*/

template<class tensor_type_left, class tuple_type_left, class tensor_type_right, class tuple_type_right>
auto operator*(
		std::pair< tensor_type_left  const&, tuple_type_left  > lhs,
		std::pair< tensor_type_right const&, tuple_type_right > rhs)
{

	using namespace foundryboost::numeric::ublas;

	auto const& tensor_left  = lhs.first;
	auto const& tensor_right = rhs.first;

	auto multi_index_left = lhs.second;
	auto multi_index_right = rhs.second;

	static constexpr auto num_equal_ind = number_equal_indexes<tuple_type_left, tuple_type_right>::value;

	if constexpr ( num_equal_ind == 0  ){
		return tensor_left * tensor_right;
	}
	else if constexpr ( num_equal_ind==std::tuple_size<tuple_type_left>::value && std::is_same<tuple_type_left, tuple_type_right>::value ){

		return foundryboost::numeric::ublas::inner_prod( tensor_left, tensor_right );
	}
	else {
		auto array_index_pairs = index_position_pairs(multi_index_left,multi_index_right);
		auto index_pairs = array_to_vector(  array_index_pairs  );
		return foundryboost::numeric::ublas::prod( tensor_left, tensor_right, index_pairs.first, index_pairs.second );
	}

}

#endif
