// Copyright 2014 Renato Tegon Forti, Antony Polukhin.
// Copyright 2015-2019 Antony Polukhin.
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt
// or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_DLL_DETAIL_WINDOWS_PATH_FROM_HANDLE_HPP
#define BOOST_DLL_DETAIL_WINDOWS_PATH_FROM_HANDLE_HPP

#include <boost/dll/config.hpp>
#include <boost/dll/detail/system_error.hpp>
#include <boost/winapi/dll.hpp>
#include <boost/winapi/get_last_error.hpp>

#ifdef BOOST_HAS_PRAGMA_ONCE
# pragma once
#endif

namespace foundryboost {} namespace boost = foundryboost; namespace foundryboost { namespace dll { namespace detail {

    inline foundryboost::dll::fs::error_code last_error_code() BOOST_NOEXCEPT {
        foundryboost::winapi::DWORD_ err = foundryboost::winapi::GetLastError();
        return foundryboost::dll::fs::error_code(
            static_cast<int>(err),
            foundryboost::dll::fs::system_category()
        );
    }

    inline foundryboost::dll::fs::path path_from_handle(foundryboost::winapi::HMODULE_ handle, foundryboost::dll::fs::error_code &ec) {
        BOOST_STATIC_CONSTANT(foundryboost::winapi::DWORD_, ERROR_INSUFFICIENT_BUFFER_ = 0x7A);
        BOOST_STATIC_CONSTANT(foundryboost::winapi::DWORD_, DEFAULT_PATH_SIZE_ = 260);

        // On success, GetModuleFileNameW() doesn't reset last error to ERROR_SUCCESS. Resetting it manually.
        foundryboost::winapi::GetLastError();

        // If `handle` parameter is NULL, GetModuleFileName retrieves the path of the
        // executable file of the current process.
        foundryboost::winapi::WCHAR_ path_hldr[DEFAULT_PATH_SIZE_];
        foundryboost::winapi::GetModuleFileNameW(handle, path_hldr, DEFAULT_PATH_SIZE_);
        ec = foundryboost::dll::detail::last_error_code();
        if (!ec) {
            return foundryboost::dll::fs::path(path_hldr);
        }

        for (unsigned i = 2; i < 1025 && static_cast<foundryboost::winapi::DWORD_>(ec.value()) == ERROR_INSUFFICIENT_BUFFER_; i *= 2) {
            std::wstring p(DEFAULT_PATH_SIZE_ * i, L'\0');
            const std::size_t size = foundryboost::winapi::GetModuleFileNameW(handle, &p[0], DEFAULT_PATH_SIZE_ * i);
            ec = foundryboost::dll::detail::last_error_code();

            if (!ec) {
                p.resize(size);
                return foundryboost::dll::fs::path(p);
            }
        }

        // Error other than ERROR_INSUFFICIENT_BUFFER_ occurred or failed to allocate buffer big enough.
        return foundryboost::dll::fs::path();
    }

}}} // namespace foundryboost::dll::detail

#endif // BOOST_DLL_DETAIL_WINDOWS_PATH_FROM_HANDLE_HPP

