//
// Copyright 2016 Pixar
//
// Licensed under the Apache License, Version 2.0 (the "Apache License")
// with the following modification; you may not use this file except in
// compliance with the Apache License and the following modification to it:
// Section 6. Trademarks. is deleted and replaced with:
//
// 6. Trademarks. This License does not grant permission to use the trade
//    names, trademarks, service marks, or product names of the Licensor
//    and its affiliates, except as required to comply with Section 4(c) of
//    the License and to reproduce the content of the NOTICE file.
//
// You may obtain a copy of the Apache License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the Apache License with the above modification is
// distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. See the Apache License for the specific
// language governing permissions and limitations under the Apache License.
//
#ifndef PXR_IMAGING_HD_ST_MESH_TOPOLOGY_H
#define PXR_IMAGING_HD_ST_MESH_TOPOLOGY_H

#include "pxr/pxr.h"
#include "pxr/imaging/hdSt/api.h"
#include "pxr/imaging/hd/version.h"
#include "pxr/imaging/hd/meshTopology.h"
#include "pxr/imaging/hd/types.h"

#include <memory>

PXR_NAMESPACE_OPEN_SCOPE


class HdStResourceRegistry;
class HdSt_Subdivision;
struct HdQuadInfo;
class SdfPath;


using HdBufferSourceWeakPtr = 
    std::weak_ptr<class HdBufferSource>;
using HdBufferSourceSharedPtr = 
    std::shared_ptr<class HdBufferSource>;

using HdBufferArrayRangeSharedPtr =
    std::shared_ptr<class HdBufferArrayRange>;

using HdComputationSharedPtr = std::shared_ptr<class HdComputation>;

using HdSt_AdjacencyBuilderComputationPtr = 
    std::weak_ptr<class HdSt_AdjacencyBuilderComputation>;

using HdSt_QuadInfoBuilderComputationPtr = 
    std::weak_ptr<class HdSt_QuadInfoBuilderComputation>;
using HdSt_QuadInfoBuilderComputationSharedPtr =
    std::shared_ptr<class HdSt_QuadInfoBuilderComputation>;

using HdSt_MeshTopologySharedPtr = std::shared_ptr<class HdSt_MeshTopology>;



/// \class HdSt_MeshTopology
///
/// Storm implementation for mesh topology.
///
class HdSt_MeshTopology final : public HdMeshTopology {
public:
    /// Specifies how subdivision mesh topology is refined.
    enum RefineMode {
        RefineModeUniform = 0,
        RefineModePatches
    };

    HDST_API
    static HdSt_MeshTopologySharedPtr New(
        const HdMeshTopology &src,
        int refineLevel,
        RefineMode refineMode = RefineModeUniform);

    HDST_API
    virtual ~HdSt_MeshTopology();

    /// Equality check between two mesh topologies.
    HDST_API
    bool operator==(HdSt_MeshTopology const &other) const;

    /// \name Triangulation
    /// @{

    /// Returns the triangle indices (for drawing) buffer source computation.
    HDST_API
    HdBufferSourceSharedPtr GetTriangleIndexBuilderComputation(
        SdfPath const &id);

    /// Returns the CPU face-varying triangulate computation
    HDST_API
    HdBufferSourceSharedPtr GetTriangulateFaceVaryingComputation(
        HdBufferSourceSharedPtr const &source,
        SdfPath const &id);

    /// @}

    ///
    /// \name Quadrangulation
    /// @{

    /// Returns the quadinfo computation for the use of primvar
    /// quadrangulation.
    /// If gpu is true, the quadrangulate table will be transferred to GPU
    /// via the resource registry.
    HDST_API
    HdSt_QuadInfoBuilderComputationSharedPtr GetQuadInfoBuilderComputation(
        bool gpu, SdfPath const &id,
        HdStResourceRegistry *resourceRegistry = nullptr);

    /// Returns the quad indices (for drawing) buffer source computation.
    HDST_API
    HdBufferSourceSharedPtr GetQuadIndexBuilderComputation(SdfPath const &id);

    /// Returns the CPU quadrangulated buffer source.
    HDST_API
    HdBufferSourceSharedPtr GetQuadrangulateComputation(
        HdBufferSourceSharedPtr const &source, SdfPath const &id);

    /// Returns the GPU quadrangulate computation.
    HDST_API
    HdComputationSharedPtr GetQuadrangulateComputationGPU(
        TfToken const &name, HdType dataType, SdfPath const &id);

    /// Returns the CPU face-varying quadrangulate computation
    HDST_API
    HdBufferSourceSharedPtr GetQuadrangulateFaceVaryingComputation(
        HdBufferSourceSharedPtr const &source, SdfPath const &id);

    /// Returns the quadrangulation table range on GPU
    HDST_API
    HdBufferArrayRangeSharedPtr const &GetQuadrangulateTableRange() const {
        return _quadrangulateTableRange;
    }

    /// Clears the quadrangulation table range
    HDST_API
    void ClearQuadrangulateTableRange() {
        _quadrangulateTableRange.reset();
    }

    /// Sets the quadrangulation struct. HdMeshTopology takes an
    /// ownership of quadInfo (caller shouldn't free)
    HDST_API
    void SetQuadInfo(HdQuadInfo const *quadInfo);

    /// Returns the quadrangulation struct.
    HDST_API
    HdQuadInfo const *GetQuadInfo() const {
        return _quadInfo;
    }

    /// @}

    ///
    /// \name Points
    /// @{

    /// Returns the point indices buffer source computation.
    HDST_API
    HdBufferSourceSharedPtr GetPointsIndexBuilderComputation();

    /// @}

    ///
    /// \name Subdivision
    /// @{


    /// Returns the subdivision struct.
    HDST_API
    HdSt_Subdivision const *GetSubdivision() const {
        return _subdivision;
    }

    /// Returns the subdivision struct (non-const).
    HDST_API
    HdSt_Subdivision *GetSubdivision() {
        return _subdivision;
    }

    /// Returns true if the subdivision on this mesh produces
    /// triangles (otherwise quads)
    HDST_API
    bool RefinesToTriangles() const;

    /// Returns true if the subdivision of this mesh produces bspline patches
    HDST_API
    bool RefinesToBSplinePatches() const;

    /// Returns true if the subdivision of this mesh produces box spline
    /// triangle patches
    HDST_API
    bool RefinesToBoxSplineTrianglePatches() const;

    /// Returns the subdivision topology computation. It computes
    /// far mesh and produces refined quad-indices buffer.
    HDST_API
    HdBufferSourceSharedPtr GetOsdTopologyComputation(SdfPath const &debugId);

    /// Returns the refined indices builder computation.
    /// this just returns index and primitive buffer, and should be preceded by
    /// topology computation.
    HDST_API
    HdBufferSourceSharedPtr GetOsdIndexBuilderComputation();

    /// Returns the subdivision primvar refine computation on CPU.
    HDST_API
    HdBufferSourceSharedPtr GetOsdRefineComputation(
        HdBufferSourceSharedPtr const &source, bool varying);

    /// Returns the subdivision primvar refine computation on GPU.
    HDST_API
    HdComputationSharedPtr GetOsdRefineComputationGPU(
        TfToken const &name, HdType dataType);

    /// @}

private:
    // quadrangulation info on CPU
    HdQuadInfo const *_quadInfo;

    // quadrangulation info on GPU
    HdBufferArrayRangeSharedPtr _quadrangulateTableRange;

    HdSt_QuadInfoBuilderComputationPtr _quadInfoBuilder;

    // OpenSubdiv
    RefineMode _refineMode;
    HdSt_Subdivision *_subdivision;
    HdBufferSourceWeakPtr _osdTopologyBuilder;

    // Must be created through factory
    explicit HdSt_MeshTopology(
        const HdMeshTopology &src,
        int refineLevel,
        RefineMode refineMode);

    // No default construction or copying.
    HdSt_MeshTopology()                                      = delete;
    HdSt_MeshTopology(const HdSt_MeshTopology &)             = delete;
    HdSt_MeshTopology &operator =(const HdSt_MeshTopology &) = delete;
};


PXR_NAMESPACE_CLOSE_SCOPE

#endif // PXR_IMAGING_HD_ST_MESH_TOPOLOGY_H
