#!/bin/bash
read -r -d '' help <<- EOF
usage: install.sh [--katana-path <path> | --path <path>] [--3delight-path <path> | --no-3delight] [--accept-eula] [--help]
EOF

DELIGHT_VERSION="2.7.8"
KATANA_VERSION="5.0v2"

while [[ $# -gt 0 ]]
do
  key="$1"

  case $key in
    -h|--help)
      HELP="yes"
      shift # past argument
      ;;

    --accept-eula)
      ACCEPTEULA="yes"
      shift # past argument
      ;;

    --katana-path|--path)
      KATANAPATH="$2"
      shift # past argument
      shift # past value
      ;;

    --3delight-path)
      DELIGHTPATH="$2"
      shift # past argument
      shift # past value
      ;;

    --no-3delight)
      NODELIGHT="yes"
      shift # past argument
      ;;
  esac
done

if [[ $EUID -ne 0 ]]
then
  # If not root, then default to some directories the user likely has access to.
  DEFAULT_KATANA_PATH="$HOME/Katana${KATANA_VERSION}"
  DEFAULT_3DELIGHT_PATH="$HOME"
else
  DEFAULT_KATANA_PATH="/opt/Katana${KATANA_VERSION}"
  DEFAULT_3DELIGHT_PATH="/opt"
fi

function get_katana_path() {
  echo
  echo "Enter path to install Katana."
  echo -n "(default: $DEFAULT_KATANA_PATH): "
  read -r KATANAPATH
  if [ -z "$KATANAPATH" ]; then
    KATANAPATH=$DEFAULT_KATANA_PATH
  fi
}

function get_3delight_path() {
  echo
  echo -n "Do you wish to install 3Delight for Katana? (y/n [y]): "
  read -r install_3Delight
  case "$install_3Delight" in
    [nN][oO]|[nN])
      return 0
  esac
  while true; do
    echo "Enter path to install 3Delight."
    echo "3Delight will be installed into a versioned sub-directory of the"\
         "specified path."
    echo -n "(default: $DEFAULT_3DELIGHT_PATH): "
    read -r DELIGHTPATH

    if [ -z "$DELIGHTPATH" ]; then
      DELIGHTPATH=$DEFAULT_3DELIGHT_PATH
      break
    elif [[ "$DELIGHTPATH" =~ ^[a-zA-Z0-9/._-]+$ ]]; then
      break
    else
      echo "3Delight cannot be installed into a directory containing spaces"\
           "or special characters."
      echo ""
    fi
  done
}

function get_eula_response() {
  more EULA.txt
  read -r -p "Do you accept the terms of the EULA? (yes/no | y/n): " response
  validresponse=0
  while [ $validresponse -eq 0 ]
  do
    case "$response" in
      [yY][eE][sS]|[yY])
        response="yes"
        validresponse=1
        ;;
      [nN][oO]|[nN])
        exit 1
        ;;
      *)
        read -r -p "Do you accept the terms of the EULA? (yes/no | y/n): " \
            response
        ;;
    esac
  done
}

SCRIPT_NAME=$(readlink -f "$0")
SCRIPT_PATH=$(dirname "$SCRIPT_NAME")
DELIGHT_ARCHIVE_BASENAME="3delightForKatana-${DELIGHT_VERSION}-Linux-x86_64.tar.xz"

if ! [ -z "$HELP" ]; then
  echo "$help" >&2
  exit 0
fi
if [ -z "$ACCEPTEULA" ]; then
  get_eula_response
fi

if [ -z "$KATANAPATH" ]; then
  # Katana command line argument not provided
  get_katana_path
fi

if [ -z "$DELIGHTPATH" -a -z "$NODELIGHT" ]; then
  # 3Delight command line argument not provided
  get_3delight_path
fi

if ! mkdir -p "$KATANAPATH" 2>/dev/null
then
  echo "Failed to create directory: $KATANAPATH" >&2
  exit 1
fi
KATANAPATH=$(readlink -f "$KATANAPATH")  # Canonicalize KATANAPATH

if ! tar xvfz "$SCRIPT_PATH"/katana_files.tar.gz -C "$KATANAPATH"
then
  echo "Failed to extract Katana archive to $KATANAPATH" >&2
  exit 1
fi

SOURCE_REPLACE=""
THIRDPARTY_INSTALLERS_PATH="$KATANAPATH/thirdPartyInstallers"

if ! [ -z "$DELIGHTPATH" ]; then
  echo "Installing 3Delight..." >&2

  if ! mkdir -p "$DELIGHTPATH" 2>/dev/null
  then
    echo "Failed to create directory: $DELIGHTPATH" >&2
    exit 1
  fi
  DELIGHTPATH=$(readlink -f "$DELIGHTPATH")  # Canonicalize DELIGHTPATH

  if ! DELIGHT_TMP_PATH=$(mktemp --directory --tmpdir="$DELIGHTPATH" \
                          .tmp.XXXXXXX 2>/dev/null)
  then
    echo "Failed to write to $DELIGHTPATH" >&2
    exit 1
  fi
  trap 'rm -rf "$DELIGHT_TMP_PATH"' EXIT

  # Unpack the 3delight installer into the temp directory
  if ! tar xf "$THIRDPARTY_INSTALLERS_PATH/$DELIGHT_ARCHIVE_BASENAME" \
    -C "$DELIGHT_TMP_PATH" --strip-components=1 2>/dev/null
  then
    echo "Failed to extract 3Delight installer" >&2
    exit 1
  fi

  # Run the install script.
  if ! (cd "$DELIGHT_TMP_PATH" && ./install --prefix "$DELIGHTPATH")
  then
    echo "3Delight installation failed" >&2
    exit 1
  fi

  # Source the 3delight bash script
  DELIGHT_BASH_PATH="$DELIGHTPATH/3delight-${DELIGHT_VERSION}/.3delight_bash"
  ESCAPED_PATH=$(sed 's/[&/\]/\\&/g' <<<"$DELIGHT_BASH_PATH")
  SOURCE_REPLACE="source \"$ESCAPED_PATH\""
  echo "3Delight installation completed" >&2
fi

sed -i s/#__SOURCE_DL_BASH_SCRIPT__/"${SOURCE_REPLACE}"/1 "$KATANAPATH"/katana

echo
echo "Installation completed successfully." >&2
echo
echo "Katana was installed to: $KATANAPATH" >&2
if [ -z "$DELIGHTPATH" ]
then
  echo
  cat >&2 << EOF
You may wish to create a start-up script that sets the DEFAULT_RENDERER and
KATANA_RESOURCES environment variables as appropriate before invoking
$KATANAPATH/katana.

Please consult your renderer plug-in's documentation for further information.
EOF

else
  cat >&2 << EOF
3Delight was installed to: $DELIGHTPATH/3delight-${DELIGHT_VERSION}

3Delight has been set as the default Katana renderer plug-in.

To use another renderer plug-in, you may wish to create a start-up script that
sets the DEFAULT_RENDERER and KATANA_RESOURCES environment variables as
appropriate before invoking $KATANAPATH/katana.

Please consult your renderer plug-in's documentation for further information.
EOF
fi

echo
echo "Launch Katana by running: $KATANAPATH/katana" >&2
